// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_BROWSER_RENDERER_HOST_MEDIA_VIDEO_CAPTURE_CONTROLLER_EVENT_HANDLER_H_
#define CONTENT_BROWSER_RENDERER_HOST_MEDIA_VIDEO_CAPTURE_CONTROLLER_EVENT_HANDLER_H_

#include "base/memory/scoped_ptr.h"
#include "base/memory/shared_memory.h"
#include "content/common/content_export.h"

namespace base {
class DictionaryValue;
class TimeTicks;
}  // namespace base

namespace gfx {
class Rect;
class Size;
}  // namespace gfx

namespace gpu {
struct MailboxHolder;
}  // namespace gpu

namespace content {

typedef int VideoCaptureControllerID;

// VideoCaptureControllerEventHandler is the interface for
// VideoCaptureController to notify clients about the events such as
// BufferReady, FrameInfo, Error, etc.
class CONTENT_EXPORT VideoCaptureControllerEventHandler {
 public:
  // An Error has occurred in the VideoCaptureDevice.
  virtual void OnError(VideoCaptureControllerID id) = 0;

  // A buffer has been newly created.
  virtual void OnBufferCreated(VideoCaptureControllerID id,
                               base::SharedMemoryHandle handle,
                               int length,
                               int buffer_id) = 0;

  // A previously created buffer has been freed and will no longer be used.
  virtual void OnBufferDestroyed(VideoCaptureControllerID id,
                                 int buffer_id) = 0;

  // A buffer has been filled with I420 video.
  virtual void OnBufferReady(VideoCaptureControllerID id,
                             int buffer_id,
                             const gfx::Size& coded_size,
                             const gfx::Rect& visible_rect,
                             const base::TimeTicks& timestamp,
                             scoped_ptr<base::DictionaryValue> metadata) = 0;

  // A texture mailbox buffer has been filled with data.
  virtual void OnMailboxBufferReady(
      VideoCaptureControllerID id,
      int buffer_id,
      const gpu::MailboxHolder& mailbox_holder,
      const gfx::Size& packed_frame_size,
      const base::TimeTicks& timestamp,
      scoped_ptr<base::DictionaryValue> metadata) = 0;

  // The capture session has ended and no more frames will be sent.
  virtual void OnEnded(VideoCaptureControllerID id) = 0;

 protected:
  virtual ~VideoCaptureControllerEventHandler() {}
};

}  // namespace content

#endif  // CONTENT_BROWSER_RENDERER_HOST_MEDIA_VIDEO_CAPTURE_CONTROLLER_EVENT_HANDLER_H_
