// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "extensions/browser/api/cast_channel/cast_auth_ica.h"

#include "base/base64.h"
#include "base/lazy_instance.h"
#include "crypto/signature_verifier.h"
#include "extensions/browser/api/cast_channel/cast_auth_util.h"
#include "extensions/common/api/cast_channel/authority_keys.pb.h"
#include "net/base/hash_value.h"
#include "net/cert/x509_certificate.h"

namespace extensions {
namespace core_api {
namespace cast_channel {
namespace {

// Fingerprints and public keys of the allowed / trusted ICAs.
static const net::SHA256HashValue kFingerprintICA1 = { {
    0x52, 0x9D, 0x9C, 0xD6, 0x7F, 0xE5, 0xEB, 0x69, 0x8E, 0x70, 0xDD, 0x26,
    0xD7, 0xD8, 0xF1, 0x26, 0x59, 0xF1, 0xE6, 0xE5, 0x23, 0x48, 0xBF, 0x6A,
    0x5C, 0xF7, 0x16, 0xE1, 0x3F, 0x41, 0x0E, 0x73
} };

static const unsigned char kPublicKeyICA1[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xBC, 0x22, 0x80,
    0xBD, 0x80, 0xF6, 0x3A, 0x21, 0x00, 0x3B, 0xAE, 0x76, 0x5E, 0x35, 0x7F,
    0x3D, 0xC3, 0x64, 0x5C, 0x55, 0x94, 0x86, 0x34, 0x2F, 0x05, 0x87, 0x28,
    0xCD, 0xF7, 0x69, 0x8C, 0x17, 0xB3, 0x50, 0xA7, 0xB8, 0x82, 0xFA, 0xDF,
    0xC7, 0x43, 0x2D, 0xD6, 0x7E, 0xAB, 0xA0, 0x6F, 0xB7, 0x13, 0x72, 0x80,
    0xA4, 0x47, 0x15, 0xC1, 0x20, 0x99, 0x50, 0xCD, 0xEC, 0x14, 0x62, 0x09,
    0x5B, 0xA4, 0x98, 0xCD, 0xD2, 0x41, 0xB6, 0x36, 0x4E, 0xFF, 0xE8, 0x2E,
    0x32, 0x30, 0x4A, 0x81, 0xA8, 0x42, 0xA3, 0x6C, 0x9B, 0x33, 0x6E, 0xCA,
    0xB2, 0xF5, 0x53, 0x66, 0xE0, 0x27, 0x53, 0x86, 0x1A, 0x85, 0x1E, 0xA7,
    0x39, 0x3F, 0x4A, 0x77, 0x8E, 0xFB, 0x54, 0x66, 0x66, 0xFB, 0x58, 0x54,
    0xC0, 0x5E, 0x39, 0xC7, 0xF5, 0x50, 0x06, 0x0B, 0xE0, 0x8A, 0xD4, 0xCE,
    0xE1, 0x6A, 0x55, 0x1F, 0x8B, 0x17, 0x00, 0xE6, 0x69, 0xA3, 0x27, 0xE6,
    0x08, 0x25, 0x69, 0x3C, 0x12, 0x9D, 0x8D, 0x05, 0x2C, 0xD6, 0x2E, 0xA2,
    0x31, 0xDE, 0xB4, 0x52, 0x50, 0xD6, 0x20, 0x49, 0xDE, 0x71, 0xA0, 0xF9,
    0xAD, 0x20, 0x40, 0x12, 0xF1, 0xDD, 0x25, 0xEB, 0xD5, 0xE6, 0xB8, 0x36,
    0xF4, 0xD6, 0x8F, 0x7F, 0xCA, 0x43, 0xDC, 0xD7, 0x10, 0x5B, 0xE6, 0x3F,
    0x51, 0x8A, 0x85, 0xB3, 0xF3, 0xFF, 0xF6, 0x03, 0x2D, 0xCB, 0x23, 0x4F,
    0x9C, 0xAD, 0x18, 0xE7, 0x93, 0x05, 0x8C, 0xAC, 0x52, 0x9A, 0xF7, 0x4C,
    0xE9, 0x99, 0x7A, 0xBE, 0x6E, 0x7E, 0x4D, 0x0A, 0xE3, 0xC6, 0x1C, 0xA9,
    0x93, 0xFA, 0x3A, 0xA5, 0x91, 0x5D, 0x1C, 0xBD, 0x66, 0xEB, 0xCC, 0x60,
    0xDC, 0x86, 0x74, 0xCA, 0xCF, 0xF8, 0x92, 0x1C, 0x98, 0x7D, 0x57, 0xFA,
    0x61, 0x47, 0x9E, 0xAB, 0x80, 0xB7, 0xE4, 0x48, 0x80, 0x2A, 0x92, 0xC5,
    0x1B, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA2 = {{
    0x98, 0xD6, 0x6D, 0xE8, 0x1C, 0x56, 0x47, 0x84, 0x4D, 0x96, 0x43, 0x4D,
    0x37, 0xC4, 0x8D, 0xC2, 0xCE, 0x7F, 0x08, 0x72, 0x50, 0x19, 0xE7, 0xB6,
    0xE8, 0x92, 0x7F, 0x56, 0xD6, 0x81, 0xAA, 0x18
}};

static const unsigned char kPublicKeyICA2[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xBC, 0x22, 0x80,
    0xBD, 0x80, 0xF6, 0x3A, 0x21, 0x00, 0x3B, 0xAE, 0x76, 0x5E, 0x35, 0x7F,
    0x3D, 0xC3, 0x64, 0x5C, 0x55, 0x94, 0x86, 0x34, 0x2F, 0x05, 0x87, 0x28,
    0xCD, 0xF7, 0x69, 0x8C, 0x17, 0xB3, 0x50, 0xA7, 0xB8, 0x82, 0xFA, 0xDF,
    0xC7, 0x43, 0x2D, 0xD6, 0x7E, 0xAB, 0xA0, 0x6F, 0xB7, 0x13, 0x72, 0x80,
    0xA4, 0x47, 0x15, 0xC1, 0x20, 0x99, 0x50, 0xCD, 0xEC, 0x14, 0x62, 0x09,
    0x5B, 0xA4, 0x98, 0xCD, 0xD2, 0x41, 0xB6, 0x36, 0x4E, 0xFF, 0xE8, 0x2E,
    0x32, 0x30, 0x4A, 0x81, 0xA8, 0x42, 0xA3, 0x6C, 0x9B, 0x33, 0x6E, 0xCA,
    0xB2, 0xF5, 0x53, 0x66, 0xE0, 0x27, 0x53, 0x86, 0x1A, 0x85, 0x1E, 0xA7,
    0x39, 0x3F, 0x4A, 0x77, 0x8E, 0xFB, 0x54, 0x66, 0x66, 0xFB, 0x58, 0x54,
    0xC0, 0x5E, 0x39, 0xC7, 0xF5, 0x50, 0x06, 0x0B, 0xE0, 0x8A, 0xD4, 0xCE,
    0xE1, 0x6A, 0x55, 0x1F, 0x8B, 0x17, 0x00, 0xE6, 0x69, 0xA3, 0x27, 0xE6,
    0x08, 0x25, 0x69, 0x3C, 0x12, 0x9D, 0x8D, 0x05, 0x2C, 0xD6, 0x2E, 0xA2,
    0x31, 0xDE, 0xB4, 0x52, 0x50, 0xD6, 0x20, 0x49, 0xDE, 0x71, 0xA0, 0xF9,
    0xAD, 0x20, 0x40, 0x12, 0xF1, 0xDD, 0x25, 0xEB, 0xD5, 0xE6, 0xB8, 0x36,
    0xF4, 0xD6, 0x8F, 0x7F, 0xCA, 0x43, 0xDC, 0xD7, 0x10, 0x5B, 0xE6, 0x3F,
    0x51, 0x8A, 0x85, 0xB3, 0xF3, 0xFF, 0xF6, 0x03, 0x2D, 0xCB, 0x23, 0x4F,
    0x9C, 0xAD, 0x18, 0xE7, 0x93, 0x05, 0x8C, 0xAC, 0x52, 0x9A, 0xF7, 0x4C,
    0xE9, 0x99, 0x7A, 0xBE, 0x6E, 0x7E, 0x4D, 0x0A, 0xE3, 0xC6, 0x1C, 0xA9,
    0x93, 0xFA, 0x3A, 0xA5, 0x91, 0x5D, 0x1C, 0xBD, 0x66, 0xEB, 0xCC, 0x60,
    0xDC, 0x86, 0x74, 0xCA, 0xCF, 0xF8, 0x92, 0x1C, 0x98, 0x7D, 0x57, 0xFA,
    0x61, 0x47, 0x9E, 0xAB, 0x80, 0xB7, 0xE4, 0x48, 0x80, 0x2A, 0x92, 0xC5,
    0x1B, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA3 = {{
    0xA2, 0x48, 0xC2, 0xE8, 0x54, 0xE6, 0x56, 0xA5, 0x6D, 0xE8, 0x23, 0x1F,
    0x1E, 0xE1, 0x75, 0x6F, 0xDB, 0xE4, 0x07, 0xF9, 0xFE, 0xD4, 0x65, 0x0D,
    0x60, 0xCC, 0x5A, 0xCB, 0x65, 0x11, 0xC7, 0x20
}};

static const unsigned char kPublicKeyICA3[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xB0, 0x0E, 0x5E,
    0x07, 0x3A, 0xDF, 0xA4, 0x5F, 0x68, 0xF7, 0x21, 0xC7, 0x64, 0xDB, 0xB6,
    0x76, 0xEF, 0xEE, 0x8B, 0x93, 0xF8, 0xF6, 0x1B, 0x88, 0xE1, 0x93, 0xB7,
    0x17, 0xF0, 0x15, 0x1E, 0x7E, 0x52, 0x55, 0x77, 0x3C, 0x02, 0x8D, 0x7B,
    0x4A, 0x6C, 0xD3, 0xBD, 0xD6, 0xC1, 0x9C, 0x72, 0xC8, 0xB3, 0x15, 0xCF,
    0x11, 0xC1, 0xF5, 0x46, 0xC4, 0xD5, 0x20, 0x47, 0xFB, 0x30, 0xF4, 0xE4,
    0x61, 0x0C, 0x68, 0xF0, 0x5E, 0xAB, 0x37, 0x8E, 0x9B, 0xE1, 0xBC, 0x81,
    0xC3, 0x70, 0x8A, 0x78, 0xD6, 0x83, 0x34, 0x32, 0x9C, 0x19, 0x62, 0xEB,
    0xE4, 0x9C, 0xED, 0xE3, 0x64, 0x6C, 0x41, 0x1D, 0x9C, 0xD2, 0x8B, 0x48,
    0x4C, 0x23, 0x90, 0x95, 0xB3, 0xE7, 0x52, 0xEA, 0x05, 0x57, 0xCC, 0x60,
    0xB3, 0xBA, 0x14, 0xE4, 0xBA, 0x00, 0x39, 0xE4, 0x46, 0x55, 0x74, 0xCE,
    0x5A, 0x8E, 0x7A, 0x67, 0x23, 0xDA, 0x68, 0x0A, 0xFA, 0xC4, 0x84, 0x1E,
    0xB4, 0xC5, 0xA1, 0xA2, 0x6A, 0x73, 0x1F, 0x6E, 0xC8, 0x2E, 0x2F, 0x9A,
    0x9E, 0xA8, 0xB1, 0x0E, 0xFD, 0x87, 0xA6, 0x8F, 0x4D, 0x3D, 0x4B, 0x05,
    0xD5, 0x35, 0x5A, 0x74, 0x4D, 0xBC, 0x8E, 0x82, 0x44, 0x96, 0xF4, 0xB5,
    0x95, 0x60, 0x4E, 0xA5, 0xDF, 0x27, 0x3D, 0x41, 0x5C, 0x07, 0xA3, 0xB4,
    0x35, 0x5A, 0xB3, 0x9E, 0xF2, 0x05, 0x24, 0xCA, 0xCD, 0x31, 0x5A, 0x0D,
    0x26, 0x4C, 0xD4, 0xD3, 0xFD, 0x50, 0xE1, 0x34, 0xE9, 0x4C, 0x81, 0x58,
    0x30, 0xB2, 0xC7, 0x7A, 0xDD, 0x81, 0x89, 0xA6, 0xD4, 0x3A, 0x38, 0x84,
    0x03, 0xB7, 0x34, 0x9E, 0x77, 0x3F, 0xFF, 0x78, 0x07, 0x5B, 0x99, 0xC1,
    0xB2, 0x1F, 0x35, 0x56, 0x6E, 0x3A, 0x3C, 0x0C, 0x25, 0xE1, 0x57, 0xF6,
    0x8A, 0x7E, 0x49, 0xC0, 0xCC, 0x83, 0x11, 0x35, 0xE7, 0x91, 0x6D, 0x2E,
    0x65, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA4 = {{
    0x2B, 0x9B, 0xC7, 0x92, 0xCD, 0x2A, 0x20, 0xB3, 0x65, 0x5E, 0x57, 0xFB,
    0x10, 0x1A, 0x95, 0x93, 0x62, 0x53, 0x69, 0x50, 0x52, 0xA2, 0x71, 0x42,
    0x2B, 0xFE, 0xB5, 0xB4, 0x5D, 0xC5, 0xA5, 0xE9
}};

static const unsigned char kPublicKeyICA4[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xB7, 0xE8, 0xC3,
    0xE4, 0x2C, 0xDE, 0x74, 0x53, 0xF2, 0x49, 0x95, 0x6D, 0xD1, 0xDA, 0x69,
    0x57, 0x0D, 0x86, 0xE5, 0xED, 0xB4, 0xB9, 0xE6, 0x73, 0x9F, 0x6C, 0xAD,
    0x3B, 0x64, 0x85, 0x03, 0x0D, 0x08, 0x44, 0xAF, 0x18, 0x69, 0x82, 0xAD,
    0xA9, 0x74, 0x64, 0x37, 0x47, 0xE1, 0xE7, 0x26, 0x19, 0x33, 0x3C, 0xE2,
    0xD0, 0xB5, 0x84, 0x3C, 0xD7, 0xAC, 0x63, 0xAE, 0xC4, 0x32, 0x23, 0xF6,
    0xDC, 0x14, 0x10, 0x4B, 0x95, 0x7F, 0xE8, 0x98, 0xD7, 0x7A, 0x9E, 0x43,
    0x3D, 0x68, 0x8B, 0x2A, 0x70, 0xF7, 0x1E, 0x43, 0x70, 0xBA, 0xA5, 0xA5,
    0x93, 0xAD, 0x8A, 0xD4, 0x9F, 0xAC, 0x83, 0x16, 0xF3, 0x48, 0x5F, 0xC5,
    0xE0, 0xA5, 0x44, 0xB8, 0x4F, 0xD9, 0xD8, 0x75, 0x90, 0x25, 0x8B, 0xE3,
    0x1C, 0x6C, 0xDA, 0x88, 0xFF, 0x09, 0x2B, 0xCA, 0x1E, 0x48, 0xDD, 0x76,
    0x0F, 0x68, 0x56, 0x7B, 0x15, 0x9D, 0xCA, 0x6B, 0x1C, 0xF7, 0x48, 0xC2,
    0x89, 0xC6, 0x93, 0x0A, 0x31, 0xF2, 0x78, 0x27, 0x45, 0x3D, 0xF1, 0x0D,
    0x5B, 0x6E, 0x55, 0x32, 0xEF, 0x49, 0xA0, 0xD6, 0xAF, 0xA6, 0x30, 0x91,
    0xF2, 0x21, 0x2F, 0xDB, 0xA4, 0x29, 0xB9, 0x9B, 0x22, 0xBC, 0xCD, 0x0B,
    0xA6, 0x8B, 0xA6, 0x22, 0x79, 0xFD, 0xCF, 0x95, 0x93, 0x96, 0xB3, 0x23,
    0xC9, 0xC6, 0x30, 0x8E, 0xC0, 0xE9, 0x1F, 0xEC, 0xFB, 0xF5, 0x88, 0xDD,
    0x97, 0x72, 0x16, 0x29, 0x08, 0xFA, 0x42, 0xE7, 0x4F, 0xCA, 0xAE, 0xD7,
    0x0F, 0x23, 0x48, 0x9B, 0x82, 0xA7, 0x37, 0x4A, 0xDD, 0x60, 0x04, 0x75,
    0xDC, 0xDE, 0x09, 0x98, 0xD2, 0x16, 0x23, 0x04, 0x70, 0x4D, 0x99, 0x9F,
    0x4A, 0x82, 0x28, 0xE6, 0xBE, 0x8F, 0x9D, 0xBF, 0xA1, 0x4B, 0xA2, 0xBA,
    0xF5, 0xB2, 0x51, 0x1E, 0x4E, 0xE7, 0x80, 0x9E, 0x7A, 0x38, 0xA1, 0xC7,
    0x09, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA5 = {{
    0xFD, 0xCD, 0x0D, 0x59, 0x29, 0xF8, 0x7A, 0x62, 0xD2, 0xFE, 0x0F, 0xD8,
    0x94, 0xB8, 0x32, 0xF1, 0xD4, 0x9C, 0x73, 0xF9, 0xD3, 0x51, 0x8B, 0x64,
    0x67, 0x3E, 0x04, 0x97, 0x0B, 0x0A, 0x4F, 0xAD
}};

static const unsigned char kPublicKeyICA5[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xC2, 0xF6, 0xD5,
    0x91, 0xDC, 0x37, 0xB4, 0x9A, 0x73, 0x4A, 0xE7, 0x74, 0x6D, 0x03, 0xAE,
    0x27, 0x24, 0x41, 0x99, 0x96, 0x1B, 0x05, 0x0E, 0xC7, 0xCF, 0x09, 0xCD,
    0x65, 0x56, 0x02, 0xFC, 0x98, 0x59, 0xB4, 0xBB, 0x95, 0x71, 0xD7, 0x88,
    0x66, 0xC8, 0x08, 0xCB, 0xBF, 0x5B, 0x85, 0x65, 0x7E, 0xDE, 0xC4, 0xB5,
    0xE3, 0x71, 0x24, 0xA2, 0xFD, 0x92, 0x2C, 0x77, 0xC5, 0x08, 0xE0, 0xF0,
    0xB1, 0x8A, 0x8A, 0x54, 0xCA, 0xA6, 0xAF, 0x87, 0xB8, 0xCB, 0x7D, 0x83,
    0x28, 0x59, 0x9C, 0x01, 0xF5, 0x7B, 0x10, 0xD0, 0xF3, 0x52, 0x09, 0x3F,
    0xF5, 0x7D, 0xDA, 0x21, 0x63, 0x8F, 0xAC, 0x8B, 0x60, 0x67, 0x22, 0xEF,
    0x6B, 0x66, 0x91, 0xFC, 0x97, 0x30, 0x8D, 0xCC, 0xFE, 0xDE, 0x5C, 0xF9,
    0x19, 0xBB, 0x1C, 0x25, 0x29, 0x2C, 0x99, 0x48, 0x41, 0xC2, 0xFC, 0x5B,
    0x66, 0xD6, 0x79, 0x84, 0x16, 0x8D, 0x0D, 0x4F, 0x75, 0x01, 0x40, 0xC5,
    0x50, 0x69, 0xFA, 0xA4, 0x88, 0xF1, 0xD2, 0x3B, 0xD1, 0x23, 0xDF, 0xC5,
    0xBA, 0xE3, 0xE8, 0xBA, 0xCC, 0x1E, 0x93, 0x17, 0xF7, 0x97, 0xE2, 0x71,
    0x42, 0x75, 0x5B, 0x99, 0x55, 0x98, 0x22, 0x23, 0x98, 0xDC, 0x10, 0x89,
    0xF4, 0xE8, 0x26, 0xBB, 0x98, 0x66, 0xFD, 0xBB, 0x9A, 0x21, 0x62, 0xA2,
    0xDF, 0x90, 0xDB, 0x48, 0x6F, 0xDB, 0x2A, 0xEF, 0xDE, 0x53, 0x59, 0x31,
    0x5D, 0x38, 0xCD, 0x80, 0xA8, 0x0C, 0x6E, 0x4E, 0x37, 0x65, 0xEB, 0x36,
    0x1C, 0x13, 0xBA, 0x53, 0xD3, 0x8F, 0xCC, 0x43, 0x86, 0x02, 0x70, 0xD2,
    0x91, 0xF6, 0x96, 0x25, 0x6C, 0xA4, 0xE8, 0x1F, 0xD8, 0xB3, 0x74, 0x20,
    0xEB, 0x60, 0x9D, 0x3D, 0xD3, 0x3D, 0x2E, 0x36, 0x0F, 0xF1, 0x94, 0x10,
    0xF9, 0x7A, 0x03, 0x52, 0x7E, 0xA4, 0xEF, 0xE3, 0x40, 0x9E, 0x74, 0x0E,
    0xDF, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA6 = {{
    0xD6, 0xA1, 0xC4, 0xBC, 0x74, 0x5B, 0xC8, 0xE9, 0xED, 0xF8, 0x9C, 0x0D,
    0x55, 0xCC, 0xB2, 0xC9, 0xF3, 0x04, 0xF6, 0x6E, 0x4D, 0x3D, 0x27, 0x4C,
    0xA3, 0xFC, 0x0F, 0x4B, 0x95, 0x55, 0x10, 0x7E
}};

static const unsigned char kPublicKeyICA6[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xB5, 0xC8, 0x14,
    0x5B, 0x94, 0x2E, 0x8E, 0x40, 0xBC, 0x8A, 0xAB, 0x1F, 0x48, 0xEE, 0xA5,
    0x5C, 0x5D, 0xA9, 0x44, 0x23, 0x33, 0xE5, 0x09, 0xDD, 0x84, 0xDD, 0xA6,
    0x08, 0x95, 0xB0, 0xEA, 0x64, 0xEB, 0xC1, 0xCA, 0x02, 0x60, 0xFF, 0x83,
    0xF9, 0x17, 0x71, 0x2C, 0xC7, 0xAB, 0x06, 0x0F, 0xE4, 0xAD, 0x39, 0x24,
    0xFB, 0x1F, 0xED, 0xFA, 0xB2, 0x4D, 0x14, 0x5A, 0x6E, 0x5B, 0x06, 0x10,
    0x13, 0xE7, 0x77, 0x22, 0xAA, 0xE5, 0xD1, 0x2C, 0x05, 0xC4, 0x06, 0x05,
    0xB1, 0xCD, 0xBE, 0xCB, 0x4B, 0xAF, 0x11, 0x3E, 0xA0, 0x77, 0xBA, 0x6D,
    0xE4, 0xA7, 0xBA, 0xC9, 0x9D, 0x3F, 0x47, 0xE0, 0xD6, 0x20, 0x75, 0x1C,
    0xE9, 0x89, 0xD3, 0x88, 0x56, 0x4F, 0x29, 0xF6, 0x7D, 0x49, 0x96, 0xBE,
    0xE8, 0x41, 0xAB, 0x35, 0x08, 0xAD, 0x07, 0x22, 0x90, 0xA3, 0x4A, 0x98,
    0xBA, 0xC3, 0xE2, 0x29, 0xDA, 0x2E, 0xBD, 0x34, 0xF5, 0x41, 0xBC, 0x27,
    0x7D, 0xE0, 0x02, 0xBF, 0xB7, 0xAE, 0x8B, 0x1E, 0xEE, 0xE9, 0xC1, 0x59,
    0x92, 0xEA, 0xE3, 0x76, 0x0E, 0xE7, 0x77, 0xEF, 0x10, 0x7E, 0x4F, 0xD8,
    0xAD, 0xC4, 0x5D, 0xBB, 0xB7, 0x9F, 0x23, 0x0B, 0x34, 0x89, 0xF7, 0x97,
    0x9A, 0x40, 0x79, 0x00, 0xDD, 0x10, 0x9E, 0x01, 0xA7, 0xF0, 0xD8, 0xC4,
    0x37, 0xF1, 0x6A, 0xD7, 0xC2, 0xE9, 0x75, 0x94, 0x55, 0xA9, 0x81, 0xA8,
    0xF8, 0xC6, 0xF9, 0xD2, 0xCF, 0x26, 0xA0, 0x74, 0x58, 0x2E, 0xD0, 0xCB,
    0x16, 0x58, 0x1B, 0x1E, 0x2B, 0x94, 0x80, 0x26, 0x82, 0x3F, 0x01, 0x36,
    0x01, 0x97, 0x1E, 0xA6, 0x94, 0x14, 0xC0, 0xB2, 0x55, 0x95, 0x2E, 0x30,
    0x9C, 0x7B, 0xC6, 0x79, 0xF8, 0x12, 0xB3, 0xB4, 0x11, 0x93, 0x73, 0x9C,
    0xD4, 0x3F, 0x29, 0x6E, 0x6A, 0xAA, 0xA8, 0xE9, 0xA2, 0xF3, 0x20, 0x4E,
    0xE9, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA7 = {{
    0xB6, 0xB3, 0x4F, 0x73, 0x23, 0x9D, 0xA6, 0xDD, 0x9A, 0x26, 0xF7, 0xD5,
    0x9E, 0x26, 0x3B, 0xF0, 0x0C, 0x01, 0x6B, 0x21, 0x17, 0x41, 0xE5, 0xAD,
    0x92, 0xA5, 0xA0, 0x0B, 0xD7, 0x31, 0x26, 0x11
}};

static const unsigned char kPublicKeyICA7[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xCC, 0xE2, 0xB6,
    0x2F, 0x11, 0xAB, 0xFF, 0xD0, 0x5D, 0x63, 0x97, 0x59, 0xFA, 0x5F, 0x26,
    0xD7, 0x91, 0xE9, 0x90, 0x86, 0x31, 0x91, 0x8E, 0x2D, 0x29, 0x5F, 0x7A,
    0x2F, 0x12, 0x77, 0x21, 0xD9, 0x0E, 0x99, 0x0F, 0x11, 0x08, 0x69, 0x3C,
    0x9C, 0x58, 0xD4, 0xCE, 0x44, 0xB3, 0x8E, 0x46, 0x6C, 0xC1, 0x8E, 0x60,
    0x63, 0x3E, 0x99, 0x24, 0x72, 0x69, 0x07, 0xC2, 0x30, 0x0F, 0xD5, 0x74,
    0x06, 0xC6, 0x09, 0x94, 0x13, 0xD7, 0x34, 0x34, 0x75, 0x73, 0x4F, 0x4A,
    0x01, 0xFE, 0x1E, 0x3C, 0x91, 0x86, 0x9C, 0x63, 0xF8, 0xEF, 0x15, 0x79,
    0xE5, 0x5A, 0xC1, 0xF2, 0x05, 0xDC, 0xE0, 0x76, 0xDE, 0x69, 0x46, 0xDF,
    0x12, 0x3D, 0xF4, 0xD9, 0x05, 0xE2, 0x9E, 0xAD, 0x07, 0xCC, 0x69, 0x5F,
    0x1D, 0x42, 0x5E, 0x6C, 0x4A, 0xB4, 0x89, 0x7C, 0xDB, 0xBC, 0x69, 0x4E,
    0x23, 0x70, 0x9A, 0x67, 0xC5, 0xFA, 0x29, 0x88, 0x7C, 0x30, 0xFA, 0x61,
    0x98, 0x3C, 0x43, 0x4A, 0x1E, 0xCA, 0xAE, 0xA8, 0x7B, 0x65, 0x68, 0xA3,
    0xFB, 0x38, 0xB4, 0x4F, 0xCA, 0x49, 0x60, 0x85, 0xA0, 0xC1, 0x55, 0xA1,
    0xCE, 0x67, 0x78, 0x9F, 0x53, 0x81, 0xD7, 0x92, 0xC9, 0x6D, 0x44, 0xF1,
    0x97, 0x95, 0xA5, 0x7C, 0x83, 0xEC, 0xD3, 0xEB, 0x7D, 0xD7, 0x0A, 0x06,
    0xFE, 0xBA, 0xFC, 0x56, 0x5F, 0xF0, 0x70, 0xE4, 0x82, 0xBE, 0x69, 0x6D,
    0x95, 0x00, 0xDF, 0xB5, 0xE5, 0xBF, 0x9E, 0xF1, 0x12, 0x47, 0x14, 0x9C,
    0x7D, 0xE5, 0xA0, 0xB8, 0x70, 0x29, 0x6B, 0xC8, 0x8A, 0xF2, 0xBA, 0x35,
    0xD8, 0xC4, 0xD3, 0xB4, 0xB5, 0xEB, 0xDF, 0x2D, 0x27, 0x46, 0xA5, 0xFF,
    0x35, 0xB5, 0x5F, 0x85, 0x72, 0xEB, 0xCF, 0xAD, 0x09, 0x18, 0x05, 0x95,
    0x56, 0x88, 0x95, 0x22, 0xD7, 0x60, 0x47, 0xC9, 0x1F, 0xFA, 0x2D, 0x51,
    0x3F, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA8 = {{
    0xE9, 0xD4, 0x83, 0xD3, 0x78, 0x01, 0xB1, 0x1A, 0xB8, 0x8E, 0xE1, 0x87,
    0xB1, 0x88, 0xD5, 0xE2, 0xF0, 0xFE, 0x18, 0xE7, 0xCE, 0xB8, 0x10, 0x06,
    0x3E, 0xA5, 0x05, 0x4A, 0x1A, 0x9D, 0x1C, 0x11
}};

static const unsigned char kPublicKeyICA8[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xBF, 0x3A, 0x31,
    0xFC, 0xF8, 0xE6, 0xEE, 0xC0, 0x48, 0x00, 0xB4, 0x05, 0x65, 0x36, 0x23,
    0x6D, 0x34, 0xD6, 0x00, 0xD5, 0x43, 0x89, 0x6A, 0x90, 0xCB, 0x7D, 0x1B,
    0x39, 0xFE, 0x2E, 0x83, 0x84, 0x29, 0xBE, 0x51, 0xEF, 0x98, 0x66, 0x48,
    0x59, 0x8E, 0x7E, 0x10, 0x14, 0x1D, 0x9F, 0xAA, 0x52, 0xFD, 0x6B, 0xBF,
    0xC6, 0x13, 0xF2, 0xE9, 0x79, 0x62, 0xE2, 0xA0, 0x3B, 0xC6, 0x44, 0x70,
    0x94, 0x98, 0xAF, 0x92, 0x9E, 0x66, 0x3F, 0xA4, 0x6C, 0xC1, 0x2F, 0x6D,
    0xA2, 0x08, 0x8A, 0x47, 0x1B, 0xFA, 0x6D, 0x09, 0xCF, 0x94, 0xDB, 0x20,
    0xCE, 0xA2, 0xBF, 0xEA, 0x06, 0xF4, 0xD3, 0x4D, 0xF7, 0x62, 0xAE, 0x1A,
    0x64, 0xEC, 0x1F, 0xA5, 0x40, 0x2D, 0x15, 0xE7, 0xF7, 0x26, 0xFB, 0x08,
    0xD9, 0x5B, 0xFC, 0x86, 0x7E, 0xC7, 0x94, 0x18, 0x08, 0x2A, 0xF5, 0x83,
    0x44, 0x06, 0x15, 0x12, 0x5A, 0x1F, 0xBB, 0x47, 0xE3, 0x2C, 0x61, 0x64,
    0xDF, 0xFE, 0x74, 0x0E, 0x78, 0xA4, 0x65, 0xB8, 0x70, 0xC1, 0xDB, 0x3D,
    0xCA, 0x26, 0x33, 0xBD, 0x4A, 0x14, 0xDA, 0x0B, 0xEC, 0xEC, 0xB3, 0x34,
    0x23, 0x59, 0xD2, 0x11, 0xF9, 0xB0, 0x53, 0x1C, 0x75, 0x76, 0xF5, 0x65,
    0x00, 0x6C, 0xF0, 0x7F, 0xFA, 0x1A, 0x59, 0xFE, 0xF8, 0x9D, 0x1A, 0x4E,
    0x42, 0x35, 0xEE, 0x2F, 0xE3, 0xA1, 0xE2, 0xDF, 0xDB, 0x7A, 0x6B, 0x5E,
    0x6B, 0x21, 0xFF, 0xA5, 0xE1, 0x87, 0xDF, 0xB8, 0xE7, 0x52, 0xAD, 0x99,
    0xCD, 0x47, 0x88, 0xE0, 0xBA, 0xF0, 0x3D, 0x9D, 0x87, 0x93, 0xAD, 0xA7,
    0x45, 0x67, 0xF0, 0x1E, 0x46, 0xD7, 0x83, 0x9A, 0xE5, 0x49, 0x76, 0x21,
    0x82, 0xCB, 0x82, 0x67, 0xA5, 0xFF, 0x63, 0xD8, 0x97, 0x51, 0xB4, 0x44,
    0xA0, 0x76, 0xBA, 0x40, 0xD8, 0xAB, 0xA6, 0xEB, 0x70, 0xD5, 0xA4, 0x38,
    0xB9, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA9 = {{
    0x7A, 0x96, 0xE1, 0xA4, 0xDE, 0xEB, 0x04, 0x23, 0xD6, 0xC4, 0xFD, 0x98,
    0xDF, 0x25, 0x83, 0x25, 0x2C, 0x3E, 0xE0, 0x23, 0x35, 0x9F, 0x4B, 0x92,
    0x08, 0xAB, 0xE8, 0x7C, 0xB1, 0xD8, 0x4D, 0x85
}};

static const unsigned char kPublicKeyICA9[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xE5, 0x44, 0x79,
    0xD4, 0x75, 0x3A, 0xBD, 0x25, 0x6F, 0x89, 0xF7, 0x94, 0xE9, 0x23, 0xAE,
    0x19, 0x38, 0x09, 0xA7, 0x75, 0x9C, 0x5A, 0x08, 0x4A, 0xC2, 0xC6, 0xA4,
    0x95, 0x13, 0x96, 0x95, 0x4E, 0xFA, 0xF1, 0xC3, 0xD2, 0x7A, 0xBC, 0x4A,
    0xEE, 0x65, 0x2D, 0xE2, 0xB1, 0x57, 0x49, 0x02, 0x84, 0x7C, 0x35, 0x31,
    0x8A, 0xBB, 0xCB, 0x75, 0x9C, 0x14, 0x84, 0x52, 0x85, 0x40, 0xD2, 0x1D,
    0x03, 0xAE, 0x2B, 0x38, 0xA2, 0x7F, 0xEE, 0xE8, 0x3C, 0x51, 0x5B, 0x82,
    0x11, 0x9E, 0xE2, 0xC9, 0x0B, 0x7B, 0x66, 0xF2, 0xE5, 0x35, 0x64, 0x4B,
    0xF1, 0x98, 0xD9, 0x60, 0x0A, 0xA2, 0x8B, 0xB2, 0xD3, 0x96, 0x35, 0xBE,
    0x2D, 0xB4, 0x7E, 0xAC, 0x75, 0x73, 0x5F, 0xC0, 0x78, 0xC1, 0x91, 0x3E,
    0xB7, 0xB2, 0x53, 0x4F, 0x78, 0x7D, 0x58, 0x93, 0x12, 0x3B, 0xA9, 0xB3,
    0x8F, 0xA7, 0xF6, 0x7F, 0x4C, 0x2F, 0x7B, 0xFA, 0x41, 0xCA, 0xF5, 0x5A,
    0xF4, 0x8A, 0x5B, 0xFE, 0x82, 0x18, 0x90, 0xE5, 0x15, 0x01, 0xD3, 0xD8,
    0x83, 0x6B, 0x02, 0xA3, 0xAE, 0x54, 0x5C, 0xD9, 0x0B, 0x65, 0x00, 0xD6,
    0x06, 0xF6, 0x4E, 0x52, 0x2C, 0x9C, 0x06, 0x1B, 0x11, 0x53, 0xA5, 0x7E,
    0xBE, 0xAC, 0x5E, 0x01, 0xF1, 0x50, 0xF2, 0xC0, 0x26, 0xC6, 0xF9, 0xDD,
    0x89, 0x8C, 0x1D, 0xD4, 0x7A, 0x59, 0xC2, 0xD7, 0xF8, 0x7A, 0x03, 0x6A,
    0xBD, 0xC5, 0x75, 0x04, 0xED, 0x29, 0x90, 0xD8, 0x24, 0x75, 0x12, 0x38,
    0x24, 0xF2, 0x56, 0xB4, 0x87, 0xB2, 0x55, 0x0F, 0x26, 0x1D, 0xD0, 0x6B,
    0x32, 0xDF, 0x05, 0xFA, 0x73, 0x94, 0xB0, 0x6B, 0x41, 0xE7, 0x2D, 0xF0,
    0x24, 0x48, 0xA8, 0x5B, 0x03, 0x34, 0xE7, 0x48, 0x92, 0x4E, 0x99, 0x3A,
    0x6B, 0x96, 0x8E, 0x8E, 0x48, 0x52, 0xA5, 0xE8, 0x13, 0x54, 0xCF, 0x8D,
    0xA1, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA10 = {{
    0x61, 0x79, 0xBE, 0x64, 0x83, 0xD6, 0x71, 0xCE, 0x4F, 0xEE, 0x95, 0x5B,
    0xF3, 0x6A, 0x1D, 0xDB, 0xC9, 0x51, 0x2F, 0xF4, 0x0C, 0xF8, 0xA5, 0x1C,
    0x4F, 0x31, 0x2A, 0x93, 0x9A, 0x94, 0x07, 0xF7
}};

static const unsigned char kPublicKeyICA10[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xAF, 0xA1, 0x27,
    0x5B, 0x18, 0x01, 0xF9, 0x16, 0x7C, 0xB9, 0x1D, 0xBB, 0x34, 0xC4, 0x45,
    0xCF, 0x7A, 0x7B, 0x7A, 0xB7, 0xC9, 0x52, 0xC6, 0xC1, 0xBE, 0x14, 0x57,
    0xF5, 0xDC, 0xF7, 0xA3, 0xB9, 0x1F, 0x63, 0x5B, 0xDE, 0x95, 0x36, 0x23,
    0x3A, 0xD4, 0xDF, 0x33, 0xA5, 0x75, 0xF6, 0x2B, 0x70, 0x6B, 0xE6, 0x46,
    0xA9, 0x94, 0x84, 0x22, 0xD0, 0xC7, 0xF4, 0x2C, 0x8D, 0x20, 0xBC, 0x2F,
    0x8E, 0x08, 0x44, 0xA4, 0x99, 0x30, 0xE2, 0x2B, 0x37, 0x1E, 0x1A, 0xB7,
    0x0D, 0x98, 0x20, 0x40, 0x18, 0xEC, 0x7E, 0x7C, 0x65, 0xD7, 0xF7, 0x1E,
    0x7A, 0x50, 0x1C, 0x27, 0xFE, 0x82, 0x5F, 0xFA, 0xDF, 0xAC, 0xA6, 0x4E,
    0xB4, 0x91, 0x2F, 0x73, 0xFB, 0x20, 0xFC, 0x70, 0x3F, 0x5E, 0x58, 0x7B,
    0xAC, 0xC6, 0x1C, 0xAD, 0xEF, 0x0A, 0xB1, 0xB3, 0x12, 0x2E, 0xB8, 0xC3,
    0x60, 0xCB, 0xF7, 0x71, 0x5F, 0x18, 0xDD, 0x85, 0x64, 0x06, 0xA7, 0x17,
    0x60, 0x81, 0x72, 0x6D, 0xE2, 0x24, 0x57, 0xCA, 0x3D, 0x1D, 0x87, 0x75,
    0x05, 0x95, 0xDE, 0x38, 0x8A, 0xE0, 0xC2, 0xF7, 0xCB, 0x2F, 0xA6, 0xB9,
    0x24, 0x50, 0x14, 0x17, 0x12, 0x77, 0xFB, 0x41, 0xA8, 0xA1, 0x79, 0xBC,
    0xC0, 0x87, 0x06, 0x34, 0xF2, 0xAF, 0x87, 0x12, 0xB6, 0x66, 0x24, 0xDD,
    0x3E, 0xBA, 0x4E, 0x34, 0x02, 0xF2, 0x1B, 0xAB, 0x1D, 0x79, 0x72, 0x41,
    0x16, 0x0E, 0x1F, 0x9B, 0x35, 0x40, 0xD0, 0xC6, 0x07, 0xA7, 0x91, 0x53,
    0x55, 0x19, 0x0C, 0xB1, 0x1B, 0x42, 0x20, 0x41, 0xC5, 0x2A, 0xA8, 0x26,
    0x8D, 0x44, 0x50, 0x1B, 0x0B, 0x21, 0xB2, 0x16, 0xA2, 0x1B, 0xF3, 0xBD,
    0xC2, 0x1D, 0xAF, 0x4F, 0x41, 0x43, 0xAD, 0x3A, 0x76, 0x45, 0x3C, 0x2B,
    0xD3, 0x71, 0x31, 0x43, 0x37, 0xB6, 0x68, 0xA6, 0x5D, 0x8C, 0x50, 0x2B,
    0x8F, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA11 = {{
    0x9C, 0x1E, 0xB3, 0xEA, 0x88, 0xAA, 0x1C, 0x1E, 0x0A, 0xE7, 0x9C, 0x2B,
    0x43, 0xFF, 0x46, 0xD4, 0xC8, 0xF5, 0xF4, 0xFF, 0x43, 0x71, 0xFB, 0x83,
    0xE4, 0x81, 0x13, 0x8D, 0xAD, 0x82, 0x42, 0xE8
}};

static const unsigned char kPublicKeyICA11[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xA8, 0xB2, 0x65,
    0xFB, 0x07, 0xB0, 0x21, 0xBE, 0x11, 0xD1, 0xDA, 0x5D, 0x89, 0xB5, 0xAA,
    0xC2, 0xFD, 0xD3, 0x27, 0x8D, 0x3A, 0x29, 0x2F, 0x3E, 0xD4, 0x87, 0xC1,
    0x1B, 0x51, 0x39, 0x48, 0x73, 0x4A, 0xD5, 0x52, 0x5D, 0x59, 0x93, 0x8B,
    0xF3, 0x3D, 0x57, 0x6A, 0x77, 0x7D, 0x43, 0x3A, 0xED, 0xE5, 0xC2, 0x0E,
    0xFB, 0xE0, 0xFB, 0x0C, 0x65, 0x65, 0x74, 0xA9, 0x1F, 0x3C, 0x56, 0x77,
    0xC8, 0x5C, 0x8F, 0xA3, 0xCB, 0xB4, 0x68, 0xBC, 0xE9, 0x0F, 0xE4, 0x52,
    0x46, 0x1B, 0xB2, 0x23, 0x6F, 0x66, 0x9D, 0xB7, 0xCF, 0xD9, 0x48, 0xE2,
    0x7D, 0x17, 0x26, 0x45, 0x4F, 0xA5, 0x14, 0x10, 0x08, 0xCE, 0xEC, 0x18,
    0xE0, 0x78, 0x8E, 0x25, 0xB7, 0xAE, 0x23, 0xBD, 0xAE, 0x56, 0x33, 0x1F,
    0x5B, 0x02, 0x41, 0xE1, 0x22, 0x6D, 0x85, 0x8E, 0xB0, 0x87, 0x73, 0xF8,
    0xBF, 0x3A, 0x06, 0xF7, 0xDA, 0x70, 0xCB, 0x14, 0x1F, 0x1E, 0xFF, 0x78,
    0x9D, 0xC4, 0x7A, 0xFF, 0x76, 0x32, 0x35, 0x28, 0x16, 0xD4, 0xBF, 0xBC,
    0x2B, 0x4E, 0xD2, 0x86, 0x50, 0x14, 0x7A, 0x8D, 0x3F, 0x8F, 0x9E, 0x53,
    0x0B, 0xB5, 0x83, 0x6E, 0x00, 0x82, 0xB0, 0x08, 0x6F, 0x22, 0xF4, 0x26,
    0x33, 0x19, 0xCC, 0x82, 0xC7, 0x4C, 0xA0, 0x1B, 0xD2, 0x62, 0x33, 0xF7,
    0x75, 0x0B, 0x57, 0x4A, 0xDF, 0xDD, 0x68, 0xCB, 0xFD, 0x6F, 0xB8, 0xB3,
    0x8F, 0x8E, 0x45, 0x8D, 0xEE, 0xF2, 0xA2, 0xFD, 0x71, 0xF5, 0xE0, 0x1B,
    0x3E, 0x62, 0x00, 0x35, 0x98, 0x19, 0x6B, 0xA3, 0x1B, 0x1A, 0xA3, 0x5D,
    0xDE, 0x49, 0xB9, 0x20, 0x0D, 0x44, 0x8F, 0x58, 0x3C, 0xDD, 0x52, 0x6D,
    0x03, 0x7A, 0x33, 0xB3, 0x06, 0x7A, 0xC7, 0x49, 0x23, 0xC5, 0x2A, 0x24,
    0xB6, 0x96, 0x12, 0x4C, 0x16, 0xB3, 0x3A, 0xFC, 0x46, 0x03, 0xEC, 0xBB,
    0xF9, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA12 = {{
    0xFF, 0xFA, 0x45, 0x0D, 0x1D, 0xB6, 0x31, 0x13, 0xA7, 0x43, 0x70, 0x19,
    0x6E, 0xCB, 0xDB, 0xA4, 0x3F, 0x74, 0xF4, 0xBD, 0x63, 0xAD, 0x8E, 0xFD,
    0x35, 0x62, 0xB3, 0xF9, 0xE8, 0x69, 0x1F, 0xDB
}};

static const unsigned char kPublicKeyICA12[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xD9, 0xE3, 0x76,
    0x97, 0x6A, 0xEF, 0x2D, 0x16, 0xD5, 0xF9, 0xDC, 0x9D, 0x0E, 0x65, 0x12,
    0x65, 0x72, 0xC0, 0xE9, 0x11, 0x12, 0x37, 0x09, 0x5D, 0x54, 0xF4, 0x8F,
    0x3C, 0xDB, 0xF2, 0xE9, 0x42, 0x9F, 0x4E, 0xF2, 0xD0, 0x4E, 0xC8, 0x10,
    0x31, 0xE8, 0x1B, 0xFE, 0x5B, 0xFB, 0xC8, 0xD3, 0xFB, 0x77, 0x25, 0xC4,
    0x69, 0xFA, 0x53, 0x03, 0x16, 0x12, 0x7F, 0x23, 0x9F, 0x4C, 0xFB, 0x35,
    0x60, 0xAE, 0xFB, 0xA5, 0x94, 0xD3, 0x5A, 0x97, 0x38, 0x91, 0x6E, 0x87,
    0xE4, 0xB5, 0xA1, 0x6E, 0x23, 0x1C, 0x7A, 0x41, 0x55, 0x27, 0xA3, 0x9E,
    0x6E, 0xF3, 0xD0, 0xA7, 0x19, 0x52, 0x0C, 0x7C, 0xBC, 0xEC, 0xB6, 0xB8,
    0x54, 0x40, 0x77, 0x0E, 0x67, 0x14, 0x0D, 0x19, 0x1B, 0x74, 0xD4, 0x2C,
    0x16, 0x01, 0xE5, 0x57, 0x6C, 0x03, 0x1E, 0xE3, 0x9E, 0xA3, 0x8E, 0x72,
    0xA6, 0x63, 0x3A, 0xED, 0x25, 0xEC, 0x15, 0x2F, 0xE8, 0xCE, 0x52, 0x1E,
    0xCB, 0x50, 0x39, 0x36, 0x7E, 0xC2, 0xEC, 0x7C, 0xCA, 0x4A, 0xB8, 0x73,
    0x91, 0xC8, 0x88, 0x98, 0x31, 0x0F, 0x2E, 0x68, 0x45, 0x53, 0x22, 0x66,
    0xF1, 0xF5, 0xBF, 0xF9, 0x11, 0x88, 0xB6, 0x36, 0x8E, 0xAA, 0x1A, 0xB8,
    0xC9, 0x18, 0x90, 0x44, 0xBE, 0xBD, 0xDD, 0xB1, 0x81, 0x98, 0xBE, 0xEB,
    0x1F, 0xF6, 0x28, 0x85, 0xB3, 0xA4, 0xA1, 0xAE, 0x14, 0xD2, 0x91, 0x9D,
    0xD3, 0xB1, 0x0B, 0xEC, 0x72, 0x3D, 0x43, 0xEB, 0xD3, 0x79, 0x2A, 0x7D,
    0xAD, 0x79, 0xA5, 0xB5, 0xA0, 0xDD, 0x88, 0x89, 0x6E, 0xB4, 0xC8, 0x11,
    0xB6, 0x11, 0xED, 0x18, 0x50, 0x43, 0x2E, 0xD7, 0xCE, 0x18, 0x58, 0xEB,
    0xCE, 0x2E, 0xE9, 0x9E, 0x20, 0x86, 0xFE, 0x97, 0xCD, 0xB2, 0x9C, 0xC1,
    0xAF, 0x24, 0x02, 0x38, 0x60, 0x6B, 0xCC, 0x66, 0xC3, 0x04, 0x72, 0xD3,
    0xF1, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA13 = {{
    0xE0, 0x2C, 0x66, 0xCD, 0x6F, 0x96, 0xFC, 0x1B, 0x5A, 0x43, 0xC4, 0xA5,
    0x19, 0xB6, 0x6F, 0x0B, 0xFE, 0xA7, 0x58, 0x0D, 0xC1, 0x02, 0x3E, 0xFE,
    0xFF, 0x9C, 0xE4, 0x9E, 0xB1, 0x7E, 0x3F, 0x79
}};

static const unsigned char kPublicKeyICA13[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xBB, 0xE5, 0x0F,
    0xD2, 0x1C, 0x6B, 0xBA, 0x23, 0xA9, 0x2F, 0x87, 0xEC, 0xDB, 0x92, 0x3F,
    0xAF, 0xC8, 0xD4, 0xB2, 0x59, 0x24, 0x2E, 0xC8, 0xCD, 0x00, 0xEF, 0x09,
    0xFB, 0xF6, 0x19, 0xE7, 0x6C, 0x8A, 0x1A, 0x3B, 0xB6, 0xCC, 0xEF, 0x3A,
    0x40, 0x6C, 0x93, 0xF8, 0xE3, 0x1B, 0xB7, 0xE5, 0x1C, 0x92, 0x65, 0xE8,
    0x5E, 0x64, 0x83, 0x90, 0xF1, 0x24, 0x4A, 0xD1, 0xC5, 0x3D, 0x8C, 0x3B,
    0x71, 0x66, 0x31, 0x58, 0xD8, 0x55, 0xC2, 0xCC, 0xD3, 0xEA, 0x0E, 0x66,
    0x88, 0x59, 0x14, 0x77, 0xED, 0x12, 0xC5, 0x96, 0x54, 0x7F, 0x97, 0x28,
    0x3B, 0x5E, 0xCA, 0xF7, 0x1B, 0xD3, 0x4B, 0x10, 0xC3, 0x45, 0x3A, 0x4D,
    0xCA, 0x36, 0x5A, 0xFF, 0x4F, 0x86, 0xDD, 0x9E, 0x69, 0xDF, 0xD5, 0x4A,
    0xD0, 0xB8, 0x9F, 0x8D, 0x31, 0x70, 0x76, 0x63, 0x33, 0xB0, 0xB8, 0xF4,
    0xB0, 0x45, 0x28, 0xB3, 0x5D, 0xF2, 0x2F, 0xC5, 0xA4, 0xD9, 0x30, 0x6F,
    0x9F, 0x69, 0x23, 0x42, 0x6D, 0x7D, 0x73, 0x29, 0x56, 0x61, 0xCC, 0x56,
    0xC9, 0xAE, 0xED, 0x13, 0x33, 0xB4, 0x0E, 0xD0, 0x25, 0xE3, 0x06, 0xC1,
    0x9A, 0x26, 0xDB, 0x8E, 0x89, 0xA6, 0xA0, 0xF9, 0x30, 0xE6, 0x92, 0xD0,
    0xEC, 0x77, 0xB9, 0xA8, 0x0C, 0x8E, 0x83, 0x5D, 0x6B, 0xB9, 0x49, 0xF2,
    0xFB, 0x1C, 0xE4, 0x79, 0xC8, 0xB3, 0x90, 0x88, 0xE9, 0x92, 0x24, 0x8A,
    0x18, 0x7E, 0xE3, 0x5C, 0xEF, 0xC0, 0x4B, 0xDD, 0xFD, 0x09, 0x14, 0x4C,
    0x9C, 0x7A, 0xB3, 0x56, 0x84, 0x96, 0xDB, 0x08, 0xA8, 0xE1, 0xCD, 0x40,
    0x94, 0xF5, 0x12, 0xF4, 0x63, 0x38, 0x0C, 0x51, 0xE4, 0x03, 0x63, 0xC4,
    0x76, 0x54, 0xB7, 0x59, 0x25, 0xCE, 0x62, 0xDE, 0x73, 0x3F, 0xAB, 0x15,
    0x56, 0xC5, 0xBC, 0x99, 0x8C, 0x3A, 0x46, 0x3F, 0x13, 0x0E, 0xF4, 0x53,
    0x1D, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA14 = {{
    0x3B, 0x60, 0x2E, 0xE9, 0x9F, 0x75, 0x7C, 0x18, 0xE3, 0x08, 0x1E, 0xC0,
    0x72, 0xE9, 0x20, 0x05, 0x0B, 0x83, 0xD7, 0xB7, 0x5E, 0x25, 0x47, 0xE7,
    0xE6, 0x44, 0x2C, 0x40, 0xF4, 0xA9, 0xA1, 0xD0
}};

static const unsigned char kPublicKeyICA14[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xAD, 0x16, 0x57,
    0x87, 0xD7, 0x0D, 0xA3, 0xA2, 0x7E, 0x8B, 0xAE, 0x45, 0x7A, 0x1F, 0xCA,
    0xFE, 0xB9, 0x15, 0xB7, 0x5B, 0x9F, 0x7D, 0x16, 0xA5, 0xA5, 0x63, 0xB1,
    0xF5, 0x6E, 0x17, 0x7F, 0x35, 0xD8, 0x4A, 0x8C, 0x97, 0xBF, 0x77, 0x84,
    0x5E, 0xC6, 0x21, 0x81, 0xAF, 0x72, 0xEF, 0xCF, 0xDE, 0x46, 0xB7, 0xE7,
    0x49, 0x61, 0x8E, 0xFC, 0x84, 0x12, 0xBC, 0x30, 0xEA, 0xF8, 0x5B, 0x78,
    0x6C, 0x3E, 0x12, 0x23, 0x33, 0x29, 0x80, 0x34, 0x6A, 0x1E, 0x8D, 0x3C,
    0x15, 0xE9, 0x47, 0x9E, 0x33, 0x27, 0x90, 0x73, 0x52, 0xD0, 0xBF, 0xCE,
    0x0D, 0x68, 0xE5, 0x5A, 0x90, 0x71, 0xB2, 0xF2, 0xBD, 0x7E, 0x69, 0xE0,
    0x92, 0xDC, 0x44, 0x9F, 0x4B, 0xA3, 0xC2, 0x58, 0x16, 0x1A, 0x35, 0x18,
    0x88, 0x3A, 0x82, 0x15, 0xFC, 0x41, 0x8C, 0x72, 0x11, 0x2E, 0xC4, 0xED,
    0xD3, 0x7E, 0x68, 0xF2, 0x00, 0x01, 0xE2, 0x71, 0xC3, 0x91, 0x91, 0xFD,
    0xF3, 0xBE, 0x11, 0xE7, 0x62, 0xB4, 0xAF, 0xC8, 0xF0, 0x12, 0xBC, 0xB2,
    0x0E, 0x58, 0x5C, 0xFF, 0x08, 0xCB, 0xCB, 0x91, 0xD2, 0xD0, 0x11, 0x87,
    0x72, 0x04, 0x99, 0x63, 0x12, 0xA6, 0x6F, 0x7D, 0x40, 0x76, 0xB7, 0xE8,
    0x89, 0xCE, 0xCD, 0x5A, 0x73, 0x18, 0x8A, 0x73, 0xAF, 0xFD, 0x21, 0x68,
    0xE5, 0x26, 0x74, 0x12, 0x2C, 0xC3, 0xE6, 0x7D, 0x1D, 0x9A, 0xC8, 0x12,
    0xCD, 0x38, 0xCB, 0x47, 0xA6, 0x54, 0x8F, 0xAD, 0x9F, 0xFA, 0xB1, 0xDF,
    0xB0, 0xBF, 0xE7, 0x12, 0x32, 0x76, 0xA7, 0xA5, 0xD7, 0x46, 0xF8, 0x62,
    0x15, 0x54, 0x78, 0xBA, 0x9E, 0x4D, 0xD8, 0x99, 0x62, 0x9E, 0xE8, 0x45,
    0x93, 0x8C, 0x14, 0x7E, 0x9C, 0xE9, 0xF7, 0x2A, 0x7E, 0x56, 0xE3, 0xBD,
    0xF1, 0x65, 0xC8, 0x6B, 0xB9, 0xE5, 0x16, 0x1E, 0x22, 0x29, 0xEC, 0xCA,
    0xD9, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA15 = {{
    0x09, 0xA7, 0x9F, 0x65, 0xE5, 0x55, 0x72, 0xAA, 0xFD, 0xDB, 0x74, 0xE9,
    0xF8, 0xFA, 0x01, 0x45, 0x1D, 0x8E, 0xA4, 0x17, 0x48, 0xAB, 0x50, 0x75,
    0xF9, 0xB1, 0x12, 0xB0, 0x6F, 0xC8, 0x08, 0xAE
}};

static const unsigned char kPublicKeyICA15[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xDD, 0xFE, 0x56,
    0x5F, 0x53, 0x05, 0x59, 0x8F, 0x7C, 0xF4, 0x8B, 0x90, 0x38, 0xED, 0x51,
    0xE6, 0x8A, 0xAC, 0x78, 0x1F, 0x7B, 0x33, 0x74, 0x7F, 0x33, 0xAA, 0x6A,
    0x13, 0x6F, 0x7C, 0xDF, 0x82, 0x8F, 0x7B, 0xC4, 0x11, 0xAE, 0x5C, 0x2B,
    0x1A, 0xDB, 0x9A, 0x95, 0x13, 0xE8, 0x64, 0x48, 0x37, 0x6C, 0x6B, 0x8C,
    0xDD, 0x42, 0xE1, 0xCE, 0xE4, 0x0C, 0xDF, 0xA2, 0x41, 0x9E, 0x78, 0x8B,
    0x26, 0xBF, 0xA1, 0x6E, 0x59, 0x8E, 0x10, 0xF8, 0x31, 0xAD, 0x13, 0xD2,
    0x1F, 0x84, 0xCC, 0xE0, 0x88, 0x59, 0x5D, 0x0C, 0xD0, 0xAB, 0x24, 0xC8,
    0x1F, 0xCB, 0xE0, 0x13, 0x72, 0xD3, 0xF3, 0x90, 0xFB, 0xB0, 0x1D, 0x36,
    0x4E, 0xB6, 0xDC, 0x42, 0xC2, 0x87, 0xD8, 0x38, 0x6D, 0x18, 0x23, 0x3F,
    0xDF, 0x95, 0x8B, 0xF1, 0x40, 0x89, 0xEE, 0x43, 0xD5, 0x09, 0x1F, 0xBB,
    0xDA, 0x96, 0x4C, 0xB7, 0x23, 0x33, 0xE9, 0x48, 0xEA, 0x9E, 0x1D, 0x30,
    0xFD, 0x72, 0x90, 0x81, 0x74, 0x1C, 0xE7, 0x8A, 0xA9, 0x8C, 0xD9, 0x4B,
    0x30, 0x35, 0x47, 0x60, 0xDC, 0x28, 0x34, 0x92, 0x39, 0xD6, 0xEA, 0x3E,
    0x1F, 0x70, 0x1B, 0xCA, 0x28, 0x64, 0xA5, 0x2E, 0x90, 0x3E, 0x25, 0x90,
    0xE7, 0x70, 0x10, 0x55, 0x1E, 0xAE, 0x9B, 0x5C, 0xB3, 0x88, 0xB7, 0x00,
    0x76, 0x7B, 0xF7, 0xB6, 0x4A, 0xD3, 0x69, 0x1E, 0x00, 0xDF, 0xB1, 0xE6,
    0x4D, 0xD1, 0x18, 0x3A, 0xAD, 0x7E, 0xB9, 0x6C, 0x6D, 0x51, 0x81, 0x75,
    0xFE, 0xC4, 0xAA, 0xE6, 0x17, 0x37, 0xBA, 0x2B, 0x3B, 0xD4, 0x4E, 0xFC,
    0xEC, 0xE6, 0x60, 0x7B, 0x20, 0x7F, 0xC3, 0x74, 0xD9, 0xA3, 0x67, 0x80,
    0x8B, 0x4A, 0x34, 0xDC, 0x25, 0x6E, 0x9B, 0xA5, 0x67, 0x97, 0x54, 0xAC,
    0x6C, 0x8E, 0x7A, 0x64, 0x20, 0xDA, 0xD6, 0xAA, 0x59, 0x6B, 0x27, 0x28,
    0x99, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const net::SHA256HashValue kFingerprintICA16 = {{
    0x49, 0x0D, 0xC9, 0x48, 0xA1, 0x6B, 0x02, 0xEA, 0xC6, 0xDA, 0x47, 0x99,
    0x36, 0x4D, 0xD2, 0xAF, 0x04, 0xF7, 0xEA, 0x46, 0x30, 0xCF, 0x33, 0x79,
    0x73, 0x10, 0xAC, 0x4C, 0x07, 0xB4, 0x67, 0xAD
}};

static const unsigned char kPublicKeyICA16[] = {
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xD2, 0xB0, 0xC9,
    0xB3, 0x88, 0xCD, 0x93, 0x23, 0x6A, 0x46, 0xD3, 0x69, 0x0A, 0xD7, 0xFF,
    0xE1, 0x51, 0x7E, 0x29, 0xA9, 0x6B, 0x71, 0x68, 0xC0, 0xCF, 0x69, 0xA3,
    0xE8, 0xAD, 0xA9, 0x58, 0x22, 0x18, 0x45, 0x9F, 0x04, 0x86, 0x7F, 0xBA,
    0x1B, 0xC8, 0x1C, 0x3A, 0x99, 0x80, 0xFF, 0x73, 0x8C, 0x65, 0xE7, 0xDA,
    0x76, 0x5F, 0xCE, 0xD6, 0xB8, 0x28, 0xCA, 0xC9, 0x20, 0x7A, 0x4B, 0xB9,
    0xC2, 0xCE, 0x89, 0x46, 0x40, 0x68, 0x60, 0x7E, 0x3B, 0xE0, 0x88, 0x1C,
    0x1B, 0xDC, 0xEE, 0xDC, 0x06, 0x42, 0x5B, 0x28, 0x1A, 0xCD, 0xCB, 0x3E,
    0x4E, 0xF3, 0x19, 0x0D, 0x83, 0xE2, 0x2A, 0x9D, 0x8C, 0xA3, 0x78, 0xE8,
    0x1D, 0x6B, 0x79, 0x7C, 0x48, 0xA9, 0xE1, 0x8B, 0x56, 0x8B, 0x66, 0x63,
    0xCE, 0x54, 0xA7, 0xD2, 0x1F, 0xE6, 0x81, 0x5B, 0x0C, 0x63, 0xB9, 0xF0,
    0x94, 0xBD, 0xC1, 0xB2, 0x5A, 0x7F, 0xEC, 0x9A, 0x09, 0x4F, 0xB8, 0x02,
    0x6D, 0x7F, 0x59, 0x64, 0xBE, 0x01, 0xF8, 0x83, 0xD4, 0xF0, 0x0E, 0x96,
    0x78, 0xDC, 0xF4, 0x9D, 0x4C, 0x91, 0x4C, 0x08, 0xB3, 0xFA, 0x28, 0x55,
    0xB7, 0xF6, 0x06, 0x1E, 0x34, 0xC3, 0x79, 0x90, 0xAB, 0x2C, 0x4E, 0x2E,
    0xD9, 0xE4, 0x78, 0x87, 0xCF, 0xF2, 0xAA, 0x83, 0x2D, 0x74, 0x8F, 0xE4,
    0xDA, 0xEE, 0xD0, 0x24, 0x06, 0xDE, 0x40, 0xE7, 0xC9, 0xC0, 0x02, 0xF5,
    0x8D, 0x7D, 0xDE, 0x28, 0x03, 0x8A, 0xAE, 0x21, 0xBD, 0xF1, 0x29, 0x36,
    0xEB, 0xD7, 0xC9, 0x3B, 0x34, 0xE3, 0x08, 0x8C, 0xCA, 0x25, 0xEE, 0x3C,
    0xE4, 0x07, 0x49, 0xB9, 0xB8, 0xDB, 0x35, 0x90, 0x99, 0x50, 0x99, 0xC2,
    0x7D, 0x6A, 0x3A, 0x33, 0x31, 0xC7, 0x61, 0x13, 0xB7, 0x71, 0x10, 0x80,
    0xC1, 0x8C, 0xE0, 0x69, 0xA2, 0xDD, 0xA3, 0xE5, 0x52, 0x8A, 0xF5, 0xEF,
    0x63, 0x02, 0x03, 0x01, 0x00, 0x01
};

// Info for trusted ICA certs.
struct ICACertInfo {
  const net::SHA256HashValue* fingerprint;
  const unsigned char* public_key;
  size_t public_key_size;
};

// Default list of allowed / trusted ICAs.
static const ICACertInfo kAllowedICAs[] = {
    {&kFingerprintICA1, kPublicKeyICA1, sizeof(kPublicKeyICA1)},
    {&kFingerprintICA2, kPublicKeyICA2, sizeof(kPublicKeyICA2)},
    {&kFingerprintICA3, kPublicKeyICA3, sizeof(kPublicKeyICA3)},
    {&kFingerprintICA4, kPublicKeyICA4, sizeof(kPublicKeyICA4)},
    {&kFingerprintICA5, kPublicKeyICA5, sizeof(kPublicKeyICA5)},
    {&kFingerprintICA6, kPublicKeyICA6, sizeof(kPublicKeyICA6)},
    {&kFingerprintICA7, kPublicKeyICA7, sizeof(kPublicKeyICA7)},
    {&kFingerprintICA8, kPublicKeyICA8, sizeof(kPublicKeyICA8)},
    {&kFingerprintICA9, kPublicKeyICA9, sizeof(kPublicKeyICA9)},
    {&kFingerprintICA10, kPublicKeyICA10, sizeof(kPublicKeyICA10)},
    {&kFingerprintICA11, kPublicKeyICA11, sizeof(kPublicKeyICA11)},
    {&kFingerprintICA12, kPublicKeyICA12, sizeof(kPublicKeyICA12)},
    {&kFingerprintICA13, kPublicKeyICA13, sizeof(kPublicKeyICA13)},
    {&kFingerprintICA14, kPublicKeyICA14, sizeof(kPublicKeyICA14)},
    {&kFingerprintICA15, kPublicKeyICA15, sizeof(kPublicKeyICA15)},
    {&kFingerprintICA16, kPublicKeyICA16, sizeof(kPublicKeyICA16)}};

// Built in public key for verifying trusted authorities data.
const uint8 kPublicKey[] = {
    0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
    0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00,
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xC3, 0x8D, 0xEC,
    0xF5, 0x94, 0x4C, 0x66, 0x90, 0xFC, 0x08, 0xED, 0x6E, 0x7D, 0xD0, 0xD6,
    0x93, 0x62, 0x94, 0x86, 0xFB, 0xBA, 0x6F, 0x06, 0xD5, 0x26, 0xAD, 0x52,
    0x84, 0x5A, 0x68, 0x52, 0xF1, 0xA9, 0x46, 0x04, 0x78, 0xC0, 0xA1, 0x0F,
    0x72, 0xCE, 0x1C, 0xB3, 0xD2, 0x34, 0xFA, 0x28, 0xD0, 0xA6, 0x2D, 0xDF,
    0xDF, 0x0C, 0x6C, 0x30, 0xCF, 0xF3, 0x67, 0x0E, 0xCE, 0xDD, 0x2D, 0x02,
    0x20, 0x05, 0x7D, 0x7F, 0x81, 0x19, 0xBC, 0x42, 0x74, 0xC0, 0x11, 0xC4,
    0x7F, 0x31, 0x0B, 0x63, 0xE7, 0x2E, 0xFE, 0xA4, 0x20, 0xE5, 0xD8, 0x19,
    0xE1, 0xBE, 0x25, 0x41, 0x88, 0xFA, 0xBB, 0x51, 0xFF, 0x84, 0x9E, 0x37,
    0x03, 0xAE, 0xAF, 0x41, 0x4E, 0x4B, 0x75, 0xF8, 0x73, 0x79, 0x20, 0x2B,
    0xA0, 0xE9, 0xFC, 0xD6, 0x03, 0x50, 0x1B, 0x2A, 0x6B, 0x31, 0xE8, 0x75,
    0x13, 0xB6, 0xEC, 0xE4, 0x5D, 0x2A, 0x3E, 0xE5, 0x49, 0x44, 0xBF, 0xE2,
    0xC2, 0x4A, 0x7B, 0xC8, 0xC2, 0xC9, 0x89, 0xEF, 0x1A, 0x67, 0x37, 0xB7,
    0x52, 0x93, 0x86, 0x69, 0xA4, 0x1A, 0x2C, 0x69, 0xD2, 0x03, 0x93, 0x15,
    0x02, 0x97, 0xB6, 0xF7, 0x55, 0x87, 0x5B, 0xAA, 0x73, 0x12, 0x3F, 0xDD,
    0x32, 0x16, 0x3B, 0x8E, 0xD9, 0xD9, 0x49, 0x06, 0x65, 0xAD, 0x28, 0x49,
    0xEA, 0x0F, 0x7E, 0x6A, 0x04, 0xF7, 0x89, 0x79, 0x8F, 0xCE, 0x3E, 0xD2,
    0x89, 0x6C, 0x43, 0x4B, 0x34, 0xBF, 0x42, 0x65, 0x2B, 0x09, 0x2F, 0x5C,
    0xCA, 0x46, 0x1C, 0x22, 0x36, 0x78, 0xEA, 0x98, 0x49, 0xA4, 0x0E, 0xAB,
    0x97, 0x0A, 0x76, 0xCD, 0x52, 0x99, 0x42, 0x44, 0xC2, 0x52, 0x40, 0x62,
    0x34, 0xC4, 0xCD, 0xE3, 0x3D, 0xCA, 0xE1, 0x33, 0x99, 0xE9, 0xF4, 0x6E,
    0xEB, 0xB0, 0x5D, 0xA5, 0x84, 0xA2, 0x5C, 0xC7, 0xBB, 0x14, 0x67, 0x0C,
    0x4F, 0x02, 0x03, 0x01, 0x00, 0x01};

// Verifies signature of the data using built-in public key.
bool VerifySignature(const std::string& signature, const std::string& data) {
  if (signature.size() != 256)
    return false;

  crypto::SignatureVerifier verifier;
  crypto::SignatureVerifier::HashAlgorithm hash_alg =
      crypto::SignatureVerifier::SHA256;
  crypto::SignatureVerifier::HashAlgorithm mask_hash_alg = hash_alg;
  unsigned int hash_len = 32;

  if (!verifier.VerifyInitRSAPSS(
          hash_alg,
          mask_hash_alg,
          hash_len,
          reinterpret_cast<const uint8*>(signature.data()),
          signature.size(),
          kPublicKey,
          sizeof(kPublicKey))) {
    return false;
  }

  verifier.VerifyUpdate(reinterpret_cast<const uint8*>(data.data()),
                        data.size());

  return verifier.VerifyFinal();
}

}  // namespace

AuthorityKeyStore::AuthorityKeyStore() {
  for (size_t i = 0; i < arraysize(kAllowedICAs); i++) {
    certificate_authorities_.insert(
        make_pair(*(kAllowedICAs[i].fingerprint),
                  base::StringPiece(
                      reinterpret_cast<const char*>(kAllowedICAs[i].public_key),
                      kAllowedICAs[i].public_key_size)));
  }
}

AuthorityKeyStore::~AuthorityKeyStore() {
}

base::StringPiece AuthorityKeyStore::GetICAPublicKeyFromFingerprint(
    const net::SHA256HashValue& fingerprint) {
  AuthorityKeysMap::const_iterator it =
      certificate_authorities_.find(fingerprint);
  return (it == certificate_authorities_.end()) ? base::StringPiece()
                                                : (*it).second;
}

base::StringPiece AuthorityKeyStore::GetDefaultICAPublicKey() {
  return GetICAPublicKeyFromFingerprint(kFingerprintICA1);
}

bool AuthorityKeyStore::Load(const std::string& keys) {
  certificate_authorities_.clear();
  authority_keys_.reset(new proto::AuthorityKeys());

  if (!authority_keys_->ParseFromString(keys))
    return false;

  for (int i = 0; i < authority_keys_->keys_size(); i++) {
    const std::string& fingerprint = authority_keys_->keys(i).fingerprint();
    net::SHA256HashValue hash;
    // Ensure it's the size of SHA-256 fingerprint data.
    if (fingerprint.size() != sizeof(hash.data)) {
      return false;
    }
    memcpy(hash.data, fingerprint.data(), sizeof(hash.data));
    const std::string& public_key = authority_keys_->keys(i).public_key();
    certificate_authorities_.insert(
        make_pair(hash, base::StringPiece(public_key)));
  }

  return true;
}

// Trusted certificate authorities.
static base::LazyInstance<AuthorityKeyStore>::Leaky g_authority_keys_store =
    LAZY_INSTANCE_INITIALIZER;

bool SetTrustedCertificateAuthorities(const std::string& keys,
                                      const std::string& signature) {
  std::string decoded_keys;
  if (!base::Base64Decode(keys, &decoded_keys))
    return false;

  std::string decoded_signature;
  if (!base::Base64Decode(signature, &decoded_signature))
    return false;

  if (!VerifySignature(decoded_signature, decoded_keys))
    return false;

  return g_authority_keys_store.Get().Load(decoded_keys);
}

base::StringPiece GetTrustedICAPublicKey(const base::StringPiece& data) {
  AuthorityKeyStore& authority_keys_store = g_authority_keys_store.Get();
  scoped_refptr<net::X509Certificate> ica_cert =
      net::X509Certificate::CreateFromBytes(data.data(), data.size());
  if (!ica_cert.get()) {
    LOG(ERROR) << "Invalid ICA certificate.";
    return "";
  }
  return authority_keys_store.GetICAPublicKeyFromFingerprint(
      net::X509Certificate::CalculateFingerprint256(
          ica_cert->os_cert_handle()));
}

base::StringPiece GetDefaultTrustedICAPublicKey() {
  AuthorityKeyStore& authority_keys_store = g_authority_keys_store.Get();
  return authority_keys_store.GetDefaultICAPublicKey();
}

}  // namespace cast_channel
}  // namespace core_api
}  // namespace extensions
