// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_CHROMEOS_PLATFORM_KEYS_KEY_PERMISSIONS_H_
#define CHROME_BROWSER_CHROMEOS_PLATFORM_KEYS_KEY_PERMISSIONS_H_

#include <string>
#include <vector>

#include "base/callback_forward.h"
#include "base/memory/scoped_ptr.h"
#include "base/memory/weak_ptr.h"

namespace base {
class Value;
}

namespace extensions {
class StateStore;
}

namespace chromeos {

// This class manages permissions for extensions to use private keys through
// chrome.platformKeys .
// It handles the following permissions:
//  * The extension that generated a key has the permission to sign arbitrary
//    data with that key at most once.
//  * The user can explicitly grant an extension the permission to sign
//    arbitrary data with a key an unlimited number of times.
class KeyPermissions {
 public:
  // Allows querying and modifying permissions and registering keys for a
  // specific extension.
  class PermissionsForExtension {
   public:
    // |key_permissions| must not be null and outlive this object.
    // Methods of this object refer implicitly to the extension with the id
    // |extension_id|. Don't use this constructor directly. Call
    // |KeyPermissions::GetPermissionsForExtension| instead.
    PermissionsForExtension(const std::string& extension_id,
                            scoped_ptr<base::Value> state_store_value,
                            KeyPermissions* key_permissions);

    ~PermissionsForExtension();

    // Returns true if the private key matching |public_key_spki_der| can be
    // used for signing by the extension with id |extension_id|.
    bool CanUseKeyForSigning(const std::string& public_key_spki_der);

    // Registers the key |public_key_spki_der| as being generated by the
    // extension with id |extension_id| and marks it for corporate usage.
    void RegisterKeyForCorporateUsage(const std::string& public_key_spki_der);

    // Sets the user granted permission that the extension with id
    // |extension_id| can use the private key matching |public_key_spki_der| for
    // signing.
    void SetUserGrantedPermission(const std::string& public_key_spki_der);

    // Must be called when the extension with id |extension_id| used the private
    // key matching |public_key_spki_der| for signing.
    // Updates the permissions accordingly. E.g. if this extension generated the
    // key and no other permission was granted then the permission to sign with
    // this key is removed.
    void SetKeyUsedForSigning(const std::string& public_key_spki_der);

   private:
    struct KeyEntry;

    // Writes the current |state_store_entries_| to the state store of
    // |extension_id_|.
    void WriteToStateStore();

    // Reads a KeyEntry list from |state| and stores them in
    // |state_store_entries_|.
    void KeyEntriesFromState(const base::Value& state);

    // Converts |state_store_entries_| to a base::Value for storing in the state
    // store.
    scoped_ptr<base::Value> KeyEntriesToState();

    // Returns an existing entry for |public_key_spki_der| from
    // |state_store_entries_|. If there is no existing entry, creates, adds and
    // returns a new entry.
    KeyPermissions::PermissionsForExtension::KeyEntry* GetKeyEntry(
        const std::string& public_key_spki_der);

    const std::string extension_id_;
    std::vector<KeyEntry> state_store_entries_;
    KeyPermissions* const key_permissions_;

    DISALLOW_COPY_AND_ASSIGN(PermissionsForExtension);
  };

  // |extensions_state_store| must not be null and outlive this object.
  explicit KeyPermissions(extensions::StateStore* extensions_state_store);

  ~KeyPermissions();

  using PermissionsCallback =
      base::Callback<void(scoped_ptr<PermissionsForExtension>)>;

  // Passes an object managing the key permissions of the extension with id
  // |extension_id| to |callback|. This can happen synchronously or
  // asynchronously.
  void GetPermissionsForExtension(const std::string& extension_id,
                                  const PermissionsCallback& callback);

 private:
  // Creates a PermissionsForExtension object from |extension_id| and |value|
  // and passes the object to |callback|.
  void CreatePermissionObjectAndPassToCallback(
      const std::string& extension_id,
      const PermissionsCallback& callback,
      scoped_ptr<base::Value> value);

  // Writes |value| to the state store of the extension with id |extension_id|.
  void SetPlatformKeysOfExtension(const std::string& extension_id,
                                  scoped_ptr<base::Value> value);

  extensions::StateStore* const extensions_state_store_;
  base::WeakPtrFactory<KeyPermissions> weak_factory_;

  DISALLOW_COPY_AND_ASSIGN(KeyPermissions);
};

}  // namespace chromeos

#endif  // CHROME_BROWSER_CHROMEOS_PLATFORM_KEYS_KEY_PERMISSIONS_H_
