// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <string>

#include "base/compiler_specific.h"
#include "base/files/file_path.h"
#include "base/values.h"
#include "chrome/browser/download/download_dir_policy_handler.h"
#include "chrome/browser/download/download_prefs.h"
#include "chrome/common/pref_names.h"
#include "components/policy/core/browser/configuration_policy_handler_parameters.h"
#include "components/policy/core/browser/configuration_policy_pref_store.h"
#include "components/policy/core/browser/configuration_policy_pref_store_test.h"
#include "components/policy/core/common/policy_details.h"
#include "components/policy/core/common/policy_map.h"
#include "components/policy/core/common/policy_types.h"
#include "policy/policy_constants.h"

#if defined(OS_CHROMEOS)
#include "chrome/browser/chromeos/drive/file_system_util.h"
#endif

namespace {

const char* kUserIDHash = "deadbeef";

#if defined(OS_CHROMEOS)
const char* kDriveNamePolicyVariableName = "${google_drive}";
const base::FilePath::CharType* kRootRelativeToDriveMount =
    FILE_PATH_LITERAL("root");
const char* kRelativeToDriveRoot = "/home/";

std::string GetExpectedDownloadDirectory() {
  return drive::util::GetDriveMountPointPathForUserIdHash(kUserIDHash)
      .Append(kRootRelativeToDriveMount)
      .value();
}

#endif

}  // namespace

class DownloadDirPolicyHandlerTest
    : public policy::ConfigurationPolicyPrefStoreTest {
 public:
  void SetUp() override {
    recommended_store_ = new policy::ConfigurationPolicyPrefStore(
        policy_service_.get(),
        &handler_list_,
        policy::POLICY_LEVEL_RECOMMENDED);
    handler_list_.AddHandler(
        make_scoped_ptr<policy::ConfigurationPolicyHandler>(
            new DownloadDirPolicyHandler));
  }

  void PopulatePolicyHandlerParameters(
      policy::PolicyHandlerParameters* parameters) override {
    parameters->user_id_hash = kUserIDHash;
  }

 protected:
  scoped_refptr<policy::ConfigurationPolicyPrefStore> recommended_store_;
};

#if !defined(OS_CHROMEOS)
TEST_F(DownloadDirPolicyHandlerTest, SetDownloadDirectory) {
  policy::PolicyMap policy;
  EXPECT_FALSE(store_->GetValue(prefs::kPromptForDownload, NULL));
  policy.Set(policy::key::kDownloadDirectory,
             policy::POLICY_LEVEL_MANDATORY,
             policy::POLICY_SCOPE_USER,
             new base::StringValue(std::string()),
             NULL);
  UpdateProviderPolicy(policy);

  // Setting a DownloadDirectory should disable the PromptForDownload pref.
  const base::Value* value = NULL;
  EXPECT_TRUE(store_->GetValue(prefs::kPromptForDownload, &value));
  ASSERT_TRUE(value);
  bool prompt_for_download = true;
  bool result = value->GetAsBoolean(&prompt_for_download);
  ASSERT_TRUE(result);
  EXPECT_FALSE(prompt_for_download);
}
#endif

#if defined(OS_CHROMEOS)
TEST_F(DownloadDirPolicyHandlerTest, SetDownloadToDrive) {
  EXPECT_FALSE(store_->GetValue(prefs::kPromptForDownload, NULL));

  policy::PolicyMap policy;
  policy.Set(policy::key::kDownloadDirectory,
             policy::POLICY_LEVEL_MANDATORY,
             policy::POLICY_SCOPE_USER,
             new base::StringValue(kDriveNamePolicyVariableName),
             NULL);
  UpdateProviderPolicy(policy);

  const base::Value* value = NULL;
  bool prompt_for_download;
  EXPECT_TRUE(store_->GetValue(prefs::kPromptForDownload, &value));
  EXPECT_TRUE(value);
  EXPECT_TRUE(value->GetAsBoolean(&prompt_for_download));
  EXPECT_FALSE(prompt_for_download);

  bool disable_drive;
  EXPECT_TRUE(store_->GetValue(prefs::kDisableDrive, &value));
  EXPECT_TRUE(value);
  EXPECT_TRUE(value->GetAsBoolean(&disable_drive));
  EXPECT_FALSE(disable_drive);

  std::string download_directory;
  EXPECT_TRUE(store_->GetValue(prefs::kDownloadDefaultDirectory, &value));
  EXPECT_TRUE(value);
  EXPECT_TRUE(value->GetAsString(&download_directory));
  EXPECT_EQ(GetExpectedDownloadDirectory(), download_directory);

  policy.Set(policy::key::kDownloadDirectory,
             policy::POLICY_LEVEL_MANDATORY,
             policy::POLICY_SCOPE_USER,
             new base::StringValue(kUserIDHash),
             NULL);
  UpdateProviderPolicy(policy);
  EXPECT_FALSE(recommended_store_->GetValue(prefs::kDisableDrive, NULL));

  policy.Set(policy::key::kDownloadDirectory,
             policy::POLICY_LEVEL_RECOMMENDED,
             policy::POLICY_SCOPE_USER,
             new base::StringValue(std::string(kDriveNamePolicyVariableName) +
                                   kRelativeToDriveRoot),
             NULL);
  UpdateProviderPolicy(policy);

  EXPECT_FALSE(recommended_store_->GetValue(prefs::kPromptForDownload, NULL));
  EXPECT_FALSE(recommended_store_->GetValue(prefs::kDisableDrive, NULL));

  EXPECT_TRUE(
      recommended_store_->GetValue(prefs::kDownloadDefaultDirectory, &value));
  EXPECT_TRUE(value);
  EXPECT_TRUE(value->GetAsString(&download_directory));
  EXPECT_EQ(GetExpectedDownloadDirectory() + kRelativeToDriveRoot,
            download_directory);

  policy.Set(policy::key::kDownloadDirectory,
             policy::POLICY_LEVEL_RECOMMENDED,
             policy::POLICY_SCOPE_USER,
             new base::StringValue(kUserIDHash),
             NULL);
  UpdateProviderPolicy(policy);

  EXPECT_FALSE(recommended_store_->GetValue(prefs::kPromptForDownload, NULL));
  EXPECT_FALSE(recommended_store_->GetValue(prefs::kDisableDrive, NULL));

  EXPECT_TRUE(
      recommended_store_->GetValue(prefs::kDownloadDefaultDirectory, &value));
  EXPECT_TRUE(value);
  EXPECT_TRUE(value->GetAsString(&download_directory));
  EXPECT_EQ(kUserIDHash, download_directory);
}
#endif
