# !/usr/bin/env python
# Copyright 2014 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

# Generates the hashed_ad_networks.[h,cc] files. Takes an input the name of a
# file with all ad network host patterns, new-line separated. If given an
# optional root output file name, generates the files <root_output_name>.h and
# <root_output_name>.cc. If no output name is given, the output name is
# 'hashed_ad_networks'.
#
# These are found at chrome/browser/extensions/activity_log/hashed_ad_networks.*
# and are used by the ActivityLog and HashedAdNetworkDatabase for recognizing
# ad injection.

from hashlib import sha256
from argparse import ArgumentParser
import sys

_LICENSE = '''\
// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// This file is automatically generated from the script in
// chrome/browser/extensions/activity_log/generate_ad_network_hashes.py.
// DO NOT MODIFY BY HAND!
'''

_H_TEMPLATE = '''\
%(license)s
#ifndef CHROME_BROWSER_EXTENSIONS_ACTIVITY_LOG_HASHED_AD_NETWORKS_H_
#define CHROME_BROWSER_EXTENSIONS_ACTIVITY_LOG_HASHED_AD_NETWORKS_H_

#include "base/basictypes.h"

namespace extensions {

extern const char* kHashedAdNetworks[];
extern const int kNumHashedAdNetworks;

}  // namespace extensions

#endif  // CHROME_BROWSER_EXTENSIONS_ACTIVITY_LOG_HASHED_AD_NETWORKS_H_
'''

_CC_TEMPLATE = '''\
%(license)s
#include "chrome/browser/extensions/activity_log/hashed_ad_networks.h"

#include "base/basictypes.h"

namespace extensions {

const char* const kHashedAdNetworks[] = {
  %(ad_networks)s
};

const int kNumHashedAdNetworks = arraysize(kHashedAdNetworks);

}  // namespace extensions
'''


def Generate(input_filename, output_root_filename):
  '''Generate the .h and .cc files for the hashed_ad_network source files.
  |input_filename|
      The name of the input file, which should have one host to be hashed per
      line.
  |output_root_filename|
      The root name of the output files. This will generate a .h and .cc file,
      like |output_root_filename|.[h,cc].
  '''
  with open(input_filename, 'r') as file:
    hashes = ['"%s"' % sha256(line.strip()).hexdigest()[:16].upper()
              for line in file.readlines()]

  # Hashes should be sorted in C++ so we can do a binary search over them.
  hashes.sort()
  ad_networks = ',\n  '.join(hashes)
  for ext, template in (('.h', _H_TEMPLATE),
                        ('.cc', _CC_TEMPLATE)):
    with open(output_root_filename + ext, 'w') as out:
      out.write(template % {
        'license': _LICENSE,
        'ad_networks': ad_networks
      })


if __name__ == '__main__':
  parser = ArgumentParser(
      description='Generate hashed_ad_networks.[h,cc] source files')
  parser.add_argument(
      'input_file',
      help='The name of the input file with the hosts to be hashed')
  parser.add_argument(
      '-o', '--out',
      help='The root name of the output source file',
      default='hashed_ad_networks')

  args = parser.parse_args()
  Generate(args.input_file, args.out)
