// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "google_apis/gcm/engine/unregistration_request.h"

#include "base/bind.h"
#include "base/message_loop/message_loop.h"
#include "base/strings/string_number_conversions.h"
#include "base/strings/string_piece.h"
#include "base/values.h"
#include "google_apis/gcm/base/gcm_util.h"
#include "google_apis/gcm/monitoring/gcm_stats_recorder.h"
#include "net/base/escape.h"
#include "net/http/http_request_headers.h"
#include "net/http/http_status_code.h"
#include "net/url_request/url_fetcher.h"
#include "net/url_request/url_request_context_getter.h"
#include "net/url_request/url_request_status.h"

namespace gcm {

namespace {

const char kRequestContentType[] = "application/x-www-form-urlencoded";

// Request constants.
const char kAppIdKey[] = "app";
const char kDeleteKey[] = "delete";
const char kDeleteValue[] = "true";
const char kDeviceIdKey[] = "device";
const char kLoginHeader[] = "AidLogin";

}  // namespace

UnregistrationRequest::RequestInfo::RequestInfo(
    uint64 android_id,
    uint64 security_token,
    const std::string& app_id)
    : android_id(android_id),
      security_token(security_token),
      app_id(app_id) {
  DCHECK(android_id != 0UL);
  DCHECK(security_token != 0UL);
}

UnregistrationRequest::RequestInfo::~RequestInfo() {}

UnregistrationRequest::CustomRequestHandler::CustomRequestHandler() {}

UnregistrationRequest::CustomRequestHandler::~CustomRequestHandler() {}

UnregistrationRequest::UnregistrationRequest(
    const GURL& registration_url,
    const RequestInfo& request_info,
    scoped_ptr<CustomRequestHandler> custom_request_handler,
    const net::BackoffEntry::Policy& backoff_policy,
    const UnregistrationCallback& callback,
    scoped_refptr<net::URLRequestContextGetter> request_context_getter,
    GCMStatsRecorder* recorder)
    : callback_(callback),
      request_info_(request_info),
      custom_request_handler_(custom_request_handler.Pass()),
      registration_url_(registration_url),
      backoff_entry_(&backoff_policy),
      request_context_getter_(request_context_getter),
      recorder_(recorder),
      weak_ptr_factory_(this) {
}

UnregistrationRequest::~UnregistrationRequest() {}

void UnregistrationRequest::Start() {
  DCHECK(!callback_.is_null());
  DCHECK(!url_fetcher_.get());

  url_fetcher_ =
      net::URLFetcher::Create(registration_url_, net::URLFetcher::POST, this);
  url_fetcher_->SetRequestContext(request_context_getter_.get());

  std::string extra_headers;
  BuildRequestHeaders(&extra_headers);
  url_fetcher_->SetExtraRequestHeaders(extra_headers);

  std::string body;
  BuildRequestBody(&body);

  DVLOG(1) << "Unregistration request: " << body;
  url_fetcher_->SetUploadData(kRequestContentType, body);

  DVLOG(1) << "Performing unregistration for: " << request_info_.app_id;
  recorder_->RecordUnregistrationSent(request_info_.app_id);
  request_start_time_ = base::TimeTicks::Now();
  url_fetcher_->Start();
}

void UnregistrationRequest::BuildRequestHeaders(std::string* extra_headers) {
  net::HttpRequestHeaders headers;
  headers.SetHeader(
      net::HttpRequestHeaders::kAuthorization,
      std::string(kLoginHeader) + " " +
          base::Uint64ToString(request_info_.android_id) + ":" +
          base::Uint64ToString(request_info_.security_token));
  headers.SetHeader(kAppIdKey, request_info_.app_id);
  *extra_headers = headers.ToString();
}

void UnregistrationRequest::BuildRequestBody(std::string* body) {
  BuildFormEncoding(kAppIdKey, request_info_.app_id, body);
  BuildFormEncoding(kDeviceIdKey,
                    base::Uint64ToString(request_info_.android_id),
                    body);
  BuildFormEncoding(kDeleteKey, kDeleteValue, body);

  DCHECK(custom_request_handler_.get());
  custom_request_handler_->BuildRequestBody(body);
}

UnregistrationRequest::Status UnregistrationRequest::ParseResponse(
    const net::URLFetcher* source) {
  if (!source->GetStatus().is_success()) {
    DVLOG(1) << "Fetcher failed";
    return URL_FETCHING_FAILED;
  }

  net::HttpStatusCode response_status = static_cast<net::HttpStatusCode>(
      source->GetResponseCode());
  if (response_status != net::HTTP_OK) {
    DVLOG(1) << "HTTP Status code is not OK, but: " << response_status;
    if (response_status == net::HTTP_SERVICE_UNAVAILABLE)
      return SERVICE_UNAVAILABLE;
    if (response_status == net::HTTP_INTERNAL_SERVER_ERROR)
      return INTERNAL_SERVER_ERROR;
    return HTTP_NOT_OK;
  }

  DCHECK(custom_request_handler_.get());
  return custom_request_handler_->ParseResponse(source);
}

void UnregistrationRequest::RetryWithBackoff(bool update_backoff) {
  if (update_backoff) {
    url_fetcher_.reset();
    backoff_entry_.InformOfRequest(false);
  }

  if (backoff_entry_.ShouldRejectRequest()) {
    DVLOG(1) << "Delaying GCM unregistration of app: "
             << request_info_.app_id << ", for "
             << backoff_entry_.GetTimeUntilRelease().InMilliseconds()
             << " milliseconds.";
    recorder_->RecordUnregistrationRetryDelayed(
        request_info_.app_id,
        backoff_entry_.GetTimeUntilRelease().InMilliseconds());
    base::MessageLoop::current()->PostDelayedTask(
        FROM_HERE,
        base::Bind(&UnregistrationRequest::RetryWithBackoff,
                   weak_ptr_factory_.GetWeakPtr(),
                   false),
        backoff_entry_.GetTimeUntilRelease());
    return;
  }

  Start();
}

void UnregistrationRequest::OnURLFetchComplete(const net::URLFetcher* source) {
  UnregistrationRequest::Status status = ParseResponse(source);

  DVLOG(1) << "UnregistrationRequestStauts: " << status;

  DCHECK(custom_request_handler_.get());
  custom_request_handler_->ReportUMAs(
      status,
      backoff_entry_.failure_count(),
      base::TimeTicks::Now() - request_start_time_);

  recorder_->RecordUnregistrationResponse(request_info_.app_id, status);

  if (status == URL_FETCHING_FAILED ||
      status == SERVICE_UNAVAILABLE ||
      status == INTERNAL_SERVER_ERROR ||
      status == INCORRECT_APP_ID ||
      status == RESPONSE_PARSING_FAILED) {
    RetryWithBackoff(true);
    return;
  }

  // status == SUCCESS || HTTP_NOT_OK || NO_RESPONSE_BODY ||
  // INVALID_PARAMETERS || UNKNOWN_ERROR

  callback_.Run(status);
}

}  // namespace gcm
