// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/offline_pages/offline_page_mhtml_archiver.h"

#include "base/bind.h"
#include "base/files/file_path.h"
#include "base/files/scoped_temp_dir.h"
#include "base/memory/ref_counted.h"
#include "base/memory/scoped_ptr.h"
#include "base/message_loop/message_loop.h"
#include "base/run_loop.h"
#include "base/strings/utf_string_conversions.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace offline_pages {

namespace {
const char kTestURL[] = "http://example.com/";
const base::string16 kTestTitle = base::ASCIIToUTF16("Test page title");
const base::FilePath::CharType kTestFilePath[] = FILE_PATH_LITERAL(
    "/archive_dir/offline_page.mhtml");
const int64 kTestFileSize = 123456LL;
}  // namespace

class TestMHTMLArchiver : public OfflinePageMHTMLArchiver {
 public:
  enum class TestScenario {
    SUCCESS,
    NOT_ABLE_TO_ARCHIVE,
    WEB_CONTENTS_MISSING,
  };

  TestMHTMLArchiver(
      const GURL& url,
      const base::string16& title,
      const TestScenario test_scenario,
      const base::FilePath& archive_dir,
      const scoped_refptr<base::SingleThreadTaskRunner>& task_runner);
  ~TestMHTMLArchiver() override;

 private:
  bool IsWebContentsValid() const override;
  void GenerateMHTML() override;

  const GURL url_;
  const base::string16 title_;
  const TestScenario test_scenario_;

  DISALLOW_COPY_AND_ASSIGN(TestMHTMLArchiver);
};

TestMHTMLArchiver::TestMHTMLArchiver(
    const GURL& url,
    const base::string16& title,
    const TestScenario test_scenario,
    const base::FilePath& archive_dir,
    const scoped_refptr<base::SingleThreadTaskRunner>& task_runner)
    : OfflinePageMHTMLArchiver(archive_dir, task_runner),
      url_(url),
      title_(title),
      test_scenario_(test_scenario) {
}

TestMHTMLArchiver::~TestMHTMLArchiver() {
}

bool TestMHTMLArchiver::IsWebContentsValid() const {
  return test_scenario_ != TestScenario::WEB_CONTENTS_MISSING;
}

void TestMHTMLArchiver::GenerateMHTML() {
  int64 file_size = kTestFileSize;
  if (test_scenario_ == TestScenario::NOT_ABLE_TO_ARCHIVE)
    file_size = -1;
  task_runner_->PostTask(FROM_HERE,
                         base::Bind(&TestMHTMLArchiver::OnGenerateMHTMLDone,
                                    base::Unretained(this),
                                    url_,
                                    title_,
                                    file_size));
}

class OfflinePageMHTMLArchiverTest : public testing::Test {
 public:
  OfflinePageMHTMLArchiverTest();
  ~OfflinePageMHTMLArchiverTest() override;

  // Creates an archiver for testing and specifies a scenario to be used.
  scoped_ptr<TestMHTMLArchiver> CreateArchiver(
      const GURL& url,
      const base::string16& title,
      TestMHTMLArchiver::TestScenario scenario);

  // Test tooling methods.
  void PumpLoop();

  base::FilePath GetTestFilePath() const {
    return base::FilePath(kTestFilePath);
  }

  const OfflinePageArchiver* last_archiver() const { return last_archiver_; }
  OfflinePageArchiver::ArchiverResult last_result() const {
    return last_result_;
  }
  const base::FilePath& last_file_path() const { return last_file_path_; }
  int64 last_file_size() const { return last_file_size_; }

  const OfflinePageArchiver::CreateArchiveCallback callback() {
    return base::Bind(&OfflinePageMHTMLArchiverTest::OnCreateArchiveDone,
                      base::Unretained(this));
  }

 private:
  void OnCreateArchiveDone(OfflinePageArchiver* archiver,
                           OfflinePageArchiver::ArchiverResult result,
                           const GURL& url,
                           const base::string16& title,
                           const base::FilePath& file_path,
                           int64 file_size);

  OfflinePageArchiver* last_archiver_;
  OfflinePageArchiver::ArchiverResult last_result_;
  GURL last_url_;
  base::string16 last_title_;
  base::FilePath last_file_path_;
  int64 last_file_size_;

  base::MessageLoop message_loop_;
  scoped_ptr<base::RunLoop> run_loop_;

  DISALLOW_COPY_AND_ASSIGN(OfflinePageMHTMLArchiverTest);
};

OfflinePageMHTMLArchiverTest::OfflinePageMHTMLArchiverTest()
    : last_archiver_(nullptr),
      last_result_(OfflinePageArchiver::ArchiverResult::
                       ERROR_ARCHIVE_CREATION_FAILED),
      last_file_size_(0L) {
}

OfflinePageMHTMLArchiverTest::~OfflinePageMHTMLArchiverTest() {
}

scoped_ptr<TestMHTMLArchiver> OfflinePageMHTMLArchiverTest::CreateArchiver(
    const GURL& url,
    const base::string16& title,
    TestMHTMLArchiver::TestScenario scenario) {
  return scoped_ptr<TestMHTMLArchiver>(new TestMHTMLArchiver(
      url, title, scenario, GetTestFilePath(), message_loop_.task_runner()));
}

void OfflinePageMHTMLArchiverTest::OnCreateArchiveDone(
    OfflinePageArchiver* archiver,
    OfflinePageArchiver::ArchiverResult result,
    const GURL& url,
    const base::string16& title,
    const base::FilePath& file_path,
    int64 file_size) {
  run_loop_->Quit();
  last_url_ = url;
  last_title_ = title;
  last_archiver_ = archiver;
  last_result_ = result;
  last_file_path_ = file_path;
  last_file_size_ = file_size;
}

void OfflinePageMHTMLArchiverTest::PumpLoop() {
  run_loop_.reset(new base::RunLoop());
  run_loop_->Run();
}

// Tests that creation of an archiver fails when web contents is missing.
TEST_F(OfflinePageMHTMLArchiverTest, WebContentsMissing) {
  GURL page_url = GURL(kTestURL);
  scoped_ptr<TestMHTMLArchiver> archiver(
      CreateArchiver(page_url,
                     kTestTitle,
                     TestMHTMLArchiver::TestScenario::WEB_CONTENTS_MISSING));
  archiver->CreateArchive(callback());
  PumpLoop();

  EXPECT_EQ(archiver.get(), last_archiver());
  EXPECT_EQ(OfflinePageArchiver::ArchiverResult::ERROR_CONTENT_UNAVAILABLE,
            last_result());
  EXPECT_EQ(base::FilePath(), last_file_path());
}

// Tests for successful creation of the offline page archive.
TEST_F(OfflinePageMHTMLArchiverTest, NotAbleToGenerateArchive) {
  GURL page_url = GURL(kTestURL);
  scoped_ptr<TestMHTMLArchiver> archiver(
      CreateArchiver(page_url,
                     kTestTitle,
                     TestMHTMLArchiver::TestScenario::NOT_ABLE_TO_ARCHIVE));
  archiver->CreateArchive(callback());
  PumpLoop();

  EXPECT_EQ(archiver.get(), last_archiver());
  EXPECT_EQ(OfflinePageArchiver::ArchiverResult::ERROR_ARCHIVE_CREATION_FAILED,
            last_result());
  EXPECT_EQ(base::FilePath(), last_file_path());
  EXPECT_EQ(0LL, last_file_size());
}

// Tests for successful creation of the offline page archive.
TEST_F(OfflinePageMHTMLArchiverTest, SuccessfullyCreateOfflineArchive) {
  GURL page_url = GURL(kTestURL);
  scoped_ptr<TestMHTMLArchiver> archiver(
      CreateArchiver(page_url,
                     kTestTitle,
                     TestMHTMLArchiver::TestScenario::SUCCESS));
  archiver->CreateArchive(callback());
  PumpLoop();

  EXPECT_EQ(archiver.get(), last_archiver());
  EXPECT_EQ(OfflinePageArchiver::ArchiverResult::SUCCESSFULLY_CREATED,
            last_result());
  EXPECT_EQ(GetTestFilePath(), last_file_path());
  EXPECT_EQ(kTestFileSize, last_file_size());
}

}  // namespace offline_pages
