// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/autofill/core/browser/autofill_country.h"

#include <stddef.h>
#include <stdint.h>
#include <map>
#include <utility>

#include "base/logging.h"
#include "base/memory/scoped_ptr.h"
#include "base/memory/singleton.h"
#include "base/stl_util.h"
#include "base/strings/string_util.h"
#include "base/strings/utf_string_conversions.h"
#include "grit/components_strings.h"
#include "third_party/icu/source/common/unicode/locid.h"
#include "third_party/icu/source/common/unicode/uloc.h"
#include "third_party/icu/source/common/unicode/unistr.h"
#include "third_party/icu/source/common/unicode/urename.h"
#include "third_party/icu/source/common/unicode/utypes.h"
#include "third_party/icu/source/i18n/unicode/coll.h"
#include "third_party/icu/source/i18n/unicode/ucol.h"
#include "ui/base/l10n/l10n_util.h"

namespace autofill {
namespace {

// The maximum capacity needed to store a locale up to the country code.
const size_t kLocaleCapacity =
    ULOC_LANG_CAPACITY + ULOC_SCRIPT_CAPACITY + ULOC_COUNTRY_CAPACITY + 1;

struct CountryData {
  int postal_code_label_id;
  int state_label_id;
  AddressRequiredFields address_required_fields;
};

struct StaticCountryData {
  char country_code[3];
  CountryData country_data;
};

// Maps country codes to localized label string identifiers.
const StaticCountryData kCountryData[] = {
  { "AC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY } },
  { "AD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PARISH,
            ADDRESS_REQUIRES_STATE } },
  { "AE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_EMIRATE,
            ADDRESS_REQUIRES_STATE } },
  { "AF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "AG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_ADDRESS_LINE_1_ONLY } },
  { "AI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "AL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "AM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "AO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "AQ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "AR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "AS", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "AT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "AU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "AW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "AX", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "AZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BB", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PARISH,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "BF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BJ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "BM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "BS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_ISLAND,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "BZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "CC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "CI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "CO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_ISLAND,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CX", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "CZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "DE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "DJ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "DK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "DM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "DO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "DZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "EC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "EE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "EG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "EH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "ER", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "ES", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "ET", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "FI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "FJ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "FK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "FM", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "FO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "FR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "GA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "GB", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_COUNTY,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "GD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "GE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "GF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "GG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "GH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "GI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_ADDRESS_LINE_1_ONLY } },
  { "GL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "GM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "GN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "GP", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "GQ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "GR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "GS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "GT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "GU", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "GW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "GY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "HK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_AREA,
            ADDRESS_REQUIRES_STATE } },
  { "HM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "HN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_STATE } },
  { "HR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "HT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "HU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "ID", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "IE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_COUNTY,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "IL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "IM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "IN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "IO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "IQ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_STATE } },
  { "IS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "IT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "JE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "JM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PARISH,
            ADDRESS_REQUIRES_CITY_STATE } },
  { "JO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "JP", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PREFECTURE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "KE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "KG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "KH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "KI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_ISLAND,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "KM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "KN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_ISLAND,
            ADDRESS_REQUIRES_CITY_STATE } },
  { "KP", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "KR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "KW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "KY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_ISLAND,
            ADDRESS_REQUIRES_STATE } },
  { "KZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "LA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "LB", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "LC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "LI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "LK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "LR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "LS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "LT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "LU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "LV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "LY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "ME", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "MG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MH", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "MK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "ML", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_ADDRESS_LINE_1_ONLY } },
  { "MP", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "MQ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "MR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "MX", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "MY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "MZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "NA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "NC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "NE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "NF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "NG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "NI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_DEPARTMENT,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "NL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "NO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "NP", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "NR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_DISTRICT,
            ADDRESS_REQUIRES_STATE } },
  { "NU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "NZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "OM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "PA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "PE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "PF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_ISLAND,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "PG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_STATE } },
  { "PH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY } },
  { "PK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "PL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "PM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "PN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "PR", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "PS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "PT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "PW", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "PY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "QA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "RE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "RO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "RS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "RU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "RW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "SA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "SB", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "SC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_ISLAND,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "SE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "SG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_ZIP } },
  { "SH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "SI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "SJ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "SK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "SL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "SM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_ZIP } },
  { "SN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "SO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_STATE } },
  { "SR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "ST", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "SV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_STATE } },
  { "SZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY } },
  { "TC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "TD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TJ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "TT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_ISLAND,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "TW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_COUNTY,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "TZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "UA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "UG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "UM", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE } },
  { "US", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "UY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "UZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "VA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "VC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "VE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_STATE } },
  { "VG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_ADDRESS_LINE_1_ONLY } },
  { "VI", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
            IDS_AUTOFILL_FIELD_LABEL_STATE,
            ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  { "VN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY } },
  { "VU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "WF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "WS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
  { "XK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY } },
  { "YE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY } },
  { "YT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "ZA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY_ZIP } },
  { "ZM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIRES_CITY } },
  { "ZW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
            IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
            ADDRESS_REQUIREMENTS_UNKNOWN } },
};

// A singleton class that encapsulates a map from country codes to country data.
class CountryDataMap {
 public:
  // A const iterator over the wrapped map data.
  typedef std::map<std::string, CountryData>::const_iterator Iterator;

  static CountryDataMap* GetInstance();
  static const Iterator Begin();
  static const Iterator End();
  static const Iterator Find(const std::string& country_code);

 private:
  CountryDataMap();
  friend struct DefaultSingletonTraits<CountryDataMap>;

  std::map<std::string, CountryData> country_data_;

  DISALLOW_COPY_AND_ASSIGN(CountryDataMap);
};

// static
CountryDataMap* CountryDataMap::GetInstance() {
  return Singleton<CountryDataMap>::get();
}

CountryDataMap::CountryDataMap() {
  // Add all the countries we have explicit data for.
  for (size_t i = 0; i < arraysize(kCountryData); ++i) {
    const StaticCountryData& static_data = kCountryData[i];
    country_data_.insert(std::make_pair(static_data.country_code,
                                        static_data.country_data));
  }

  // Add any other countries that ICU knows about, falling back to default data
  // values.
  for (const char* const* country_pointer = icu::Locale::getISOCountries();
       *country_pointer;
       ++country_pointer) {
    std::string country_code = *country_pointer;
    if (!country_data_.count(country_code)) {
      CountryData data = {
        IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
        IDS_AUTOFILL_FIELD_LABEL_PROVINCE
      };
      country_data_.insert(std::make_pair(country_code, data));
    }
  }
}

const CountryDataMap::Iterator CountryDataMap::Begin() {
  return GetInstance()->country_data_.begin();
}

const CountryDataMap::Iterator CountryDataMap::End() {
  return GetInstance()->country_data_.end();
}

const CountryDataMap::Iterator CountryDataMap::Find(
    const std::string& country_code) {
  return GetInstance()->country_data_.find(country_code);
}

// A singleton class that encapsulates mappings from country names to their
// corresponding country codes.
class CountryNames {
 public:
  static CountryNames* GetInstance();

  // Returns the country code corresponding to |country|, which should be a
  // country code or country name localized to |locale|.
  const std::string GetCountryCode(const base::string16& country,
                                   const std::string& locale);

 private:
  CountryNames();
  ~CountryNames();
  friend struct DefaultSingletonTraits<CountryNames>;

  // Populates |locales_to_localized_names_| with the mapping of country names
  // localized to |locale| to their corresponding country codes.
  void AddLocalizedNamesForLocale(const std::string& locale);

  // Interprets |country_name| as a full country name localized to the given
  // |locale| and returns the corresponding country code stored in
  // |locales_to_localized_names_|, or an empty string if there is none.
  const std::string GetCountryCodeForLocalizedName(
      const base::string16& country_name,
      const std::string& locale);

  // Returns an ICU collator -- i.e. string comparator -- appropriate for the
  // given |locale|.
  icu::Collator* GetCollatorForLocale(const std::string& locale);

  // Returns the ICU sort key corresponding to |str| for the given |collator|.
  // Uses |buffer| as temporary storage, and might resize |buffer| as a side-
  // effect. |buffer_size| should specify the |buffer|'s size, and is updated if
  // the |buffer| is resized.
  const std::string GetSortKey(const icu::Collator& collator,
                               const base::string16& str,
                               scoped_ptr<uint8_t[]>* buffer,
                               int32_t* buffer_size) const;

  // Maps from common country names, including 2- and 3-letter country codes,
  // to the corresponding 2-letter country codes. The keys are uppercase ASCII
  // strings.
  std::map<std::string, std::string> common_names_;

  // The outer map keys are ICU locale identifiers.
  // The inner maps map from localized country names to their corresponding
  // country codes. The inner map keys are ICU collation sort keys corresponding
  // to the target localized country name.
  std::map<std::string, std::map<std::string, std::string> >
      locales_to_localized_names_;

  // Maps ICU locale names to their corresponding collators.
  std::map<std::string, icu::Collator*> collators_;

  DISALLOW_COPY_AND_ASSIGN(CountryNames);
};

// static
CountryNames* CountryNames::GetInstance() {
  return Singleton<CountryNames>::get();
}

CountryNames::CountryNames() {
  // Add 2- and 3-letter ISO country codes.
  for (CountryDataMap::Iterator it = CountryDataMap::Begin();
       it != CountryDataMap::End();
       ++it) {
    const std::string& country_code = it->first;
    common_names_.insert(std::make_pair(country_code, country_code));

    std::string iso3_country_code =
        icu::Locale(NULL, country_code.c_str()).getISO3Country();

    // ICU list of countries can be out-of-date with CLDR.
    if (!iso3_country_code.empty())
      common_names_.insert(std::make_pair(iso3_country_code, country_code));
  }

  // Add a few other common synonyms.
  common_names_.insert(std::make_pair("UNITED STATES OF AMERICA", "US"));
  common_names_.insert(std::make_pair("U.S.A.", "US"));
  common_names_.insert(std::make_pair("GREAT BRITAIN", "GB"));
  common_names_.insert(std::make_pair("UK", "GB"));
  common_names_.insert(std::make_pair("BRASIL", "BR"));
  common_names_.insert(std::make_pair("DEUTSCHLAND", "DE"));
}

CountryNames::~CountryNames() {
  STLDeleteContainerPairSecondPointers(collators_.begin(),
                                       collators_.end());
}

const std::string CountryNames::GetCountryCode(const base::string16& country,
                                               const std::string& locale) {
  // First, check common country names, including 2- and 3-letter country codes.
  std::string country_utf8 = base::UTF16ToUTF8(
      base::StringToUpperASCII(country));
  std::map<std::string, std::string>::const_iterator result =
      common_names_.find(country_utf8);
  if (result != common_names_.end())
    return result->second;

  // Next, check country names localized to |locale|.
  std::string country_code = GetCountryCodeForLocalizedName(country, locale);
  if (!country_code.empty())
    return country_code;

  // Finally, check country names localized to US English.
  return GetCountryCodeForLocalizedName(country, "en_US");
}

void CountryNames::AddLocalizedNamesForLocale(const std::string& locale) {
  // Nothing to do if we've previously added the localized names for the given
  // |locale|.
  if (locales_to_localized_names_.count(locale))
    return;

  std::map<std::string, std::string> localized_names;
  const icu::Collator* collator = GetCollatorForLocale(locale);
  int32_t buffer_size = 1000;
  scoped_ptr<uint8_t[]> buffer(new uint8_t[buffer_size]);

  for (CountryDataMap::Iterator it = CountryDataMap::Begin();
       it != CountryDataMap::End();
       ++it) {
    const std::string& country_code = it->first;
    base::string16 country_name = l10n_util::GetDisplayNameForCountry(
        country_code, locale);
    std::string sort_key = GetSortKey(*collator,
                                      country_name,
                                      &buffer,
                                      &buffer_size);

    localized_names.insert(std::make_pair(sort_key, country_code));
  }

  locales_to_localized_names_.insert(std::make_pair(locale, localized_names));
}

const std::string CountryNames::GetCountryCodeForLocalizedName(
    const base::string16& country_name,
    const std::string& locale) {
  AddLocalizedNamesForLocale(locale);

  icu::Collator* collator = GetCollatorForLocale(locale);

  // As recommended[1] by ICU, initialize the buffer size to four times the
  // source string length.
  // [1] http://userguide.icu-project.org/collation/api#TOC-Examples
  int32_t buffer_size = country_name.size() * 4;
  scoped_ptr<uint8_t[]> buffer(new uint8_t[buffer_size]);
  std::string sort_key = GetSortKey(*collator,
                                    country_name,
                                    &buffer,
                                    &buffer_size);

  const std::map<std::string, std::string>& localized_names =
      locales_to_localized_names_[locale];
  std::map<std::string, std::string>::const_iterator result =
      localized_names.find(sort_key);

  if (result != localized_names.end())
    return result->second;

  return std::string();
}

icu::Collator* CountryNames::GetCollatorForLocale(const std::string& locale) {
  if (!collators_.count(locale)) {
    icu::Locale icu_locale(locale.c_str());
    UErrorCode ignored = U_ZERO_ERROR;
    icu::Collator* collator(icu::Collator::createInstance(icu_locale, ignored));

    // Compare case-insensitively and ignoring punctuation.
    ignored = U_ZERO_ERROR;
    collator->setAttribute(UCOL_STRENGTH, UCOL_SECONDARY, ignored);
    ignored = U_ZERO_ERROR;
    collator->setAttribute(UCOL_ALTERNATE_HANDLING, UCOL_SHIFTED, ignored);

    collators_.insert(std::make_pair(locale, collator));
  }

  return collators_[locale];
}

const std::string CountryNames::GetSortKey(const icu::Collator& collator,
                                           const base::string16& str,
                                           scoped_ptr<uint8_t[]>* buffer,
                                           int32_t* buffer_size) const {
  DCHECK(buffer);
  DCHECK(buffer_size);

  icu::UnicodeString icu_str(str.c_str(), str.length());
  int32_t expected_size = collator.getSortKey(icu_str, buffer->get(),
                                              *buffer_size);
  if (expected_size > *buffer_size) {
    // If there wasn't enough space, grow the buffer and try again.
    *buffer_size = expected_size;
    buffer->reset(new uint8_t[*buffer_size]);
    DCHECK(buffer->get());

    expected_size = collator.getSortKey(icu_str, buffer->get(), *buffer_size);
    DCHECK_EQ(*buffer_size, expected_size);
  }

  return std::string(reinterpret_cast<const char*>(buffer->get()));
}

}  // namespace

AutofillCountry::AutofillCountry(const std::string& country_code,
                                 const std::string& locale) {
  const CountryDataMap::Iterator result = CountryDataMap::Find(country_code);
  DCHECK(result != CountryDataMap::End());
  const CountryData& data = result->second;

  country_code_ = country_code;
  name_ = l10n_util::GetDisplayNameForCountry(country_code, locale);
  postal_code_label_ = l10n_util::GetStringUTF16(data.postal_code_label_id);
  state_label_ = l10n_util::GetStringUTF16(data.state_label_id);
  address_required_fields_ = data.address_required_fields;
}

AutofillCountry::~AutofillCountry() {
}

// static
void AutofillCountry::GetAvailableCountries(
    std::vector<std::string>* country_codes) {
  DCHECK(country_codes);

  for (CountryDataMap::Iterator it = CountryDataMap::Begin();
       it != CountryDataMap::End();
       ++it) {
    country_codes->push_back(it->first);
  }
}

// static
const std::string AutofillCountry::CountryCodeForLocale(
    const std::string& locale) {
  // Add likely subtags to the locale. In particular, add any likely country
  // subtags -- e.g. for locales like "ru" that only include the language.
  std::string likely_locale;
  UErrorCode error_ignored = U_ZERO_ERROR;
  uloc_addLikelySubtags(locale.c_str(),
                        base::WriteInto(&likely_locale, kLocaleCapacity),
                        kLocaleCapacity,
                        &error_ignored);

  // Extract the country code.
  std::string country_code = icu::Locale(likely_locale.c_str()).getCountry();

  // Default to the United States if we have no better guess.
  if (CountryDataMap::Find(country_code) == CountryDataMap::End())
    return "US";

  return country_code;
}

// static
const std::string AutofillCountry::GetCountryCode(const base::string16& country,
                                                  const std::string& locale) {
  return CountryNames::GetInstance()->GetCountryCode(country, locale);
}

AutofillCountry::AutofillCountry(const std::string& country_code,
                                 const base::string16& name,
                                 const base::string16& postal_code_label,
                                 const base::string16& state_label)
    : country_code_(country_code),
      name_(name),
      postal_code_label_(postal_code_label),
      state_label_(state_label) {
}

}  // namespace autofill
