#!/usr/bin/python
# Copyright 2014 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Unittests for generate_resources_map.py"""

import unittest

import generate_resources_map


class GenerateResourcesMapUnittest(unittest.TestCase):
  TEST_INPUT = """
// This file is automatically generated by GRIT. Do not edit.

#pragma once

#define IDS_BOOKMARKS_NO_ITEMS 12500
#define IDS_BOOKMARK_BAR_IMPORT_LINK 12501
#define IDS_BOOKMARK_GROUP_FROM_IE 12502
#define IDS_BOOKMARK_GROUP_FROM_FIREFOX 12503
"""

  def testGetResourceListFromString(self):
    expected_tuples = [(301430091, "IDS_BOOKMARKS_NO_ITEMS", "12500"),
                       (2654138887, "IDS_BOOKMARK_BAR_IMPORT_LINK", "12501"),
                       (2894469061, "IDS_BOOKMARK_GROUP_FROM_IE", "12502"),
                       (3847176170, "IDS_BOOKMARK_GROUP_FROM_FIREFOX", "12503")]
    expected = [generate_resources_map.Resource(*t) for t in expected_tuples]

    actual_tuples = generate_resources_map._GetResourceListFromString(
        self.TEST_INPUT)

    self.assertEqual(expected_tuples, actual_tuples)


  def testCheckForHashCollisions(self):
    collisions_tuples = [(123, "IDS_FOO", "12500"),
                         (456, "IDS_BAR", "12501"),
                         (456, "IDS_BAZ", "12502"),
                         (890, "IDS_QUX", "12503"),
                         (899, "IDS_NO", "12504"),
                         (899, "IDS_YES", "12505")]
    list_with_collisions = [generate_resources_map.Resource(*t)
                            for t in collisions_tuples]

    expected_collision_tuples = [(456, "IDS_BAR", "12501"),
                                 (456, "IDS_BAZ", "12502"),
                                 (899, "IDS_NO", "12504"),
                                 (899, "IDS_YES", "12505")]
    expected_collisions = [generate_resources_map.Resource(*t)
                          for t in expected_collision_tuples]

    actual_collisions = sorted(
        generate_resources_map._CheckForHashCollisions(list_with_collisions))
    actual_collisions

    self.assertEqual(expected_collisions, actual_collisions)

  def testGenerateFileContent(self):
    expected = (
        """// This file was generated by generate_resources_map.py. Do not edit.


#include "chrome/browser/metrics/variations/generated_resources_map.h"

namespace chrome_variations {

const size_t kNumResources = 4;

const uint32_t kResourceHashes[] = {
    301430091U,  // IDS_BOOKMARKS_NO_ITEMS
    2654138887U,  // IDS_BOOKMARK_BAR_IMPORT_LINK
    2894469061U,  // IDS_BOOKMARK_GROUP_FROM_IE
    3847176170U,  // IDS_BOOKMARK_GROUP_FROM_FIREFOX
};

const int kResourceIndices[] = {
    12500,  // IDS_BOOKMARKS_NO_ITEMS
    12501,  // IDS_BOOKMARK_BAR_IMPORT_LINK
    12502,  // IDS_BOOKMARK_GROUP_FROM_IE
    12503,  // IDS_BOOKMARK_GROUP_FROM_FIREFOX
};

}  // namespace chrome_variations
""")
    actual = generate_resources_map._GenerateFileContent(self.TEST_INPUT)

    self.assertEqual(expected, actual)

if __name__ == '__main__':
  unittest.main()
