#!/usr/bin/env python
#
# Copyright 2013 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Symbolizes stack traces generated by Chromium for Android.

Sample usage:
  adb logcat chromium:V | symbolize.py
"""

import os
import re
import sys

from pylib import constants

# Uses symbol.py from third_party/android_platform, not python's.
sys.path.insert(0,
                os.path.join(constants.DIR_SOURCE_ROOT,
                            'third_party/android_platform/development/scripts'))
import symbol

# Sample output from base/debug/stack_trace_android.cc
#00 0x693cd34f /path/to/some/libfoo.so+0x0007434f
TRACE_LINE = re.compile(r'(?P<frame>\#[0-9]+ 0x[0-9a-f]{8,8}) '
                        r'(?P<lib>[^+]+)\+0x(?P<addr>[0-9a-f]{8,8})')

class Symbolizer(object):
  def __init__(self, output):
    self._output = output

  def write(self, data):
    while True:
      match = re.search(TRACE_LINE, data)
      if not match:
        self._output.write(data)
        break

      frame = match.group('frame')
      lib = match.group('lib')
      addr = match.group('addr')

      # TODO(scherkus): Doing a single lookup per line is pretty slow,
      # especially with larger libraries. Consider caching strategies such as:
      # 1) Have Python load the libraries and do symbol lookups instead of
      #    calling out to addr2line each time.
      # 2) Have Python keep multiple addr2line instances open as subprocesses,
      #    piping addresses and reading back symbols as we find them
      # 3) Read ahead the entire stack trace until we find no more, then batch
      #    the symbol lookups.
      #
      # TODO(scherkus): These results are memoized, which could result in
      # incorrect lookups when running this script on long-lived instances
      # (e.g., adb logcat) when doing incremental development. Consider clearing
      # the cache when modification timestamp of libraries change.
      sym = symbol.SymbolInformation(lib, addr, False)[0][0]

      if not sym:
        post = match.end('addr')
        self._output.write(data[:post])
        data = data[post:]
        continue

      pre = match.start('frame')
      post = match.end('addr')

      self._output.write(data[:pre])
      self._output.write(frame)
      self._output.write(' ')
      self._output.write(sym)

      data = data[post:]

  def flush(self):
    self._output.flush()


def main():
  symbolizer = Symbolizer(sys.stdout)
  for line in sys.stdin:
    symbolizer.write(line)
  symbolizer.flush()


if __name__ == '__main__':
  main()
