// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_OFFLINE_PAGES_OFFLINE_PAGE_MODEL_H_
#define COMPONENTS_OFFLINE_PAGES_OFFLINE_PAGE_MODEL_H_

#include <map>
#include <vector>

#include "base/callback.h"
#include "base/macros.h"
#include "base/memory/ref_counted.h"
#include "base/memory/scoped_ptr.h"
#include "base/memory/scoped_vector.h"
#include "base/memory/weak_ptr.h"
#include "base/observer_list.h"
#include "base/scoped_observer.h"
#include "components/bookmarks/browser/base_bookmark_model_observer.h"
#include "components/keyed_service/core/keyed_service.h"
#include "components/offline_pages/offline_page_archiver.h"

class GURL;
namespace base {
class SequencedTaskRunner;
}
namespace bookmarks {
class BookmarkModel;
}

namespace offline_pages {

struct OfflinePageItem;
class OfflinePageMetadataStore;

// Service for saving pages offline, storing the offline copy and metadata, and
// retrieving them upon request.
//
// Example usage:
//   class ArchiverImpl : public OfflinePageArchiver {
//     // This is a class that knows how to create archiver
//     void CreateArchiver(...) override;
//     ...
//   }
//
//   // In code using the OfflinePagesModel to save a page:
//   scoped_ptr<ArchiverImpl> archiver(new ArchiverImpl());
//   // Callback is of type SavePageCallback.
//   model->SavePage(url, archiver.Pass(), callback);
//
// TODO(fgorski): Things to describe:
// * how to cancel requests and what to expect
class OfflinePageModel : public KeyedService,
                         public bookmarks::BaseBookmarkModelObserver {
 public:
  // Result of saving a page offline.
  // A Java counterpart will be generated for this enum.
  // GENERATED_JAVA_ENUM_PACKAGE: org.chromium.components.offlinepages
  enum class SavePageResult {
    SUCCESS,
    CANCELLED,
    DEVICE_FULL,
    CONTENT_UNAVAILABLE,
    ARCHIVE_CREATION_FAILED,
    STORE_FAILURE,
    ALREADY_EXISTS,
    // NOTE: always keep this entry at the end. Add new result types only
    // immediately above this line. Make sure to update the corresponding
    // histogram enum accordingly.
    RESULT_COUNT,
  };

  // Result of deleting an offline page.
  // A Java counterpart will be generated for this enum.
  // GENERATED_JAVA_ENUM_PACKAGE: org.chromium.components.offlinepages
  enum class DeletePageResult {
    SUCCESS,
    CANCELLED,
    STORE_FAILURE,
    DEVICE_FAILURE,
    NOT_FOUND,
    // NOTE: always keep this entry at the end. Add new result types only
    // immediately above this line. Make sure to update the corresponding
    // histogram enum accordingly.
    RESULT_COUNT,
  };

  // Result of loading all pages.
  enum class LoadResult {
    SUCCESS,
    CANCELLED,
    STORE_FAILURE,
  };

  // Observer of the OfflinePageModel.
  class Observer {
   public:
    // Invoked when the model has finished loading.
    virtual void OfflinePageModelLoaded(OfflinePageModel* model) = 0;

   protected:
    virtual ~Observer() {}
  };

  typedef base::Callback<void(SavePageResult)> SavePageCallback;
  typedef base::Callback<void(DeletePageResult)> DeletePageCallback;

  // All blocking calls/disk access will happen on the provided |task_runner|.
  OfflinePageModel(
      scoped_ptr<OfflinePageMetadataStore> store,
      const scoped_refptr<base::SequencedTaskRunner>& task_runner);
  ~OfflinePageModel() override;

  // Starts the OfflinePageModel and registers it as a BookmarkModelObserver.
  // Calling this method is optional, but offline pages will not be deleted
  // when the bookmark is deleted, i.e. due to sync, until this method is
  // called.
  void Start(bookmarks::BookmarkModel* model);

  // KeyedService implementation.
  void Shutdown() override;

  void AddObserver(Observer* observer);
  void RemoveObserver(Observer* observer);

  // Attempts to save a page addressed by |url| offline. Requires that the model
  // is loaded.
  void SavePage(const GURL& url,
                int64 bookmark_id,
                scoped_ptr<OfflinePageArchiver> archiver,
                const SavePageCallback& callback);

  // Deletes an offline page related to the passed |bookmark_id|. Requires that
  // the model is loaded.
  void DeletePageByBookmarkId(int64 bookmark_id,
                              const DeletePageCallback& callback);

  // Deletes offline pages related to the passed |bookmark_ids|. Requires that
  // the model is loaded.
  void DeletePagesByBookmarkId(const std::vector<int64>& bookmark_ids,
                               const DeletePageCallback& callback);

  // Gets all available offline pages. Requires that the model is loaded.
  const std::vector<OfflinePageItem> GetAllPages() const;

  // Gets pages that should be removed to clean up storage. Requires that the
  // model is loaded.
  const std::vector<OfflinePageItem> GetPagesToCleanUp() const;

  // Returns an offline page associated with a specified |bookmark_id|. nullptr
  // is returned if not found.
  const OfflinePageItem* GetPageByBookmarkId(int64 bookmark_id) const;

  // Returns an offline page that is stored as |offline_url|. nullptr is
  // returned if not found.
  const OfflinePageItem* GetPageByOfflineURL(const GURL& offline_url) const;

  // Methods for testing only:
  OfflinePageMetadataStore* GetStoreForTesting();

  bool is_loaded() const { return is_loaded_; }

 private:
  typedef ScopedVector<OfflinePageArchiver> PendingArchivers;

  // BaseBookmarkModelObserver:
  void BookmarkModelChanged() override;
  void BookmarkNodeRemoved(bookmarks::BookmarkModel* model,
                           const bookmarks::BookmarkNode* parent,
                           int old_index,
                           const bookmarks::BookmarkNode* node,
                           const std::set<GURL>& removed_urls) override;

  // Callback for loading pages from the offline page metadata store.
  void OnLoadDone(bool success,
                  const std::vector<OfflinePageItem>& offline_pages);

  // Steps for saving a page offline.
  void OnCreateArchiveDone(const GURL& requested_url,
                           int64 bookmark_id,
                           const SavePageCallback& callback,
                           OfflinePageArchiver* archiver,
                           OfflinePageArchiver::ArchiverResult result,
                           const GURL& url,
                           const base::FilePath& file_path,
                           int64 file_size);
  void OnAddOfflinePageDone(OfflinePageArchiver* archiver,
                            const SavePageCallback& callback,
                            const OfflinePageItem& offline_page,
                            bool success);
  void InformSavePageDone(const SavePageCallback& callback,
                          SavePageResult result);
  void DeletePendingArchiver(OfflinePageArchiver* archiver);

  // Steps for deleting files and data for an offline page.
  void OnDeleteArchiveFilesDone(
      const std::vector<int64>& bookmark_ids,
      const DeletePageCallback& callback,
      const bool* success);
  void OnRemoveOfflinePagesDone(const std::vector<int64>& bookmark_ids,
                                const DeletePageCallback& callback,
                                bool success);
  void InformDeletePageDone(const DeletePageCallback& callback,
                            DeletePageResult result);

  // Persistent store for offline page metadata.
  scoped_ptr<OfflinePageMetadataStore> store_;

  // The observers.
  base::ObserverList<Observer> observers_;

  bool is_loaded_;

  // In memory copy of the offline page metadata, keyed by bookmark IDs.
  std::map<int64, OfflinePageItem> offline_pages_;

  scoped_refptr<base::SequencedTaskRunner> task_runner_;

  // Pending archivers owned by this model.
  PendingArchivers pending_archivers_;

  // Delayed tasks that should be invoked after the loading is done.
  std::vector<base::Closure> delayed_tasks_;

  ScopedObserver<bookmarks::BookmarkModel, bookmarks::BookmarkModelObserver>
      scoped_observer_;

  base::WeakPtrFactory<OfflinePageModel> weak_ptr_factory_;

  DISALLOW_COPY_AND_ASSIGN(OfflinePageModel);
};

}  // namespace offline_pages

#endif  // COMPONENTS_OFFLINE_PAGES_OFFLINE_PAGE_MODEL_H_
