// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_BROWSER_LOADER_CERTIFICATE_RESOURCE_HANDLER_H_
#define CONTENT_BROWSER_LOADER_CERTIFICATE_RESOURCE_HANDLER_H_

#include <string>

#include "base/macros.h"
#include "base/memory/ref_counted.h"
#include "content/browser/loader/resource_handler.h"
#include "net/base/mime_util.h"

namespace net {
class GrowableIOBuffer;
class URLRequest;
class URLRequestStatus;
}  // namespace net

namespace content {

// This class handles certificate mime types such as:
// - "application/x-x509-user-cert"
// - "application/x-x509-ca-cert"
// - "application/x-pkcs12"
//
class CertificateResourceHandler : public ResourceHandler {
 public:
  explicit CertificateResourceHandler(net::URLRequest* request);
  ~CertificateResourceHandler() override;

  // Not needed, as this event handler ought to be the final resource.
  bool OnRequestRedirected(const net::RedirectInfo& redirect_info,
                           ResourceResponse* resp,
                           bool* defer) override;

  // Check if this indeed an X509 cert.
  bool OnResponseStarted(ResourceResponse* resp, bool* defer) override;

  // Pass-through implementation.
  bool OnWillStart(const GURL& url, bool* defer) override;

  // Pass-through implementation.
  bool OnBeforeNetworkStart(const GURL& url, bool* defer) override;

  // Create a new buffer to store received data.
  bool OnWillRead(scoped_refptr<net::IOBuffer>* buf,
                  int* buf_size,
                  int min_size) override;

  // A read was completed, maybe allocate a new buffer for further data.
  bool OnReadCompleted(int bytes_read, bool* defer) override;

  // Done downloading the certificate.
  void OnResponseCompleted(const net::URLRequestStatus& urs,
                           const std::string& sec_info,
                           bool* defer) override;

  // N/A to cert downloading.
  void OnDataDownloaded(int bytes_downloaded) override;

 private:
  scoped_refptr<net::GrowableIOBuffer> buffer_;
  net::CertificateMimeType cert_type_;

  DISALLOW_COPY_AND_ASSIGN(CertificateResourceHandler);
};

}  // namespace content

#endif  // CONTENT_BROWSER_LOADER_CERTIFICATE_RESOURCE_HANDLER_H_
