// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_PUBLIC_BROWSER_ANDROID_SYNCHRONOUS_COMPOSITOR_H_
#define CONTENT_PUBLIC_BROWSER_ANDROID_SYNCHRONOUS_COMPOSITOR_H_

#include "base/memory/ref_counted.h"
#include "content/common/content_export.h"
#include "gpu/command_buffer/service/in_process_command_buffer.h"
#include "ui/gfx/geometry/rect.h"
#include "ui/gfx/geometry/size.h"

class SkCanvas;

namespace cc {
class CompositorFrame;
class CompositorFrameAck;
}

namespace gfx {
class ScrollOffset;
class Transform;
};

namespace gpu {
class GLInProcessContext;
}

namespace content {

class SynchronousCompositorClient;
class WebContents;

// Interface for embedders that wish to direct compositing operations
// synchronously under their own control. Only meaningful when the
// kEnableSyncrhonousRendererCompositor flag is specified.
class CONTENT_EXPORT SynchronousCompositor {
 public:
  // Must be called once per WebContents instance. Will create the compositor
  // instance as needed, but only if |client| is non-nullptr.
  static void SetClientForWebContents(WebContents* contents,
                                      SynchronousCompositorClient* client);

  static void SetGpuService(
      scoped_refptr<gpu::InProcessCommandBuffer::Service> service);

  // Turn on using ipc-based command buffer at run time. This should be removed
  // once this feature is fully launched.
  static void SetUseIpcCommandBuffer();

  // "On demand" hardware draw. The content is first clipped to |damage_area|,
  // then transformed through |transform|, and finally clipped to |view_size|.
  virtual scoped_ptr<cc::CompositorFrame> DemandDrawHw(
      gfx::Size surface_size,
      const gfx::Transform& transform,
      gfx::Rect viewport,
      gfx::Rect clip,
      gfx::Rect viewport_rect_for_tile_priority,
      const gfx::Transform& transform_for_tile_priority) = 0;

  // For delegated rendering, return resources from parent compositor to this.
  // Note that all resources must be returned before ReleaseHwDraw.
  virtual void ReturnResources(const cc::CompositorFrameAck& frame_ack) = 0;

  // "On demand" SW draw, into the supplied canvas (observing the transform
  // and clip set there-in).
  virtual bool DemandDrawSw(SkCanvas* canvas) = 0;

  // Set the memory limit policy of this compositor.
  virtual void SetMemoryPolicy(size_t bytes_limit) = 0;

  // Should be called by the embedder after the embedder had modified the
  // scroll offset of the root layer.
  virtual void DidChangeRootLayerScrollOffset(
      const gfx::ScrollOffset& root_offset) = 0;

  // Called by the embedder to notify that the compositor is active. The
  // compositor won't ask for vsyncs when it's inactive. NOTE: The compositor
  // starts off as inactive and needs a SetActive(true) call to begin.
  virtual void SetIsActive(bool is_active) = 0;

  // Called by the embedder to notify that the OnComputeScroll step is happening
  // and if any input animation is active, it should tick now.
  virtual void OnComputeScroll(base::TimeTicks animation_time) = 0;

 protected:
  virtual ~SynchronousCompositor() {}
};

}  // namespace content

#endif  // CONTENT_PUBLIC_BROWSER_ANDROID_SYNCHRONOUS_COMPOSITOR_H_
