// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_SCHEDULER_BASE_POLLABLE_THREAD_SAFE_FLAG_H_
#define COMPONENTS_SCHEDULER_BASE_POLLABLE_THREAD_SAFE_FLAG_H_

#include "base/atomicops.h"
#include "base/synchronization/lock.h"

// A PollableThreadSafeFlag can be polled without requiring a lock, but can only
// be updated if a lock is held. This enables lock-free checking as to whether a
// condition has changed, while protecting operations which update the condition
// with a lock. You must ensure that the flag is only updated within the same
// lock-protected critical section as any other variables on which the condition
// depends.
class PollableThreadSafeFlag {
 public:
  explicit PollableThreadSafeFlag(base::Lock* write_lock);
  ~PollableThreadSafeFlag();

  // Set the flag. May only be called if |write_lock| is held.
  void SetWhileLocked(bool value);

  // Returns true iff the flag is set to true.
  bool IsSet() const;

 private:
  base::subtle::Atomic32 flag_;
  base::Lock* write_lock_;  // Not owned.

  DISALLOW_COPY_AND_ASSIGN(PollableThreadSafeFlag);
};

#endif  // COMPONENTS_SCHEDULER_BASE_POLLABLE_THREAD_SAFE_FLAG_H_
