// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/password_manager/core/browser/password_manager_util.h"

#include "base/strings/utf_string_conversions.h"
#include "components/password_manager/core/browser/password_manager_test_utils.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace {

const char kTestAndroidRealm[] = "android://hash@com.example.beta.android";
const char kTestFederationURL[] = "https://google.com/";
const char kTestUsername[] = "Username";
const char kTestUsername2[] = "Username2";
const char kTestPassword[] = "12345";

autofill::PasswordForm GetTestAndroidCredentials(const char* signon_realm) {
  autofill::PasswordForm form;
  form.scheme = autofill::PasswordForm::SCHEME_HTML;
  form.signon_realm = signon_realm;
  form.username_value = base::ASCIIToUTF16(kTestUsername);
  form.password_value = base::ASCIIToUTF16(kTestPassword);
  form.ssl_valid = true;
  return form;
}

}  // namespace

using password_manager::UnorderedPasswordFormElementsAre;

TEST(PasswordManagerUtil, TrimUsernameOnlyCredentials) {
  ScopedVector<autofill::PasswordForm> forms, expected_forms;
  forms.push_back(
      new autofill::PasswordForm(GetTestAndroidCredentials(kTestAndroidRealm)));
  expected_forms.push_back(
      new autofill::PasswordForm(GetTestAndroidCredentials(kTestAndroidRealm)));

  autofill::PasswordForm username_only;
  username_only.scheme = autofill::PasswordForm::SCHEME_USERNAME_ONLY;
  username_only.signon_realm = kTestAndroidRealm;
  username_only.username_value = base::ASCIIToUTF16(kTestUsername2);
  forms.push_back(new autofill::PasswordForm(username_only));

  username_only.federation_url = GURL(kTestFederationURL);
  username_only.skip_zero_click = false;
  forms.push_back(new autofill::PasswordForm(username_only));
  username_only.skip_zero_click = true;
  expected_forms.push_back(new autofill::PasswordForm(username_only));

  password_manager_util::TrimUsernameOnlyCredentials(&forms);

  EXPECT_THAT(forms.get(),
              UnorderedPasswordFormElementsAre(expected_forms.get()));
}
