// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.firstrun;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.preference.PreferenceManager;

import org.chromium.sync.signin.AccountManagerHelper;

import java.util.HashSet;
import java.util.List;

/**
 * This receiver is notified when a user goes through the Setup Wizard and acknowledges
 * the Chrome ToS so that we don't show the ToS string during our first run.
 */
public class ToSAckedReceiver extends BroadcastReceiver {
    private static final String TOS_ACKED_ACCOUNTS = "ToS acknowledged accounts";
    private static final String EXTRA_ACCOUNT_NAME = "TosAckedReceiver.account";

    @Override
    public void onReceive(Context context, Intent intent) {
        Bundle args = intent.getExtras();
        if (args == null) return;
        String accountName = args.getString(EXTRA_ACCOUNT_NAME, null);
        if (accountName == null) return;

        SharedPreferences prefs = PreferenceManager.getDefaultSharedPreferences(context);
        HashSet<String> accounts = (HashSet<String>) prefs.getStringSet(
                TOS_ACKED_ACCOUNTS, new HashSet<String>());
        accounts.add(accountName);
        prefs.edit().remove(TOS_ACKED_ACCOUNTS).apply();
        prefs.edit().putStringSet(TOS_ACKED_ACCOUNTS, accounts).apply();
    }

    /**
     * Checks whether any of the current google accounts has seen the ToS in setup wizard.
     * @param context Context for the app.
     * @return Whether or not the the ToS has been seen.
     */
    public static boolean checkAnyUserHasSeenToS(Context context) {
        HashSet<String> toSAckedAccounts = (HashSet<String>) PreferenceManager
                .getDefaultSharedPreferences(context).getStringSet(TOS_ACKED_ACCOUNTS, null);
        if (toSAckedAccounts == null || toSAckedAccounts.isEmpty()) return false;
        AccountManagerHelper accountHelper = AccountManagerHelper.get(context);
        List<String> accountNames = accountHelper.getGoogleAccountNames();
        if (accountNames.isEmpty()) return false;
        for (int k = 0; k < accountNames.size(); k++) {
            if (toSAckedAccounts.contains(accountNames.get(k))) return true;
        }
        return false;
    }
}