// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/sync/test_profile_sync_service.h"

#include <utility>

#include "base/location.h"
#include "base/memory/scoped_ptr.h"
#include "base/single_thread_task_runner.h"
#include "base/thread_task_runner_handle.h"
#include "chrome/browser/chrome_notification_types.h"
#include "chrome/browser/invalidation/profile_invalidation_provider_factory.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/signin/profile_oauth2_token_service_factory.h"
#include "chrome/browser/signin/signin_manager_factory.h"
#include "chrome/browser/sync/chrome_sync_client.h"
#include "chrome/browser/sync/profile_sync_service_factory.h"
#include "chrome/browser/sync/profile_sync_test_util.h"
#include "chrome/browser/sync/test/test_http_bridge_factory.h"
#include "chrome/common/channel_info.h"
#include "components/invalidation/impl/profile_invalidation_provider.h"
#include "components/signin/core/browser/signin_manager.h"
#include "components/sync_driver/glue/sync_backend_host.h"
#include "components/sync_driver/glue/sync_backend_host_core.h"
#include "components/sync_driver/signin_manager_wrapper.h"
#include "components/sync_driver/sync_api_component_factory_mock.h"
#include "content/public/browser/browser_thread.h"
#include "google_apis/gaia/gaia_constants.h"
#include "sync/internal_api/public/test/sync_manager_factory_for_profile_sync_test.h"
#include "sync/internal_api/public/test/test_internal_components_factory.h"
#include "sync/internal_api/public/user_share.h"
#include "sync/protocol/encryption.pb.h"
#include "testing/gmock/include/gmock/gmock.h"

using content::BrowserThread;
using syncer::InternalComponentsFactory;
using syncer::TestInternalComponentsFactory;
using syncer::UserShare;

namespace {

ProfileSyncService::InitParams GetInitParams(
    Profile* profile,
    SigninManagerBase* signin,
    ProfileOAuth2TokenService* oauth2_token_service,
    browser_sync::ProfileSyncServiceStartBehavior behavior) {
  ProfileSyncService::InitParams init_params =
      CreateProfileSyncServiceParamsForTest(profile);

  init_params.signin_wrapper =
      make_scoped_ptr(new SigninManagerWrapper(signin));
  init_params.oauth2_token_service = oauth2_token_service;
  init_params.start_behavior = behavior;

  return init_params;
}

}  // namespace

namespace browser_sync {

SyncBackendHostForProfileSyncTest::SyncBackendHostForProfileSyncTest(
    Profile* profile,
    sync_driver::SyncClient* sync_client,
    const scoped_refptr<base::SingleThreadTaskRunner>& ui_thread,
    invalidation::InvalidationService* invalidator,
    const base::WeakPtr<sync_driver::SyncPrefs>& sync_prefs,
    base::Closure callback)
    : browser_sync::SyncBackendHostImpl(
          profile->GetDebugName(),
          sync_client,
          ui_thread,
          invalidator,
          sync_prefs,
          profile->GetPath().Append(base::FilePath(FILE_PATH_LITERAL("test")))),
      callback_(callback) {}

SyncBackendHostForProfileSyncTest::~SyncBackendHostForProfileSyncTest() {}

void SyncBackendHostForProfileSyncTest::InitCore(
    scoped_ptr<DoInitializeOptions> options) {
  options->http_bridge_factory =
      scoped_ptr<syncer::HttpPostProviderFactory>(
          new browser_sync::TestHttpBridgeFactory());
  options->sync_manager_factory.reset(
      new syncer::SyncManagerFactoryForProfileSyncTest(callback_));
  options->credentials.email = "testuser@gmail.com";
  options->credentials.sync_token = "token";
  options->credentials.scope_set.insert(GaiaConstants::kChromeSyncOAuth2Scope);
  options->restored_key_for_bootstrapping = "";

  // It'd be nice if we avoided creating the InternalComponentsFactory in the
  // first place, but SyncBackendHost will have created one by now so we must
  // free it. Grab the switches to pass on first.
  InternalComponentsFactory::Switches factory_switches =
      options->internal_components_factory->GetSwitches();
  options->internal_components_factory.reset(
      new TestInternalComponentsFactory(
          factory_switches, InternalComponentsFactory::STORAGE_IN_MEMORY,
          NULL));

  SyncBackendHostImpl::InitCore(std::move(options));
}

void SyncBackendHostForProfileSyncTest::RequestConfigureSyncer(
    syncer::ConfigureReason reason,
    syncer::ModelTypeSet to_download,
    syncer::ModelTypeSet to_purge,
    syncer::ModelTypeSet to_journal,
    syncer::ModelTypeSet to_unapply,
    syncer::ModelTypeSet to_ignore,
    const syncer::ModelSafeRoutingInfo& routing_info,
    const base::Callback<void(syncer::ModelTypeSet,
                              syncer::ModelTypeSet)>& ready_task,
    const base::Closure& retry_callback) {
  syncer::ModelTypeSet failed_configuration_types;

  // The first parameter there should be the set of enabled types.  That's not
  // something we have access to from this strange test harness.  We'll just
  // send back the list of newly configured types instead and hope it doesn't
  // break anything.
  // Posted to avoid re-entrancy issues.
  base::ThreadTaskRunnerHandle::Get()->PostTask(
      FROM_HERE,
      base::Bind(&SyncBackendHostForProfileSyncTest::
                     FinishConfigureDataTypesOnFrontendLoop,
                 base::Unretained(this),
                 syncer::Difference(to_download, failed_configuration_types),
                 syncer::Difference(to_download, failed_configuration_types),
                 failed_configuration_types, ready_task));
}

}  // namespace browser_sync

syncer::TestIdFactory* TestProfileSyncService::id_factory() {
  return &id_factory_;
}

syncer::WeakHandle<syncer::JsEventHandler>
TestProfileSyncService::GetJsEventHandler() {
  return syncer::WeakHandle<syncer::JsEventHandler>();
}

TestProfileSyncService::TestProfileSyncService(
    Profile* profile,
    SigninManagerBase* signin,
    ProfileOAuth2TokenService* oauth2_token_service,
    browser_sync::ProfileSyncServiceStartBehavior behavior)
    : ProfileSyncService(
          GetInitParams(profile, signin, oauth2_token_service, behavior)) {
  static_cast<browser_sync::ChromeSyncClient*>(GetSyncClient())
      ->SetSyncApiComponentFactoryForTesting(
          make_scoped_ptr(new SyncApiComponentFactoryMock));
  SetSyncSetupCompleted();
}

TestProfileSyncService::~TestProfileSyncService() {
}

// static
scoped_ptr<KeyedService> TestProfileSyncService::TestFactoryFunction(
    content::BrowserContext* context) {
  Profile* profile = static_cast<Profile*>(context);
  SigninManagerBase* signin =
      SigninManagerFactory::GetForProfile(profile);
  ProfileOAuth2TokenService* oauth2_token_service =
      ProfileOAuth2TokenServiceFactory::GetForProfile(profile);
  return make_scoped_ptr(new TestProfileSyncService(
      profile, signin, oauth2_token_service, browser_sync::AUTO_START));
}

// static
TestProfileSyncService* TestProfileSyncService::BuildAutoStartAsyncInit(
    Profile* profile, base::Closure callback) {
  TestProfileSyncService* sync_service = static_cast<TestProfileSyncService*>(
        ProfileSyncServiceFactory::GetInstance()->SetTestingFactoryAndUse(
            profile, &TestProfileSyncService::TestFactoryFunction));
  SyncApiComponentFactoryMock* components =
      sync_service->GetSyncApiComponentFactoryMock();
  // TODO(tim): Convert to a fake instead of mock.
  EXPECT_CALL(*components, CreateSyncBackendHost(testing::_, testing::_,
                                                 testing::_, testing::_))
      .WillOnce(
          testing::Return(new browser_sync::SyncBackendHostForProfileSyncTest(
              profile, sync_service->GetSyncClient(),
              BrowserThread::GetMessageLoopProxyForThread(BrowserThread::UI),
              invalidation::ProfileInvalidationProviderFactory::GetForProfile(
                  profile)
                  ->GetInvalidationService(),
              sync_service->sync_prefs_.AsWeakPtr(), callback)));
  return sync_service;
}

SyncApiComponentFactoryMock*
TestProfileSyncService::GetSyncApiComponentFactoryMock() {
  // We always create a mock factory, see Build* routines.
  return static_cast<SyncApiComponentFactoryMock*>(
      GetSyncClient()->GetSyncApiComponentFactory());
}

void TestProfileSyncService::OnConfigureDone(
    const sync_driver::DataTypeManager::ConfigureResult& result) {
  ProfileSyncService::OnConfigureDone(result);
  base::MessageLoop::current()->QuitWhenIdle();
}

UserShare* TestProfileSyncService::GetUserShare() const {
  return backend_->GetUserShare();
}

bool TestProfileSyncService::NeedBackup() const {
  return false;
}
