// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/memory/scoped_ptr.h"
#include "chrome/browser/ui/autofill/save_card_bubble_controller.h"
#import "chrome/browser/ui/cocoa/autofill/save_card_bubble_view_bridge.h"
#import "chrome/browser/ui/cocoa/browser_window_controller.h"
#include "chrome/browser/ui/cocoa/cocoa_profile_test.h"
#include "components/autofill/core/browser/credit_card.h"
#include "testing/gtest/include/gtest/gtest.h"
#import "ui/events/test/cocoa_test_event_utils.h"

#include <Carbon/Carbon.h>  // For the kVK_* constants.

namespace autofill {

namespace {

class TestSaveCardBubbleController : public SaveCardBubbleController {
 public:
  TestSaveCardBubbleController() {
    lines_.reset(new LegalMessageLines());

    on_save_button_was_called_ = false;
    on_cancel_button_was_called_ = false;
    on_learn_more_was_called_ = false;
    on_bubble_closed_was_called_ = false;
  }

  // SaveCardBubbleController:
  base::string16 GetWindowTitle() const override { return base::string16(); }

  base::string16 GetExplanatoryMessage() const override {
    return base::string16();
  }

  const CreditCard GetCard() const override {
    return CreditCard();
  }

  void OnSaveButton() override { on_save_button_was_called_ = true; }
  void OnCancelButton() override { on_cancel_button_was_called_ = true; }
  void OnLearnMoreClicked() override { on_learn_more_was_called_ = true; }
  void OnLegalMessageLinkClicked(const GURL& url) override {}
  void OnBubbleClosed() override { on_bubble_closed_was_called_ = true; }

  const LegalMessageLines& GetLegalMessageLines() const override {
    return *lines_;
  }

  // Testing state.
  bool on_save_button_was_called() { return on_save_button_was_called_; }
  bool on_cancel_button_was_called() { return on_cancel_button_was_called_; }
  bool on_learn_more_was_called() { return on_learn_more_was_called_; }
  bool on_bubble_closed_was_called() { return on_bubble_closed_was_called_; }

 private:
  scoped_ptr<LegalMessageLines> lines_;

  bool on_save_button_was_called_;
  bool on_cancel_button_was_called_;
  bool on_learn_more_was_called_;
  bool on_bubble_closed_was_called_;
};

class SaveCardBubbleViewTest : public CocoaProfileTest {
 public:
  void SetUp() override {
    CocoaProfileTest::SetUp();
    ASSERT_TRUE(browser());

    browser_window_controller_ =
        [[BrowserWindowController alloc] initWithBrowser:browser()
                                           takeOwnership:NO];

    bubble_controller_.reset(new TestSaveCardBubbleController());

    // This will show the SaveCardBubbleViewCocoa.
    bridge_ = new SaveCardBubbleViewBridge(bubble_controller_.get(),
                                           browser_window_controller_);
  }

  void TearDown() override {
    [browser_window_controller_ close];
    CocoaProfileTest::TearDown();
  }

 protected:
  BrowserWindowController* browser_window_controller_;
  scoped_ptr<TestSaveCardBubbleController> bubble_controller_;
  SaveCardBubbleViewBridge* bridge_;
};

}  // namespace

TEST_F(SaveCardBubbleViewTest, SaveShouldClose) {
  [bridge_->view_controller_ onSaveButton:nil];

  EXPECT_TRUE(bubble_controller_->on_save_button_was_called());
  EXPECT_FALSE(bubble_controller_->on_cancel_button_was_called());
  EXPECT_FALSE(bubble_controller_->on_learn_more_was_called());

  EXPECT_TRUE(bubble_controller_->on_bubble_closed_was_called());
}

TEST_F(SaveCardBubbleViewTest, CancelShouldClose) {
  [bridge_->view_controller_ onCancelButton:nil];

  EXPECT_FALSE(bubble_controller_->on_save_button_was_called());
  EXPECT_TRUE(bubble_controller_->on_cancel_button_was_called());
  EXPECT_FALSE(bubble_controller_->on_learn_more_was_called());

  EXPECT_TRUE(bubble_controller_->on_bubble_closed_was_called());
}

TEST_F(SaveCardBubbleViewTest, LearnMoreShouldNotClose) {
  NSTextView* textView = nil;
  NSObject* link = nil;
  [bridge_->view_controller_ textView:textView clickedOnLink:link atIndex:0];

  EXPECT_FALSE(bubble_controller_->on_save_button_was_called());
  EXPECT_FALSE(bubble_controller_->on_cancel_button_was_called());
  EXPECT_TRUE(bubble_controller_->on_learn_more_was_called());

  EXPECT_FALSE(bubble_controller_->on_bubble_closed_was_called());
}

TEST_F(SaveCardBubbleViewTest, ReturnInvokesDefaultAction) {
  [[bridge_->view_controller_ window]
      performKeyEquivalent:cocoa_test_event_utils::KeyEventWithKeyCode(
                               kVK_Return, '\r', NSKeyDown, 0)];

  EXPECT_TRUE(bubble_controller_->on_save_button_was_called());
  EXPECT_TRUE(bubble_controller_->on_bubble_closed_was_called());
}

TEST_F(SaveCardBubbleViewTest, EscapeCloses) {
  [[bridge_->view_controller_ window]
      performKeyEquivalent:cocoa_test_event_utils::KeyEventWithKeyCode(
                               kVK_Escape, '\e', NSKeyDown, 0)];

  EXPECT_TRUE(bubble_controller_->on_bubble_closed_was_called());
}

}  // namespace autofill
