// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef EXTENSIONS_BROWSER_ERROR_MAP_H_
#define EXTENSIONS_BROWSER_ERROR_MAP_H_

#include <stddef.h>

#include <deque>
#include <map>
#include <set>
#include <string>

#include "base/macros.h"
#include "base/memory/scoped_ptr.h"
#include "extensions/browser/extension_error.h"

namespace extensions {

typedef std::deque<const ExtensionError*> ErrorList;

// An ErrorMap is responsible for storing Extension-related errors, keyed by
// Extension ID. The errors are owned by the ErrorMap, and are deleted upon
// destruction.
class ErrorMap {
 public:
  ErrorMap();
  ~ErrorMap();

  struct Filter {
    Filter(const std::string& restrict_to_extension_id,
           int restrict_to_type,
           const std::set<int>& restrict_to_ids,
           bool restrict_to_incognito);
    Filter(const Filter& other);
    ~Filter();

    // Convenience methods to get a specific type of filter. Prefer these over
    // the constructor when possible.
    static Filter ErrorsForExtension(const std::string& extension_id);
    static Filter ErrorsForExtensionWithType(const std::string& extension_id,
                                             ExtensionError::Type type);
    static Filter ErrorsForExtensionWithIds(const std::string& extension_id,
                                            const std::set<int>& ids);
    static Filter ErrorsForExtensionWithTypeAndIds(
        const std::string& extension_id,
        ExtensionError::Type type,
        const std::set<int>& ids);
    static Filter IncognitoErrors();

    bool Matches(const ExtensionError* error) const;

    const std::string restrict_to_extension_id;
    const int restrict_to_type;
    const std::set<int> restrict_to_ids;
    const bool restrict_to_incognito;
  };

  // Return the list of all errors associated with the given extension.
  const ErrorList& GetErrorsForExtension(const std::string& extension_id) const;

  // Add the |error| to the ErrorMap.
  const ExtensionError* AddError(scoped_ptr<ExtensionError> error);

  // Removes errors that match the given |filter| from the map. If non-null,
  // |affected_ids| will be populated with the set of extension ids that were
  // affected by this removal.
  void RemoveErrors(const Filter& filter, std::set<std::string>* affected_ids);

  // Remove all errors for all extensions, and clear the map.
  void RemoveAllErrors();

  size_t size() const { return map_.size(); }

 private:
  // An Entry is created for each Extension ID, and stores the errors related to
  // that Extension.
  class ExtensionEntry;
  using EntryMap = std::map<std::string, ExtensionEntry*>;

  // The mapping between Extension IDs and their corresponding Entries.
  EntryMap map_;

  DISALLOW_COPY_AND_ASSIGN(ErrorMap);
};

}  // namespace extensions

#endif  // EXTENSIONS_BROWSER_ERROR_MAP_H_
