// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "net/cert/x509_util.h"

#include <algorithm>

#include "base/memory/ref_counted.h"
#include "base/memory/scoped_ptr.h"
#include "base/time/time.h"
#include "crypto/rsa_private_key.h"
#include "net/cert/x509_certificate.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace net {

namespace x509_util {

TEST(X509UtilTest, SortClientCertificates) {
  CertificateList certs;

  const base::Time now = base::Time::Now();
  const base::TimeDelta five_days = base::TimeDelta::FromDays(5);

  certs.push_back(scoped_refptr<X509Certificate>(NULL));
  certs.push_back(new X509Certificate(
      "expired", "expired",
      base::Time::UnixEpoch(), base::Time::UnixEpoch()));
  certs.push_back(new X509Certificate(
      "not yet valid", "not yet valid",
      base::Time::Max(), base::Time::Max()));
  certs.push_back(new X509Certificate(
      "older cert", "older cert",
      now - five_days, now + five_days));
  certs.push_back(scoped_refptr<X509Certificate>(NULL));
  certs.push_back(new X509Certificate(
      "newer cert", "newer cert",
      now - base::TimeDelta::FromDays(3), now + five_days));

  std::sort(certs.begin(), certs.end(), ClientCertSorter());

  ASSERT_TRUE(certs[0].get());
  EXPECT_EQ("newer cert", certs[0]->subject().common_name);
  ASSERT_TRUE(certs[1].get());
  EXPECT_EQ("older cert", certs[1]->subject().common_name);
  ASSERT_TRUE(certs[2].get());
  EXPECT_EQ("not yet valid", certs[2]->subject().common_name);
  ASSERT_TRUE(certs[3].get());
  EXPECT_EQ("expired", certs[3]->subject().common_name);
  ASSERT_FALSE(certs[4].get());
  ASSERT_FALSE(certs[5].get());
}

// This test creates a self-signed cert and a private key and then verifies the
// content of the certificate.
TEST(X509UtilTest, CreateKeyAndSelfSigned) {
  scoped_ptr<crypto::RSAPrivateKey> private_key;

  std::string der_cert;
  ASSERT_TRUE(x509_util::CreateKeyAndSelfSignedCert(
      "CN=subject",
      1,
      base::Time::Now(),
      base::Time::Now() + base::TimeDelta::FromDays(1),
      &private_key,
      &der_cert));

  ASSERT_TRUE(private_key.get());

  scoped_refptr<X509Certificate> cert(X509Certificate::CreateFromBytes(
      der_cert.data(), der_cert.size()));
  ASSERT_TRUE(cert.get());

  EXPECT_EQ("subject", cert->subject().GetDisplayName());
  EXPECT_FALSE(cert->HasExpired());
}

// This test creates a self-signed cert from a private key and then verifies the
// content of the certificate.
TEST(X509UtilTest, CreateSelfSigned) {
  const uint8_t private_key_info[] = {
    0x30, 0x82, 0x02, 0x78, 0x02, 0x01, 0x00, 0x30,
    0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x02, 0x62, 0x30, 0x82, 0x02, 0x5e, 0x02, 0x01,
    0x00, 0x02, 0x81, 0x81, 0x00, 0xb8, 0x7f, 0x2b,
    0x20, 0xdc, 0x7c, 0x9b, 0x0c, 0xdc, 0x51, 0x61,
    0x99, 0x0d, 0x36, 0x0f, 0xd4, 0x66, 0x88, 0x08,
    0x55, 0x84, 0xd5, 0x3a, 0xbf, 0x2b, 0xa4, 0x64,
    0x85, 0x7b, 0x0c, 0x04, 0x13, 0x3f, 0x8d, 0xf4,
    0xbc, 0x38, 0x0d, 0x49, 0xfe, 0x6b, 0xc4, 0x5a,
    0xb0, 0x40, 0x53, 0x3a, 0xd7, 0x66, 0x09, 0x0f,
    0x9e, 0x36, 0x74, 0x30, 0xda, 0x8a, 0x31, 0x4f,
    0x1f, 0x14, 0x50, 0xd7, 0xc7, 0x20, 0x94, 0x17,
    0xde, 0x4e, 0xb9, 0x57, 0x5e, 0x7e, 0x0a, 0xe5,
    0xb2, 0x65, 0x7a, 0x89, 0x4e, 0xb6, 0x47, 0xff,
    0x1c, 0xbd, 0xb7, 0x38, 0x13, 0xaf, 0x47, 0x85,
    0x84, 0x32, 0x33, 0xf3, 0x17, 0x49, 0xbf, 0xe9,
    0x96, 0xd0, 0xd6, 0x14, 0x6f, 0x13, 0x8d, 0xc5,
    0xfc, 0x2c, 0x72, 0xba, 0xac, 0xea, 0x7e, 0x18,
    0x53, 0x56, 0xa6, 0x83, 0xa2, 0xce, 0x93, 0x93,
    0xe7, 0x1f, 0x0f, 0xe6, 0x0f, 0x02, 0x03, 0x01,
    0x00, 0x01, 0x02, 0x81, 0x80, 0x03, 0x61, 0x89,
    0x37, 0xcb, 0xf2, 0x98, 0xa0, 0xce, 0xb4, 0xcb,
    0x16, 0x13, 0xf0, 0xe6, 0xaf, 0x5c, 0xc5, 0xa7,
    0x69, 0x71, 0xca, 0xba, 0x8d, 0xe0, 0x4d, 0xdd,
    0xed, 0xb8, 0x48, 0x8b, 0x16, 0x93, 0x36, 0x95,
    0xc2, 0x91, 0x40, 0x65, 0x17, 0xbd, 0x7f, 0xd6,
    0xad, 0x9e, 0x30, 0x28, 0x46, 0xe4, 0x3e, 0xcc,
    0x43, 0x78, 0xf9, 0xfe, 0x1f, 0x33, 0x23, 0x1e,
    0x31, 0x12, 0x9d, 0x3c, 0xa7, 0x08, 0x82, 0x7b,
    0x7d, 0x25, 0x4e, 0x5e, 0x19, 0xa8, 0x9b, 0xed,
    0x86, 0xb2, 0xcb, 0x3c, 0xfe, 0x4e, 0xa1, 0xfa,
    0x62, 0x87, 0x3a, 0x17, 0xf7, 0x60, 0xec, 0x38,
    0x29, 0xe8, 0x4f, 0x34, 0x9f, 0x76, 0x9d, 0xee,
    0xa3, 0xf6, 0x85, 0x6b, 0x84, 0x43, 0xc9, 0x1e,
    0x01, 0xff, 0xfd, 0xd0, 0x29, 0x4c, 0xfa, 0x8e,
    0x57, 0x0c, 0xc0, 0x71, 0xa5, 0xbb, 0x88, 0x46,
    0x29, 0x5c, 0xc0, 0x4f, 0x01, 0x02, 0x41, 0x00,
    0xf5, 0x83, 0xa4, 0x64, 0x4a, 0xf2, 0xdd, 0x8c,
    0x2c, 0xed, 0xa8, 0xd5, 0x60, 0x5a, 0xe4, 0xc7,
    0xcc, 0x61, 0xcd, 0x38, 0x42, 0x20, 0xd3, 0x82,
    0x18, 0xf2, 0x35, 0x00, 0x72, 0x2d, 0xf7, 0x89,
    0x80, 0x67, 0xb5, 0x93, 0x05, 0x5f, 0xdd, 0x42,
    0xba, 0x16, 0x1a, 0xea, 0x15, 0xc6, 0xf0, 0xb8,
    0x8c, 0xbc, 0xbf, 0x54, 0x9e, 0xf1, 0xc1, 0xb2,
    0xb3, 0x8b, 0xb6, 0x26, 0x02, 0x30, 0xc4, 0x81,
    0x02, 0x41, 0x00, 0xc0, 0x60, 0x62, 0x80, 0xe1,
    0x22, 0x78, 0xf6, 0x9d, 0x83, 0x18, 0xeb, 0x72,
    0x45, 0xd7, 0xc8, 0x01, 0x7f, 0xa9, 0xca, 0x8f,
    0x7d, 0xd6, 0xb8, 0x31, 0x2b, 0x84, 0x7f, 0x62,
    0xd9, 0xa9, 0x22, 0x17, 0x7d, 0x06, 0x35, 0x6c,
    0xf3, 0xc1, 0x94, 0x17, 0x85, 0x5a, 0xaf, 0x9c,
    0x5c, 0x09, 0x3c, 0xcf, 0x2f, 0x44, 0x9d, 0xb6,
    0x52, 0x68, 0x5f, 0xf9, 0x59, 0xc8, 0x84, 0x2b,
    0x39, 0x22, 0x8f, 0x02, 0x41, 0x00, 0xb2, 0x04,
    0xe2, 0x0e, 0x56, 0xca, 0x03, 0x1a, 0xc0, 0xf9,
    0x12, 0x92, 0xa5, 0x6b, 0x42, 0xb8, 0x1c, 0xda,
    0x4d, 0x93, 0x9d, 0x5f, 0x6f, 0xfd, 0xc5, 0x58,
    0xda, 0x55, 0x98, 0x74, 0xfc, 0x28, 0x17, 0x93,
    0x1b, 0x75, 0x9f, 0x50, 0x03, 0x7f, 0x7e, 0xae,
    0xc8, 0x95, 0x33, 0x75, 0x2c, 0xd6, 0xa4, 0x35,
    0xb8, 0x06, 0x03, 0xba, 0x08, 0x59, 0x2b, 0x17,
    0x02, 0xdc, 0x4c, 0x7a, 0x50, 0x01, 0x02, 0x41,
    0x00, 0x9d, 0xdb, 0x39, 0x59, 0x09, 0xe4, 0x30,
    0xa0, 0x24, 0xf5, 0xdb, 0x2f, 0xf0, 0x2f, 0xf1,
    0x75, 0x74, 0x0d, 0x5e, 0xb5, 0x11, 0x73, 0xb0,
    0x0a, 0xaa, 0x86, 0x4c, 0x0d, 0xff, 0x7e, 0x1d,
    0xb4, 0x14, 0xd4, 0x09, 0x91, 0x33, 0x5a, 0xfd,
    0xa0, 0x58, 0x80, 0x9b, 0xbe, 0x78, 0x2e, 0x69,
    0x82, 0x15, 0x7c, 0x72, 0xf0, 0x7b, 0x18, 0x39,
    0xff, 0x6e, 0xeb, 0xc6, 0x86, 0xf5, 0xb4, 0xc7,
    0x6f, 0x02, 0x41, 0x00, 0x8d, 0x1a, 0x37, 0x0f,
    0x76, 0xc4, 0x82, 0xfa, 0x5c, 0xc3, 0x79, 0x35,
    0x3e, 0x70, 0x8a, 0xbf, 0x27, 0x49, 0xb0, 0x99,
    0x63, 0xcb, 0x77, 0x5f, 0xa8, 0x82, 0x65, 0xf6,
    0x03, 0x52, 0x51, 0xf1, 0xae, 0x2e, 0x05, 0xb3,
    0xc6, 0xa4, 0x92, 0xd1, 0xce, 0x6c, 0x72, 0xfb,
    0x21, 0xb3, 0x02, 0x87, 0xe4, 0xfd, 0x61, 0xca,
    0x00, 0x42, 0x19, 0xf0, 0xda, 0x5a, 0x53, 0xe3,
    0xb1, 0xc5, 0x15, 0xf3
  };

  std::vector<uint8_t> input;
  input.resize(sizeof(private_key_info));
  memcpy(&input.front(), private_key_info, sizeof(private_key_info));

  scoped_ptr<crypto::RSAPrivateKey> private_key(
      crypto::RSAPrivateKey::CreateFromPrivateKeyInfo(input));
  ASSERT_TRUE(private_key.get());

  std::string der_cert;
  ASSERT_TRUE(x509_util::CreateSelfSignedCert(
      private_key.get(),
      x509_util::DIGEST_SHA1,
      "CN=subject",
      1,
      base::Time::Now(),
      base::Time::Now() + base::TimeDelta::FromDays(1),
      &der_cert));

  scoped_refptr<X509Certificate> cert =
      X509Certificate::CreateFromBytes(der_cert.data(), der_cert.size());
  ASSERT_TRUE(cert.get());

  EXPECT_EQ("subject", cert->subject().GetDisplayName());
  EXPECT_FALSE(cert->HasExpired());
}

#if defined(USE_OPENSSL)
// This is a test case based on
// http://blogs.msdn.com/b/openspecification/archive/2013/03/26/ntlm-and-channel-binding-hash-aka-exteneded-protection-for-authentication.aspx
// There doesn't seem to be too many public test vectors for channel bindings.
TEST(X509UtilTest, CreateChannelBindings_SHA1) {
  // Certificate:
  //     Data:
  //         Version: 3 (0x2)
  //         Serial Number:
  //              (Negative)34:96:86:32:ae:8a:3a:48:b4:98:cf:7c:93:87:bb:d9
  //     Signature Algorithm: sha1WithRSA
  //       ...
  const uint8_t kCertificateDataDER[] = {
      0x30, 0x82, 0x02, 0x09, 0x30, 0x82, 0x01, 0x76, 0xa0, 0x03, 0x02, 0x01,
      0x02, 0x02, 0x10, 0xcb, 0x69, 0x79, 0xcd, 0x51, 0x75, 0xc5, 0xb7, 0x4b,
      0x67, 0x30, 0x83, 0x6c, 0x78, 0x44, 0x27, 0x30, 0x09, 0x06, 0x05, 0x2b,
      0x0e, 0x03, 0x02, 0x1d, 0x05, 0x00, 0x30, 0x16, 0x31, 0x14, 0x30, 0x12,
      0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0b, 0x44, 0x43, 0x2d, 0x57, 0x53,
      0x32, 0x30, 0x30, 0x38, 0x52, 0x32, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x32,
      0x31, 0x31, 0x31, 0x37, 0x30, 0x30, 0x35, 0x39, 0x32, 0x31, 0x5a, 0x17,
      0x0d, 0x33, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35,
      0x39, 0x5a, 0x30, 0x16, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04,
      0x03, 0x13, 0x0b, 0x44, 0x43, 0x2d, 0x57, 0x53, 0x32, 0x30, 0x30, 0x38,
      0x52, 0x32, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
      0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00,
      0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0x9b, 0x00, 0xf8, 0x1a, 0x2d,
      0x37, 0xc6, 0x8d, 0xa1, 0x39, 0x91, 0x46, 0xf3, 0x6a, 0x1b, 0xf9, 0x60,
      0x6c, 0xb3, 0x6c, 0xa0, 0xac, 0xed, 0x85, 0xe0, 0x3f, 0xdc, 0x92, 0x86,
      0x36, 0xbd, 0x64, 0xbf, 0x36, 0x51, 0xdb, 0x57, 0x3a, 0x8a, 0x82, 0x6b,
      0xd8, 0x94, 0x17, 0x7b, 0xd3, 0x91, 0x11, 0x98, 0xef, 0x19, 0x06, 0x52,
      0x30, 0x03, 0x73, 0x67, 0xc8, 0xed, 0x8e, 0xfa, 0x0b, 0x3d, 0x4c, 0xc9,
      0x10, 0x63, 0x9f, 0xcf, 0xb4, 0xcf, 0x39, 0xd8, 0xfe, 0x99, 0xeb, 0x5b,
      0x11, 0xf2, 0xfc, 0xfa, 0x86, 0x24, 0xd9, 0xff, 0xd9, 0x19, 0xf5, 0x69,
      0xb4, 0xdf, 0x5a, 0x5a, 0xc4, 0x94, 0xb4, 0xb0, 0x07, 0x25, 0x97, 0x13,
      0xad, 0x7e, 0x38, 0x14, 0xfb, 0xd6, 0x33, 0x65, 0x6f, 0xe6, 0xf7, 0x48,
      0x4b, 0x2d, 0xb3, 0x51, 0x2e, 0x6d, 0xc7, 0xea, 0x11, 0x76, 0x9a, 0x2b,
      0xf0, 0x00, 0x4d, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x60, 0x30, 0x5e,
      0x30, 0x13, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x0c, 0x30, 0x0a, 0x06,
      0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x47, 0x06,
      0x03, 0x55, 0x1d, 0x01, 0x04, 0x40, 0x30, 0x3e, 0x80, 0x10, 0xeb, 0x65,
      0x26, 0x03, 0x95, 0x4b, 0xd6, 0xc0, 0x54, 0x75, 0x78, 0x7c, 0xb6, 0x2a,
      0xa1, 0xbb, 0xa1, 0x18, 0x30, 0x16, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03,
      0x55, 0x04, 0x03, 0x13, 0x0b, 0x44, 0x43, 0x2d, 0x57, 0x53, 0x32, 0x30,
      0x30, 0x38, 0x52, 0x32, 0x82, 0x10, 0xcb, 0x69, 0x79, 0xcd, 0x51, 0x75,
      0xc5, 0xb7, 0x4b, 0x67, 0x30, 0x83, 0x6c, 0x78, 0x44, 0x27, 0x30, 0x09,
      0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1d, 0x05, 0x00, 0x03, 0x81, 0x81,
      0x00, 0x7b, 0xfa, 0xfe, 0xee, 0x74, 0x05, 0xac, 0xbb, 0x79, 0xe9, 0xda,
      0xca, 0x00, 0x44, 0x96, 0x94, 0x71, 0x92, 0xb1, 0xdb, 0xc9, 0x9b, 0x71,
      0x29, 0xc0, 0xe4, 0x28, 0x5e, 0x6a, 0x50, 0x99, 0xcd, 0xa8, 0x17, 0xe4,
      0x56, 0xb9, 0xef, 0x7f, 0x02, 0x7d, 0x96, 0xa3, 0x48, 0x14, 0x72, 0x75,
      0x2f, 0xb0, 0xb5, 0x87, 0xee, 0x55, 0xe9, 0x6a, 0x6d, 0x28, 0x3c, 0xc1,
      0xfd, 0x00, 0xe4, 0x76, 0xe3, 0x80, 0x88, 0x78, 0x26, 0x0d, 0x6c, 0x8c,
      0xb8, 0x64, 0x61, 0x63, 0xb7, 0x13, 0x3a, 0xab, 0xc7, 0xdd, 0x1d, 0x0a,
      0xd7, 0x15, 0x45, 0xa1, 0xd6, 0xd9, 0x34, 0xc7, 0x21, 0x48, 0xfb, 0x43,
      0x87, 0x38, 0xda, 0x1f, 0x50, 0x47, 0xb1, 0xa5, 0x5c, 0x47, 0xed, 0x04,
      0x44, 0x97, 0xd3, 0xac, 0x74, 0x2d, 0xeb, 0x09, 0x77, 0x59, 0xbf, 0xa3,
      0x54, 0x5b, 0xde, 0x42, 0xd5, 0x23, 0x5a, 0x71, 0x9f};

  const unsigned char kExpectedServerEndPointToken[] = {
      0x74, 0x6c, 0x73, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x2d,
      0x65, 0x6e, 0x64, 0x2d, 0x70, 0x6f, 0x69, 0x6e, 0x74, 0x3a, 0xea,
      0x05, 0xfe, 0xfe, 0xcc, 0x6b, 0x0b, 0xd5, 0x71, 0xdb, 0xbc, 0x5b,
      0xaa, 0x3e, 0xd4, 0x53, 0x86, 0xd0, 0x44, 0x68, 0x35, 0xf7, 0xb7,
      0x4c, 0x85, 0x62, 0x1b, 0x99, 0x83, 0x47, 0x5f, 0x95,
  };

  scoped_refptr<X509Certificate> cert = X509Certificate::CreateFromBytes(
      reinterpret_cast<const char*>(kCertificateDataDER),
      sizeof(kCertificateDataDER));

  std::string channel_bindings;
  ASSERT_TRUE(
      x509_util::GetTLSServerEndPointChannelBinding(*cert, &channel_bindings));

  std::string expected_channel_bindings(
      std::begin(kExpectedServerEndPointToken),
      std::end(kExpectedServerEndPointToken));
  EXPECT_EQ(expected_channel_bindings, channel_bindings);
}

TEST(X509UtilTest, CreateChannelBindings_SHA256) {
  // Certificate:
  //  Data:
  //      Version: 3 (0x2)
  //      Serial Number: 14673274151129443507 (0xcba1f1191dfdecb3)
  //  Signature Algorithm: sha256WithRSAEncryption
  //    ...
  const uint8_t kCertificateDataDER[] = {
      0x30, 0x82, 0x03, 0x8f, 0x30, 0x82, 0x02, 0x77, 0xa0, 0x03, 0x02, 0x01,
      0x02, 0x02, 0x09, 0x00, 0xcb, 0xa1, 0xf1, 0x19, 0x1d, 0xfd, 0xec, 0xb3,
      0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
      0x0b, 0x05, 0x00, 0x30, 0x5e, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
      0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
      0x55, 0x04, 0x08, 0x0c, 0x02, 0x4d, 0x41, 0x31, 0x12, 0x30, 0x10, 0x06,
      0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69,
      0x64, 0x67, 0x65, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0a,
      0x0c, 0x0b, 0x4d, 0x6d, 0x6d, 0x6b, 0x61, 0x79, 0x20, 0x49, 0x6e, 0x63,
      0x2e, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f,
      0x77, 0x77, 0x77, 0x2e, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e,
      0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x33, 0x31,
      0x37, 0x31, 0x39, 0x33, 0x39, 0x34, 0x34, 0x5a, 0x17, 0x0d, 0x31, 0x36,
      0x30, 0x34, 0x31, 0x36, 0x31, 0x39, 0x33, 0x39, 0x34, 0x34, 0x5a, 0x30,
      0x5e, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
      0x55, 0x53, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c,
      0x02, 0x4d, 0x41, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x07,
      0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x31,
      0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x0b, 0x4d, 0x6d,
      0x6d, 0x6b, 0x61, 0x79, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x31, 0x18, 0x30,
      0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77, 0x77, 0x2e,
      0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x30,
      0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
      0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30,
      0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xc6, 0x31, 0xfe, 0x13,
      0x0a, 0xb2, 0x87, 0xc0, 0xb9, 0xa4, 0xb9, 0x86, 0x98, 0x92, 0xc1, 0x48,
      0x41, 0x89, 0xd9, 0xe6, 0xe9, 0x36, 0x1a, 0xd6, 0x1c, 0xcb, 0x86, 0x3a,
      0xec, 0x34, 0x3a, 0xbf, 0x2a, 0xe7, 0x91, 0x33, 0xf4, 0x3e, 0xa4, 0x1b,
      0x9a, 0xc3, 0xa4, 0x66, 0xa0, 0x6d, 0xbf, 0x75, 0x44, 0x1a, 0x79, 0xda,
      0x23, 0x06, 0x5c, 0x07, 0x5c, 0x52, 0x84, 0x46, 0x40, 0xcf, 0x26, 0xa1,
      0x65, 0x65, 0x4e, 0x36, 0x1a, 0xd2, 0xc9, 0x40, 0x28, 0x1e, 0x6f, 0x4f,
      0x7e, 0xb7, 0x10, 0xcd, 0x55, 0x6c, 0xca, 0xf1, 0xfa, 0x66, 0xfb, 0x3e,
      0xb0, 0xf4, 0xc1, 0x92, 0xec, 0xec, 0x0c, 0x1c, 0x79, 0x23, 0x5d, 0xf6,
      0xc0, 0xed, 0xb7, 0x39, 0xa7, 0x59, 0x98, 0xa7, 0x9d, 0x9e, 0x3b, 0xe1,
      0x77, 0x8b, 0x2d, 0x2b, 0x02, 0x5f, 0x30, 0x46, 0xa9, 0x78, 0x47, 0x6f,
      0x05, 0x7d, 0xbc, 0x84, 0x37, 0x09, 0x40, 0x20, 0xba, 0x16, 0x96, 0x4c,
      0xfd, 0xea, 0x83, 0x3e, 0x50, 0xda, 0x24, 0xf4, 0x61, 0x52, 0xfe, 0x9b,
      0xd5, 0x71, 0x37, 0x18, 0x11, 0xe0, 0x7c, 0xbe, 0x98, 0x95, 0x6c, 0x61,
      0xab, 0x9b, 0xdc, 0x5c, 0x59, 0x7d, 0x85, 0x0c, 0xc5, 0x32, 0x07, 0x65,
      0x1e, 0x6a, 0x15, 0xe6, 0xbd, 0xc6, 0xbd, 0xd5, 0x91, 0xea, 0x9d, 0x2f,
      0x0f, 0xdf, 0xef, 0xff, 0x1e, 0x2d, 0x50, 0x44, 0xca, 0x38, 0x27, 0xe3,
      0x20, 0x79, 0x44, 0x47, 0x7a, 0xe3, 0xea, 0x7a, 0x70, 0x41, 0x26, 0x87,
      0xe0, 0x1a, 0x49, 0x1d, 0x62, 0x34, 0xe5, 0xde, 0xc3, 0xb3, 0x12, 0xb8,
      0xa2, 0xf9, 0xad, 0x1b, 0x6a, 0x0f, 0x04, 0xa8, 0xfb, 0xca, 0xe4, 0x01,
      0x6d, 0x77, 0x1f, 0x72, 0xff, 0x58, 0x49, 0x05, 0x1c, 0x1a, 0xb7, 0x76,
      0x29, 0x77, 0x93, 0xcc, 0x09, 0xe8, 0xb7, 0x03, 0x2f, 0x1b, 0xe2, 0xcd,
      0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x50, 0x30, 0x4e, 0x30, 0x1d, 0x06,
      0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xb7, 0x64, 0x60, 0xe7,
      0x6e, 0xb6, 0xaa, 0x25, 0x46, 0xe2, 0x8e, 0x98, 0xac, 0x81, 0xb0, 0xe7,
      0x10, 0x14, 0x85, 0x6d, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
      0x18, 0x30, 0x16, 0x80, 0x14, 0xb7, 0x64, 0x60, 0xe7, 0x6e, 0xb6, 0xaa,
      0x25, 0x46, 0xe2, 0x8e, 0x98, 0xac, 0x81, 0xb0, 0xe7, 0x10, 0x14, 0x85,
      0x6d, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05, 0x30, 0x03,
      0x01, 0x01, 0xff, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
      0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x44,
      0xc4, 0x46, 0x89, 0x69, 0x21, 0xd0, 0x81, 0x30, 0xe3, 0x38, 0xb8, 0x80,
      0xa5, 0x23, 0xd4, 0xfc, 0xe5, 0x12, 0x0f, 0xab, 0x01, 0x44, 0x67, 0xe4,
      0x59, 0x86, 0xd0, 0xa1, 0x8c, 0x2b, 0x36, 0xf0, 0x63, 0x51, 0xbe, 0x24,
      0xfa, 0xf3, 0x1d, 0xc9, 0xfd, 0x40, 0x99, 0x48, 0xea, 0x95, 0x6e, 0xab,
      0xcf, 0xeb, 0x2b, 0x2d, 0x4e, 0xd2, 0xdf, 0xb2, 0xb9, 0x18, 0x27, 0xe3,
      0xc9, 0x17, 0xde, 0x25, 0x6c, 0xcc, 0x80, 0x5a, 0xd5, 0x3b, 0xc5, 0xc8,
      0x3a, 0xc6, 0xcb, 0xa0, 0x33, 0xd3, 0x80, 0x20, 0x4e, 0x3f, 0x61, 0x67,
      0x16, 0x0e, 0xbc, 0xf3, 0x25, 0xe9, 0x62, 0xfa, 0x7f, 0x0e, 0x75, 0x11,
      0xcb, 0x68, 0x24, 0x34, 0x96, 0x1d, 0xb2, 0x88, 0xb7, 0xeb, 0x5e, 0x9c,
      0xc7, 0xa0, 0x3b, 0xcf, 0x4e, 0x3c, 0x7c, 0x6c, 0x29, 0x1b, 0xa5, 0x74,
      0x4d, 0x90, 0xe1, 0xd9, 0x6a, 0x6c, 0x54, 0x6b, 0xce, 0x3d, 0x70, 0x99,
      0xf0, 0x11, 0xc3, 0xda, 0xce, 0xe9, 0xba, 0xc4, 0x91, 0x97, 0xe7, 0x44,
      0x09, 0xa4, 0x39, 0x36, 0xb5, 0xff, 0x3a, 0xce, 0x5d, 0xf1, 0x0b, 0x03,
      0xab, 0xea, 0x0c, 0xe6, 0xbd, 0xd4, 0x43, 0xed, 0x8d, 0x9c, 0xcd, 0x8e,
      0x14, 0xf9, 0xa7, 0xc7, 0xdc, 0xe0, 0xd8, 0xe7, 0x37, 0xf5, 0x48, 0xe8,
      0x84, 0x4c, 0xcd, 0x70, 0x1b, 0xe5, 0x71, 0xa7, 0x4d, 0xc3, 0x03, 0x5e,
      0x50, 0x8a, 0xc4, 0x7c, 0x9c, 0xa4, 0x6f, 0x91, 0xfa, 0x9f, 0xdc, 0xef,
      0x2b, 0x42, 0xe7, 0xd0, 0x0d, 0x60, 0x72, 0x71, 0xe7, 0x13, 0xff, 0x43,
      0xcb, 0x64, 0x33, 0x37, 0x2e, 0xe9, 0x09, 0xeb, 0x7e, 0x8f, 0x15, 0x99,
      0xc9, 0x0d, 0xb0, 0xcb, 0xf0, 0x1f, 0x02, 0xf7, 0xb7, 0x3a, 0xa0, 0x6b,
      0x57, 0xb3, 0x33, 0x14, 0xdb, 0x06, 0x7f, 0x98, 0x6f, 0xc4, 0x32, 0xd5,
      0x14, 0x83, 0x7a};

  const unsigned char kExpectedServerEndPointToken[] = {
      0x74, 0x6c, 0x73, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x2d,
      0x65, 0x6e, 0x64, 0x2d, 0x70, 0x6f, 0x69, 0x6e, 0x74, 0x3a, 0x96,
      0x4c, 0xd0, 0x44, 0x53, 0x81, 0x0b, 0x35, 0x01, 0x05, 0x54, 0x02,
      0x15, 0x79, 0xac, 0x74, 0x3c, 0xa4, 0x91, 0xa2, 0xb7, 0x60, 0xbe,
      0x75, 0x57, 0x0e, 0xaf, 0xa3, 0x10, 0xbf, 0xf1, 0xbb,
  };

  scoped_refptr<X509Certificate> cert = X509Certificate::CreateFromBytes(
      reinterpret_cast<const char*>(kCertificateDataDER),
      sizeof(kCertificateDataDER));

  std::string channel_bindings;
  ASSERT_TRUE(
      x509_util::GetTLSServerEndPointChannelBinding(*cert, &channel_bindings));

  std::string expected_channel_bindings(
      std::begin(kExpectedServerEndPointToken),
      std::end(kExpectedServerEndPointToken));
  EXPECT_EQ(expected_channel_bindings, channel_bindings);
}

TEST(X509UtilTest, CreateChannelBindings_SHA384) {
  // Certificate:
  //  Data:
  //      Version: 3 (0x2)
  //      Serial Number: 10071585730232401378 (0x8bc575dc2f653de2)
  //  Signature Algorithm: sha384WithRSAEncryption
  //    ...
  const uint8_t kCertificateDataDER[] = {
      0x30, 0x82, 0x03, 0xe1, 0x30, 0x82, 0x02, 0xc9, 0xa0, 0x03, 0x02, 0x01,
      0x02, 0x02, 0x09, 0x00, 0x8b, 0xc5, 0x75, 0xdc, 0x2f, 0x65, 0x3d, 0xe2,
      0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
      0x0c, 0x05, 0x00, 0x30, 0x81, 0x86, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
      0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x0b, 0x30, 0x09, 0x06,
      0x03, 0x55, 0x04, 0x08, 0x0c, 0x02, 0x4d, 0x41, 0x31, 0x12, 0x30, 0x10,
      0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72,
      0x69, 0x64, 0x67, 0x65, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04,
      0x0a, 0x0c, 0x0b, 0x4d, 0x6d, 0x6d, 0x6b, 0x61, 0x79, 0x20, 0x49, 0x6e,
      0x63, 0x2e, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c,
      0x1d, 0x52, 0x61, 0x6e, 0x64, 0x6f, 0x6d, 0x20, 0x43, 0x65, 0x72, 0x74,
      0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x47, 0x65, 0x6e, 0x65,
      0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03,
      0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77, 0x77, 0x2e, 0x65, 0x78, 0x61,
      0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d,
      0x31, 0x36, 0x30, 0x33, 0x31, 0x37, 0x31, 0x39, 0x35, 0x36, 0x34, 0x39,
      0x5a, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x34, 0x31, 0x36, 0x31, 0x39, 0x35,
      0x36, 0x34, 0x39, 0x5a, 0x30, 0x81, 0x86, 0x31, 0x0b, 0x30, 0x09, 0x06,
      0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x0b, 0x30, 0x09,
      0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x02, 0x4d, 0x41, 0x31, 0x12, 0x30,
      0x10, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62,
      0x72, 0x69, 0x64, 0x67, 0x65, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55,
      0x04, 0x0a, 0x0c, 0x0b, 0x4d, 0x6d, 0x6d, 0x6b, 0x61, 0x79, 0x20, 0x49,
      0x6e, 0x63, 0x2e, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x0b,
      0x0c, 0x1d, 0x52, 0x61, 0x6e, 0x64, 0x6f, 0x6d, 0x20, 0x43, 0x65, 0x72,
      0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x47, 0x65, 0x6e,
      0x65, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x18, 0x30, 0x16, 0x06,
      0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77, 0x77, 0x2e, 0x65, 0x78,
      0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x82, 0x01,
      0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
      0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01,
      0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xb4, 0xfa, 0x6c, 0xe7, 0xac, 0x91,
      0xa4, 0x00, 0xe1, 0xe8, 0xed, 0x5b, 0xe5, 0x86, 0x6b, 0x3d, 0x1d, 0x87,
      0xf7, 0x49, 0x5a, 0xda, 0x67, 0x65, 0x34, 0x57, 0x7b, 0x70, 0x71, 0x11,
      0xbd, 0xee, 0xfe, 0x01, 0x91, 0xb6, 0x43, 0x11, 0x0f, 0xec, 0x3f, 0x2f,
      0x05, 0xdb, 0x28, 0x44, 0x7e, 0x9c, 0x2b, 0xbb, 0xe9, 0x49, 0x2b, 0x62,
      0x1e, 0x8b, 0x4c, 0xf9, 0xa8, 0x2f, 0xfb, 0x24, 0x83, 0x13, 0xeb, 0x7f,
      0x52, 0x4c, 0xbb, 0xc5, 0x89, 0x6c, 0xe4, 0x22, 0xc0, 0x0c, 0x71, 0xda,
      0xb2, 0x36, 0xae, 0xc8, 0xe5, 0x10, 0x8f, 0x4a, 0x68, 0x65, 0xc3, 0x07,
      0xb5, 0xcf, 0xec, 0x08, 0xd9, 0x4b, 0x13, 0x4a, 0x25, 0x56, 0xec, 0x74,
      0x4d, 0xb3, 0xb5, 0x77, 0x3f, 0xf9, 0xa5, 0x0b, 0x15, 0x80, 0xf9, 0xe7,
      0x01, 0x4b, 0x73, 0x6a, 0x1e, 0xc8, 0xb0, 0x7a, 0x58, 0x13, 0x25, 0x50,
      0x40, 0x7e, 0x5c, 0x5d, 0xff, 0x2b, 0x9d, 0xf5, 0x43, 0xd2, 0xed, 0xca,
      0x87, 0xe3, 0x4e, 0x9f, 0x25, 0xa7, 0x75, 0xe9, 0x48, 0x7e, 0xb9, 0x38,
      0x74, 0x3f, 0x30, 0x6b, 0xec, 0x94, 0xb8, 0x0f, 0x00, 0x41, 0x60, 0xaa,
      0x48, 0x7d, 0x17, 0x65, 0x1a, 0x9a, 0x0d, 0x49, 0x18, 0xe8, 0x72, 0xbb,
      0x16, 0x37, 0xa0, 0x65, 0x0c, 0xe4, 0x69, 0x0f, 0x93, 0xca, 0x80, 0x7e,
      0xf6, 0x90, 0xbd, 0xf2, 0x86, 0xa3, 0xb6, 0x81, 0x7c, 0xe3, 0x40, 0x7a,
      0x65, 0x33, 0x71, 0xfd, 0x8c, 0xe6, 0xf6, 0x63, 0xcf, 0x40, 0xdd, 0x29,
      0xf2, 0x15, 0x25, 0x96, 0x1a, 0xa5, 0x14, 0x1e, 0xa2, 0xfd, 0x53, 0xb8,
      0xbc, 0x20, 0x10, 0x87, 0xcb, 0x8e, 0x24, 0xa9, 0x47, 0x60, 0x47, 0xcc,
      0xb2, 0x2e, 0xe5, 0x0f, 0x3e, 0x2c, 0x22, 0x9a, 0x1b, 0x46, 0x5a, 0xe4,
      0x2b, 0x7f, 0x4a, 0xab, 0x33, 0x49, 0xea, 0xd8, 0x6a, 0xd5, 0x02, 0x03,
      0x01, 0x00, 0x01, 0xa3, 0x50, 0x30, 0x4e, 0x30, 0x1d, 0x06, 0x03, 0x55,
      0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x0f, 0x62, 0xc9, 0x94, 0xc0, 0x00,
      0x55, 0x51, 0x60, 0x36, 0xce, 0xd6, 0x4d, 0x6b, 0x5e, 0x20, 0x02, 0x6c,
      0x2e, 0xed, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30,
      0x16, 0x80, 0x14, 0x0f, 0x62, 0xc9, 0x94, 0xc0, 0x00, 0x55, 0x51, 0x60,
      0x36, 0xce, 0xd6, 0x4d, 0x6b, 0x5e, 0x20, 0x02, 0x6c, 0x2e, 0xed, 0x30,
      0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01,
      0xff, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
      0x01, 0x0c, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x31, 0x7f, 0xcd,
      0x8b, 0xf1, 0x9c, 0x87, 0x59, 0xed, 0xe2, 0x36, 0x18, 0x84, 0x1c, 0x54,
      0x4e, 0x92, 0x45, 0xaf, 0xe9, 0xa1, 0x78, 0x89, 0x6e, 0xad, 0xec, 0x51,
      0xf9, 0x81, 0x70, 0xd9, 0x94, 0x01, 0xa9, 0x2a, 0x0d, 0x5f, 0xda, 0x83,
      0x49, 0x32, 0x24, 0x7e, 0xfd, 0x73, 0x97, 0xa2, 0x05, 0xd4, 0xf1, 0xb7,
      0x3f, 0xa0, 0x49, 0xbc, 0x0c, 0x85, 0x3e, 0xd3, 0x15, 0x67, 0xef, 0x5f,
      0xfb, 0xbe, 0x0d, 0x2a, 0x97, 0xcd, 0xaa, 0x24, 0x1e, 0x5a, 0xdd, 0x47,
      0xcf, 0x70, 0xe4, 0x93, 0xe6, 0xc3, 0xad, 0x78, 0x84, 0x60, 0xfa, 0x16,
      0x2e, 0xcc, 0x98, 0x88, 0xde, 0x80, 0xbb, 0x8f, 0x46, 0xfd, 0x59, 0x95,
      0xb4, 0x0b, 0x07, 0x8a, 0x67, 0x04, 0x10, 0xbd, 0x32, 0xb0, 0xa2, 0xfe,
      0x0f, 0xe0, 0x3e, 0x49, 0x84, 0x89, 0xc6, 0x18, 0xd9, 0xca, 0xdd, 0x63,
      0x0d, 0x55, 0x46, 0x56, 0xd1, 0x83, 0xcb, 0x0d, 0x23, 0x21, 0xb3, 0x59,
      0xa9, 0xd5, 0x56, 0x83, 0x4f, 0xf0, 0x92, 0x0b, 0xcc, 0xed, 0x93, 0x02,
      0xe7, 0x07, 0x92, 0x15, 0x90, 0x18, 0x78, 0xc9, 0xf4, 0x59, 0x0e, 0xbe,
      0xaa, 0x3e, 0x91, 0xb7, 0x81, 0xa2, 0x0c, 0x28, 0x04, 0xde, 0x78, 0xb0,
      0xd1, 0x54, 0x90, 0x8f, 0x3c, 0xfd, 0x2e, 0x36, 0x6e, 0xfe, 0x7b, 0xf6,
      0xce, 0x65, 0xae, 0x99, 0xa5, 0x38, 0x29, 0x4b, 0x13, 0x2f, 0x01, 0xa8,
      0xb0, 0xd0, 0x43, 0x22, 0x01, 0xd1, 0x71, 0x35, 0x93, 0xcf, 0x1e, 0x9e,
      0x05, 0x03, 0x05, 0xff, 0xfc, 0x35, 0x34, 0x6f, 0x49, 0x04, 0xe6, 0x58,
      0xa6, 0x0a, 0xb6, 0x13, 0x67, 0xd8, 0x1e, 0xe7, 0x33, 0x0f, 0x51, 0xb1,
      0x5c, 0x75, 0x1c, 0xea, 0x2b, 0x94, 0xec, 0x32, 0x7e, 0xb0, 0x8c, 0xb7,
      0x63, 0xf4, 0x83, 0x58, 0x3e, 0x47, 0xb7, 0x92, 0x67, 0x2c, 0xd5, 0xdb,
      0x48};

  const unsigned char kExpectedServerEndPointToken[] = {
      0x74, 0x6c, 0x73, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x2d, 0x65,
      0x6e, 0x64, 0x2d, 0x70, 0x6f, 0x69, 0x6e, 0x74, 0x3a, 0x9c, 0xd9, 0x91,
      0xd4, 0x57, 0x1b, 0x64, 0x9d, 0x84, 0x31, 0xfd, 0xda, 0x3d, 0xaa, 0x2e,
      0x26, 0xab, 0xee, 0x40, 0x6b, 0x95, 0x36, 0x82, 0xc4, 0x5f, 0xad, 0xc1,
      0x26, 0x29, 0x72, 0xda, 0xe6, 0xfd, 0xa6, 0xd8, 0x0e, 0xab, 0xa2, 0xca,
      0x7d, 0x7d, 0x18, 0xa8, 0xee, 0x80, 0x26, 0xb2, 0x1f,
  };

  scoped_refptr<X509Certificate> cert = X509Certificate::CreateFromBytes(
      reinterpret_cast<const char*>(kCertificateDataDER),
      sizeof(kCertificateDataDER));

  std::string channel_bindings;
  ASSERT_TRUE(
      x509_util::GetTLSServerEndPointChannelBinding(*cert, &channel_bindings));

  std::string expected_channel_bindings(
      std::begin(kExpectedServerEndPointToken),
      std::end(kExpectedServerEndPointToken));
  EXPECT_EQ(expected_channel_bindings, channel_bindings);
}

TEST(X509UtilTest, CreateChannelBindings_SHA512) {
  // Certificate:
  //  Data:
  //      Version: 3 (0x2)
  //      Serial Number: 15197703563697605045 (0xd2e916decbe249b5)
  //  Signature Algorithm: sha512WithRSAEncryption
  //    ...
  const uint8_t kCertificateDataDER[] = {
      0x30, 0x82, 0x03, 0x81, 0x30, 0x82, 0x02, 0x69, 0xa0, 0x03, 0x02, 0x01,
      0x02, 0x02, 0x09, 0x00, 0xd2, 0xe9, 0x16, 0xde, 0xcb, 0xe2, 0x49, 0xb5,
      0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
      0x0d, 0x05, 0x00, 0x30, 0x57, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
      0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
      0x55, 0x04, 0x08, 0x0c, 0x02, 0x4d, 0x41, 0x31, 0x12, 0x30, 0x10, 0x06,
      0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69,
      0x64, 0x67, 0x65, 0x31, 0x0d, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x04, 0x0a,
      0x0c, 0x04, 0x42, 0x6c, 0x61, 0x68, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03,
      0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77, 0x77, 0x2e, 0x65, 0x78, 0x61,
      0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d,
      0x31, 0x36, 0x30, 0x33, 0x31, 0x37, 0x32, 0x30, 0x31, 0x33, 0x31, 0x30,
      0x5a, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x34, 0x31, 0x36, 0x32, 0x30, 0x31,
      0x33, 0x31, 0x30, 0x5a, 0x30, 0x57, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
      0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x0b, 0x30, 0x09, 0x06,
      0x03, 0x55, 0x04, 0x08, 0x0c, 0x02, 0x4d, 0x41, 0x31, 0x12, 0x30, 0x10,
      0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72,
      0x69, 0x64, 0x67, 0x65, 0x31, 0x0d, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x04,
      0x0a, 0x0c, 0x04, 0x42, 0x6c, 0x61, 0x68, 0x31, 0x18, 0x30, 0x16, 0x06,
      0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77, 0x77, 0x2e, 0x65, 0x78,
      0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x82, 0x01,
      0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
      0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01,
      0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xd3, 0xec, 0x61, 0x08, 0xbd, 0x91,
      0x94, 0x07, 0xaf, 0xf1, 0xf5, 0xeb, 0x84, 0x6d, 0x4b, 0x42, 0x0b, 0x40,
      0xb6, 0x6a, 0x51, 0x4d, 0x41, 0xde, 0x1c, 0xad, 0xf2, 0xcb, 0x73, 0xdb,
      0x41, 0x54, 0xd2, 0x13, 0xd6, 0xe9, 0x9b, 0xbe, 0x49, 0x3c, 0x6a, 0xeb,
      0x16, 0xa1, 0xee, 0x88, 0xbf, 0x58, 0xe6, 0x25, 0x52, 0x84, 0xdd, 0x3d,
      0x62, 0x34, 0x22, 0x81, 0x98, 0x55, 0x0d, 0xce, 0x6e, 0xf1, 0x72, 0xae,
      0x70, 0x34, 0x51, 0x4b, 0x54, 0xc8, 0x79, 0xe3, 0x57, 0xab, 0x24, 0xc0,
      0x29, 0x67, 0x38, 0xb1, 0x7f, 0x71, 0x32, 0x60, 0x08, 0x34, 0xa1, 0x92,
      0x6d, 0x78, 0xd6, 0x16, 0x73, 0x68, 0x32, 0x8b, 0xf8, 0x95, 0xa5, 0x79,
      0xfb, 0xa5, 0xd3, 0x8d, 0xb2, 0xa4, 0x4d, 0xc3, 0x22, 0xff, 0x2f, 0xae,
      0x6f, 0x12, 0xc9, 0xa6, 0xad, 0xe3, 0x6d, 0xc7, 0x40, 0x9b, 0x6a, 0xba,
      0x3c, 0x8c, 0xf3, 0x85, 0xa6, 0xc0, 0xbd, 0x49, 0xb6, 0x4f, 0xed, 0xaf,
      0x10, 0xb4, 0x17, 0x26, 0x40, 0x21, 0x01, 0xb7, 0xa9, 0xf0, 0x06, 0x98,
      0xfa, 0x03, 0x09, 0xe1, 0xfc, 0x7e, 0xcd, 0x70, 0x09, 0xcb, 0x19, 0x00,
      0x0a, 0x3c, 0x3a, 0xd8, 0x95, 0xfc, 0xd7, 0x2e, 0x92, 0x06, 0x25, 0x03,
      0xbd, 0x3f, 0x05, 0xf4, 0x89, 0x23, 0x41, 0x08, 0xc8, 0x3c, 0x62, 0xdc,
      0x25, 0xed, 0x01, 0x68, 0xf3, 0x75, 0x4d, 0xc5, 0xc2, 0xb0, 0x36, 0xb3,
      0x0f, 0x78, 0x4a, 0x2c, 0xa7, 0xbb, 0x13, 0x38, 0x09, 0xfd, 0x49, 0x95,
      0x99, 0x4c, 0x23, 0x36, 0x0a, 0x0f, 0x70, 0xbe, 0x02, 0xc3, 0x50, 0x30,
      0xb4, 0xaa, 0x7d, 0x47, 0x42, 0xbe, 0x62, 0x59, 0xbe, 0x88, 0xc4, 0x96,
      0x42, 0x63, 0xf3, 0x27, 0x09, 0x30, 0x9e, 0x58, 0xe6, 0xbe, 0x12, 0x2b,
      0x22, 0x11, 0x72, 0x65, 0x6c, 0x47, 0x50, 0x8c, 0x69, 0x0b, 0x02, 0x03,
      0x01, 0x00, 0x01, 0xa3, 0x50, 0x30, 0x4e, 0x30, 0x1d, 0x06, 0x03, 0x55,
      0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x40, 0x1f, 0x60, 0x96, 0x13, 0x3e,
      0xb4, 0x93, 0x65, 0x71, 0x0e, 0x1b, 0xa3, 0xe6, 0x25, 0x21, 0xd6, 0x90,
      0x69, 0xed, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30,
      0x16, 0x80, 0x14, 0x40, 0x1f, 0x60, 0x96, 0x13, 0x3e, 0xb4, 0x93, 0x65,
      0x71, 0x0e, 0x1b, 0xa3, 0xe6, 0x25, 0x21, 0xd6, 0x90, 0x69, 0xed, 0x30,
      0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01,
      0xff, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
      0x01, 0x0d, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x03, 0x4e, 0x0b,
      0xdd, 0x81, 0xe9, 0xf9, 0x91, 0x92, 0xa2, 0x77, 0xd0, 0x8f, 0x73, 0xb6,
      0x28, 0x7d, 0x3e, 0x2f, 0x36, 0x25, 0x01, 0x0f, 0x66, 0x14, 0x4d, 0x98,
      0x9d, 0x90, 0x08, 0x41, 0xfa, 0xe0, 0x8e, 0x15, 0xb6, 0xf9, 0x2c, 0x67,
      0x02, 0x31, 0x79, 0xa4, 0x2f, 0xb7, 0xf0, 0xf2, 0xfa, 0x28, 0xcf, 0xe5,
      0x71, 0x57, 0xd2, 0x18, 0xd5, 0xbd, 0x2b, 0xa8, 0x88, 0x10, 0xaf, 0xe9,
      0x68, 0x97, 0xb0, 0x04, 0x3f, 0x07, 0x5d, 0xb0, 0xc6, 0xfd, 0x34, 0xf0,
      0x14, 0xea, 0x81, 0xf5, 0x00, 0xd7, 0xe5, 0x0b, 0xde, 0x9a, 0x78, 0x93,
      0xe9, 0x64, 0x38, 0x93, 0x53, 0x57, 0x5c, 0x22, 0x1d, 0xec, 0xec, 0x77,
      0xc9, 0xed, 0x79, 0x8b, 0x85, 0xe6, 0xf2, 0xb0, 0x11, 0x34, 0x25, 0x20,
      0xdb, 0x3a, 0x82, 0x72, 0x15, 0x6e, 0xbb, 0x2b, 0x6c, 0xad, 0x26, 0x4d,
      0xb4, 0x2b, 0xd4, 0xd5, 0x45, 0x19, 0x6f, 0x3c, 0xd3, 0x42, 0x53, 0xdb,
      0x70, 0x6b, 0xd1, 0x9f, 0x24, 0x40, 0x55, 0xad, 0x64, 0x7e, 0x66, 0x80,
      0x45, 0xfb, 0x30, 0x97, 0x20, 0x3d, 0x3e, 0x14, 0xfe, 0x88, 0x0d, 0xd9,
      0x3a, 0x7e, 0x30, 0x5c, 0x75, 0x87, 0x59, 0x37, 0xe7, 0xb9, 0xff, 0x3d,
      0xf8, 0x47, 0xe4, 0xb7, 0xa6, 0xcc, 0x62, 0xc8, 0xfa, 0x55, 0x93, 0xe8,
      0xe4, 0x33, 0x29, 0x33, 0x91, 0xeb, 0x16, 0x95, 0xdc, 0x63, 0xaf, 0x5e,
      0x9b, 0x0c, 0xa4, 0x99, 0x20, 0x90, 0x80, 0x1c, 0xbe, 0x9f, 0x22, 0x93,
      0x21, 0xe1, 0x1c, 0x16, 0x92, 0x42, 0x42, 0x67, 0xfb, 0xf3, 0x31, 0x8c,
      0x78, 0x41, 0x43, 0xed, 0x5f, 0x09, 0x0b, 0x21, 0x72, 0x40, 0x95, 0xdb,
      0xd7, 0x8d, 0x2a, 0xe1, 0x52, 0x03, 0x57, 0xc8, 0x7b, 0x15, 0x90, 0x92,
      0xdf, 0x07, 0x11, 0xfb, 0xf7, 0x25, 0xbe, 0xcb, 0x0b, 0xfa, 0xa6, 0x34,
      0x61,
  };

  const unsigned char kExpectedServerEndPointToken[] = {
      0x74, 0x6c, 0x73, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x2d,
      0x65, 0x6e, 0x64, 0x2d, 0x70, 0x6f, 0x69, 0x6e, 0x74, 0x3a, 0x2c,
      0x33, 0xc9, 0xee, 0x30, 0xa1, 0x99, 0xdc, 0x19, 0x48, 0x0a, 0xec,
      0xc5, 0x1d, 0x94, 0xc7, 0x84, 0x3d, 0xc7, 0x9e, 0x51, 0xee, 0xbc,
      0x23, 0x33, 0xfd, 0xeb, 0x20, 0x78, 0x3b, 0x93, 0xe8, 0xe6, 0x78,
      0x10, 0xd7, 0x5e, 0x37, 0x48, 0x04, 0x07, 0x5f, 0x57, 0xe6, 0x16,
      0xe2, 0x45, 0x44, 0xa3, 0x71, 0x8c, 0xef, 0x95, 0x08, 0x6b, 0x0c,
      0xaa, 0x65, 0x51, 0x51, 0x60, 0x4f, 0x58, 0x28,
  };

  scoped_refptr<X509Certificate> cert = X509Certificate::CreateFromBytes(
      reinterpret_cast<const char*>(kCertificateDataDER),
      sizeof(kCertificateDataDER));

  std::string channel_bindings;
  ASSERT_TRUE(
      x509_util::GetTLSServerEndPointChannelBinding(*cert, &channel_bindings));

  std::string expected_channel_bindings(
      std::begin(kExpectedServerEndPointToken),
      std::end(kExpectedServerEndPointToken));
  EXPECT_EQ(expected_channel_bindings, channel_bindings);
}

TEST(X509UtilTest, CreateChannelBindings_Unsupported_MD4) {
  // Certificate:
  //  Data:
  //      Version: 3 (0x2)
  //      Serial Number: 12629177056471137087 (0xaf43d99ee079bb3f)
  //  Signature Algorithm: md4WithRSAEncryption
  //  ...
  const uint8_t kCertificateDataDER[] = {
      0x30, 0x82, 0x03, 0x87, 0x30, 0x82, 0x02, 0x6f, 0xa0, 0x03, 0x02, 0x01,
      0x02, 0x02, 0x09, 0x00, 0xaf, 0x43, 0xd9, 0x9e, 0xe0, 0x79, 0xbb, 0x3f,
      0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
      0x03, 0x05, 0x00, 0x30, 0x5a, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
      0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
      0x55, 0x04, 0x08, 0x0c, 0x02, 0x4d, 0x41, 0x31, 0x12, 0x30, 0x10, 0x06,
      0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69,
      0x64, 0x67, 0x65, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a,
      0x0c, 0x07, 0x45, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x31, 0x18, 0x30,
      0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77, 0x77, 0x2e,
      0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x30,
      0x1e, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x33, 0x31, 0x37, 0x32, 0x30, 0x31,
      0x39, 0x35, 0x39, 0x5a, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x34, 0x31, 0x36,
      0x32, 0x30, 0x31, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x5a, 0x31, 0x0b, 0x30,
      0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x0b,
      0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x02, 0x4d, 0x41, 0x31,
      0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61,
      0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x31, 0x10, 0x30, 0x0e, 0x06,
      0x03, 0x55, 0x04, 0x0a, 0x0c, 0x07, 0x45, 0x78, 0x61, 0x6d, 0x70, 0x6c,
      0x65, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f,
      0x77, 0x77, 0x77, 0x2e, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e,
      0x63, 0x6f, 0x6d, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a,
      0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82,
      0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00,
      0xaf, 0xf8, 0xa5, 0xba, 0x20, 0x40, 0xcd, 0x60, 0x88, 0x1f, 0x27, 0x34,
      0x57, 0x2f, 0x1b, 0xc9, 0xfc, 0xf3, 0x0b, 0x9f, 0xb3, 0xb8, 0x26, 0x67,
      0x78, 0x79, 0x2b, 0xfe, 0x68, 0xa3, 0x95, 0x8c, 0x40, 0x6f, 0x80, 0x8b,
      0x15, 0x8f, 0x0b, 0x4c, 0x66, 0x8d, 0x17, 0x32, 0x6a, 0xa0, 0x57, 0x74,
      0x2d, 0x31, 0xc4, 0x8e, 0xa4, 0x76, 0x6b, 0xff, 0x93, 0x05, 0x44, 0x9b,
      0xe9, 0xf2, 0x81, 0x1e, 0x3b, 0x2e, 0x05, 0x67, 0x01, 0x85, 0x5a, 0xed,
      0xad, 0xc8, 0x55, 0x4b, 0x8e, 0x80, 0x1e, 0x7a, 0x5a, 0xbd, 0xc2, 0x8e,
      0x5f, 0x4f, 0x3a, 0x52, 0xda, 0x2d, 0x86, 0x0d, 0x6b, 0xaf, 0xe2, 0x12,
      0xd1, 0x6e, 0x15, 0x78, 0xb5, 0x06, 0x1c, 0xa2, 0xb4, 0xda, 0x3c, 0xfe,
      0x8b, 0x4c, 0x70, 0x5c, 0xc5, 0x5c, 0x86, 0xf6, 0x95, 0x01, 0x88, 0x44,
      0x99, 0xd2, 0x97, 0x47, 0xdb, 0xdb, 0x4f, 0x6d, 0x1a, 0xd1, 0x1d, 0x74,
      0x0b, 0x89, 0x5c, 0x05, 0x19, 0xe7, 0xe7, 0x17, 0xee, 0xb5, 0xf8, 0x75,
      0x96, 0x95, 0x5a, 0xfd, 0x37, 0x4d, 0xa1, 0xda, 0x5c, 0xf6, 0x5c, 0x60,
      0xa1, 0x04, 0x92, 0xa5, 0x42, 0x02, 0xbd, 0x40, 0xb3, 0x1e, 0xab, 0x28,
      0x0b, 0x28, 0x79, 0x92, 0xdc, 0x23, 0xde, 0xad, 0x28, 0x06, 0xba, 0x93,
      0x28, 0x3f, 0xf4, 0x6a, 0x10, 0x11, 0xf1, 0xe4, 0x92, 0x5b, 0x65, 0xce,
      0x0f, 0x13, 0x34, 0x87, 0xb6, 0xa4, 0x1d, 0xe3, 0x4a, 0xea, 0xc8, 0xcf,
      0x4e, 0x21, 0xf2, 0x70, 0x18, 0x9f, 0x3a, 0x62, 0x43, 0x70, 0x89, 0xb9,
      0xca, 0xab, 0xa6, 0xb1, 0x55, 0x42, 0x3d, 0x25, 0x35, 0xd6, 0xa2, 0x77,
      0x0f, 0x66, 0x88, 0xd2, 0x43, 0x6b, 0xfa, 0x3c, 0xa3, 0xcf, 0xa1, 0x56,
      0x84, 0xe5, 0x9b, 0x25, 0xff, 0x02, 0xb6, 0x9d, 0xaf, 0x6a, 0x4d, 0x8c,
      0xf8, 0x3e, 0xf7, 0xe5, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x50, 0x30,
      0x4e, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14,
      0x62, 0xf2, 0x1f, 0xd6, 0x29, 0xfa, 0x61, 0x88, 0x4c, 0x36, 0x39, 0xed,
      0x9f, 0x8e, 0xcf, 0x3b, 0xdc, 0x90, 0x04, 0xa5, 0x30, 0x1f, 0x06, 0x03,
      0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x62, 0xf2, 0x1f,
      0xd6, 0x29, 0xfa, 0x61, 0x88, 0x4c, 0x36, 0x39, 0xed, 0x9f, 0x8e, 0xcf,
      0x3b, 0xdc, 0x90, 0x04, 0xa5, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13,
      0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06, 0x09, 0x2a,
      0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x03, 0x05, 0x00, 0x03, 0x82,
      0x01, 0x01, 0x00, 0x9e, 0x36, 0xff, 0xf5, 0x77, 0x46, 0x80, 0x22, 0xc4,
      0x01, 0x05, 0x69, 0x84, 0xd1, 0xdc, 0x0c, 0xce, 0xda, 0x2f, 0x21, 0x11,
      0x9a, 0x80, 0x24, 0x81, 0xea, 0x1b, 0x1a, 0xb9, 0x5d, 0x70, 0x60, 0xaf,
      0xbc, 0xca, 0xdb, 0xc4, 0x10, 0x6e, 0x17, 0xdb, 0x58, 0x11, 0x0c, 0xec,
      0x0e, 0xdd, 0xf8, 0xbf, 0x42, 0x27, 0x2d, 0x06, 0x75, 0x25, 0xc0, 0xdb,
      0xb0, 0x35, 0x0c, 0x66, 0xfb, 0xfd, 0x0e, 0xb9, 0x16, 0x9e, 0x34, 0xaf,
      0xdb, 0x72, 0x6e, 0xb3, 0x6d, 0x5e, 0xb2, 0xf0, 0x6d, 0x46, 0x32, 0xba,
      0x35, 0xb5, 0x3b, 0xc5, 0x72, 0xb7, 0x1c, 0x88, 0x6a, 0x84, 0xc2, 0xde,
      0x1d, 0x45, 0x30, 0x41, 0x3d, 0xc5, 0xbb, 0x0a, 0x78, 0xe9, 0xef, 0x59,
      0xc6, 0xbc, 0x0c, 0x9d, 0x36, 0x1f, 0xc8, 0xb1, 0x1b, 0xd3, 0x1c, 0xec,
      0x57, 0x69, 0x8b, 0xcb, 0x1b, 0x93, 0x99, 0x2e, 0xab, 0x8d, 0xd7, 0x63,
      0xc5, 0x1f, 0x58, 0x9d, 0x06, 0x8e, 0xf1, 0x47, 0x64, 0xc5, 0xf3, 0x03,
      0x8d, 0x2c, 0x8a, 0x99, 0x2c, 0xe6, 0x89, 0x87, 0xb2, 0x42, 0xdd, 0x18,
      0xf0, 0xc9, 0xdb, 0x6c, 0xee, 0x3e, 0x5e, 0xc5, 0x75, 0xc7, 0x45, 0xd3,
      0xd0, 0xaa, 0xf7, 0x0d, 0xac, 0xaa, 0x4c, 0xb4, 0xa4, 0xa2, 0xb7, 0xfc,
      0x65, 0xb0, 0x2e, 0x83, 0xbd, 0x7a, 0x65, 0xfb, 0x1b, 0x92, 0x3f, 0x8d,
      0x47, 0x0c, 0xbb, 0x4a, 0x38, 0xc5, 0x67, 0x87, 0x8e, 0x79, 0xd6, 0x48,
      0xc7, 0xf2, 0x92, 0xdb, 0x6e, 0xf9, 0x54, 0x01, 0xed, 0xdb, 0xbd, 0xe0,
      0x12, 0x18, 0xd9, 0xcf, 0x36, 0x5d, 0x7e, 0xe9, 0xb9, 0x27, 0x09, 0x80,
      0x1a, 0x2f, 0xb9, 0x58, 0xc6, 0x6f, 0xa2, 0x42, 0x7a, 0x7b, 0x20, 0xf2,
      0xc9, 0x85, 0x54, 0x11, 0xa2, 0xe3, 0x31, 0x0f, 0x71, 0x6a, 0xea, 0x0e,
      0xef, 0xba, 0x6d, 0x5e, 0x88, 0x01, 0xaf,
  };

  scoped_refptr<X509Certificate> cert = X509Certificate::CreateFromBytes(
      reinterpret_cast<const char*>(kCertificateDataDER),
      sizeof(kCertificateDataDER));

  std::string channel_bindings;
  ASSERT_FALSE(
      x509_util::GetTLSServerEndPointChannelBinding(*cert, &channel_bindings));
  EXPECT_TRUE(channel_bindings.empty());
}
#endif

}  // namespace x509_util

}  // namespace net
