#!/usr/bin/env python
# Copyright 2015 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

'''Generates a list of runtime Blimp Engine runtime dependencies.
'''


import argparse
import fnmatch
import os
import subprocess
import sys

# Returns True if |entry| matches any of the patterns in |blacklist|.
def IsBlacklisted(entry, blacklist):
  return any([next_pat for next_pat in blacklist
              if fnmatch.fnmatch(entry, next_pat)])

def main():
  parser = argparse.ArgumentParser(description=__doc__)
  parser.add_argument('--build-dir',
                      help=('build output directory (e.g. out/Debug)'),
                      required=True,
                      metavar='DIR')
  parser.add_argument('--target',
                      help=('build target of engine'),
                      required=True)
  parser.add_argument('--output',
                      help=('name and path of manifest file to create'),
                      required=True,
                      metavar='FILE')
  args = parser.parse_args()

  try:
    deps = subprocess.check_output(['gn', 'desc', args.build_dir, args.target,
                                    'runtime_deps']).split()
  except subprocess.CalledProcessError as e:
    print "Error: " + ' '.join(e.cmd)
    print e.output
    exit(1)

  command_line = ' '.join([os.path.basename(sys.argv[0])] + sys.argv[1:])
  header = [
      '# Runtime dependencies for the Blimp Engine',
      '#',
      '# This file was generated by running:',
      '# ' + command_line + '',
      '#',
      '# Note: Any unnecessary dependencies should be added to',
      '#       manifest-blacklist.txt and this file should be regenerated.',
      '',
  ]

  blacklist_patterns = []
  with open(os.path.join(os.sys.path[0], 'manifest-blacklist.txt'), 'r') \
      as blacklist_file:
    blacklist_patterns = \
        [entry.partition('#')[0].strip() for entry \
         in blacklist_file.readlines()]

  with open(args.output, 'w') as manifest:
    manifest.write('\n'.join(header))
    manifest.write('\n'.join([dep for dep in deps
                              if not IsBlacklisted(dep, blacklist_patterns)]))

  print 'Created ' + args.output

if __name__ == "__main__":
  main()
