// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/android/contextualsearch/contextual_search_field_trial.h"

#include "base/command_line.h"
#include "base/strings/string_number_conversions.h"
#include "components/variations/variations_associated_data.h"

namespace {

const char kContextualSearchFieldTrialName[] = "ContextualSearch";
const char kFalseValue[] = "false";
const char kAnyNonEmptyValue[] = "1";
const char kContextualSearchResolverUrl[] = "contextual-search-resolver-url";
const char kContextualSearchSurroundingSizeParamName[] = "surrounding_size";
const char kContextualSearchIcingSurroundingSizeParamName[] =
    "icing_surrounding_size";
const char kContextualSearchSendURLDisabledParamName[] = "disable_send_url";
const char kContextualSearchDecodeMentionsDisabledParamName[] =
    "disable_decode_mentions";
// The default size of the content surrounding the selection to gather, allowing
// room for other parameters.
const int kContextualSearchDefaultContentSize = 1536;

}  // namespace

// static
const int
    ContextualSearchFieldTrial::kContextualSearchDefaultIcingSurroundingSize =
        400;

ContextualSearchFieldTrial::ContextualSearchFieldTrial()
    : is_resolver_url_prefix_cached_(false),
      is_surrounding_size_cached_(false),
      surrounding_size_(0),
      is_icing_surrounding_size_cached_(false),
      icing_surrounding_size_(0),
      is_send_base_page_url_disabled_cached_(false),
      is_send_base_page_url_disabled_(false),
      is_decode_mentions_disabled_cached_(false),
      is_decode_mentions_disabled_(false) {}

ContextualSearchFieldTrial::~ContextualSearchFieldTrial() {}

std::string ContextualSearchFieldTrial::GetResolverURLPrefix() {
  if (!is_resolver_url_prefix_cached_) {
    is_resolver_url_prefix_cached_ = true;
    resolver_url_prefix_ = GetSwitch(kContextualSearchResolverUrl);
    if (resolver_url_prefix_.empty())
      resolver_url_prefix_ = GetParam(kContextualSearchResolverUrl);
  }
  return resolver_url_prefix_;
}

int ContextualSearchFieldTrial::GetSurroundingSize() {
  return GetIntParamValueOrDefault(kContextualSearchSurroundingSizeParamName,
                                   kContextualSearchDefaultContentSize,
                                   &is_surrounding_size_cached_,
                                   &surrounding_size_);
}

int ContextualSearchFieldTrial::GetIcingSurroundingSize() {
  return GetIntParamValueOrDefault(
      kContextualSearchIcingSurroundingSizeParamName,
      kContextualSearchDefaultIcingSurroundingSize,
      &is_icing_surrounding_size_cached_, &icing_surrounding_size_);
}

bool ContextualSearchFieldTrial::IsSendBasePageURLDisabled() {
  return GetBooleanParam(kContextualSearchSendURLDisabledParamName,
                         &is_send_base_page_url_disabled_cached_,
                         &is_send_base_page_url_disabled_);
}

bool ContextualSearchFieldTrial::IsDecodeMentionsDisabled() {
  return GetBooleanParam(kContextualSearchDecodeMentionsDisabledParamName,
                         &is_decode_mentions_disabled_cached_,
                         &is_decode_mentions_disabled_);
}

bool ContextualSearchFieldTrial::GetBooleanParam(const std::string& name,
                                                 bool* is_value_cached,
                                                 bool* cached_value) {
  if (!*is_value_cached) {
    *is_value_cached = true;
    std::string string_value = GetSwitch(name);
    // A switch with an empty value is true.
    bool has_switch = HasSwitch(name);
    if (has_switch && string_value.empty())
      string_value = kAnyNonEmptyValue;
    if (!has_switch)
      string_value = GetParam(name);
    *cached_value = !string_value.empty() && string_value != kFalseValue;
  }
  return *cached_value;
}

int ContextualSearchFieldTrial::GetIntParamValueOrDefault(
    const std::string& name,
    const int default_value,
    bool* is_value_cached,
    int* cached_value) {
  if (!*is_value_cached) {
    *is_value_cached = true;
    std::string param_string = GetSwitch(name);
    if (param_string.empty())
      param_string = GetParam(name);

    int param_int;
    if (!param_string.empty() && base::StringToInt(param_string, &param_int))
      *cached_value = param_int;
    else
      *cached_value = default_value;
  }
  return *cached_value;
}

bool ContextualSearchFieldTrial::HasSwitch(const std::string& name) {
  return base::CommandLine::ForCurrentProcess()->HasSwitch(name);
}

std::string ContextualSearchFieldTrial::GetSwitch(const std::string& name) {
  if (!HasSwitch(name))
    return std::string();
  else
    return base::CommandLine::ForCurrentProcess()->GetSwitchValueASCII(name);
}

std::string ContextualSearchFieldTrial::GetParam(const std::string& name) {
  return variations::GetVariationParamValue(kContextualSearchFieldTrialName,
                                            name);
}
