// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/views/profiles/signin_view_controller_delegate_views.h"

#include "base/macros.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/profiles/profile_avatar_icon_util.h"
#include "chrome/browser/signin/signin_promo.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/browser/ui/tabs/tab_strip_model.h"
#include "chrome/common/url_constants.h"
#include "components/constrained_window/constrained_window_views.h"
#include "components/signin/core/common/profile_management_switches.h"
#include "content/public/browser/render_widget_host_view.h"
#include "content/public/browser/web_contents.h"
#include "ui/views/controls/webview/webview.h"
#include "ui/views/widget/widget.h"

const int kPasswordCombinedFixedGaiaViewHeight = 440;
const int kPasswordCombinedFixedGaiaViewWidth = 360;
const int kFixedGaiaViewHeight = 612;
const int kModalDialogWidth = 448;
const int kSyncConfirmationDialogHeight = 487;

SigninViewControllerDelegateViews::SigninViewControllerDelegateViews(
    SigninViewController* signin_view_controller,
    views::WebView* content_view,
    Browser* browser,
    bool wait_for_size)
    : SigninViewControllerDelegate(signin_view_controller,
                                   content_view->GetWebContents()),
      content_view_(content_view),
      modal_signin_widget_(nullptr),
      wait_for_size_(wait_for_size),
      browser_(browser) {
  if (!wait_for_size_)
    DisplayModal();
}

SigninViewControllerDelegateViews::~SigninViewControllerDelegateViews() {}

// views::DialogDelegateView:
views::View* SigninViewControllerDelegateViews::GetContentsView() {
  return content_view_;
}

views::Widget* SigninViewControllerDelegateViews::GetWidget() {
  return content_view_->GetWidget();
}

const views::Widget* SigninViewControllerDelegateViews::GetWidget() const {
  return content_view_->GetWidget();
}

void SigninViewControllerDelegateViews::DeleteDelegate() {
  ResetSigninViewControllerDelegate();
  delete this;
}

ui::ModalType SigninViewControllerDelegateViews::GetModalType() const {
  return ui::MODAL_TYPE_CHILD;
}

bool SigninViewControllerDelegateViews::ShouldShowCloseButton() const {
  return false;
}

int SigninViewControllerDelegateViews::GetDialogButtons() const {
  return ui::DIALOG_BUTTON_NONE;
}

void SigninViewControllerDelegateViews::PerformClose() {
  modal_signin_widget_->Close();
}

void SigninViewControllerDelegateViews::ResizeNativeView(int height) {
  content_view_->SetPreferredSize(gfx::Size(kModalDialogWidth, height));
  content_view_->Layout();

  if (wait_for_size_) {
    // The modal wasn't displayed yet so just show it with the already resized
    // view.
    DisplayModal();
  }
}

void SigninViewControllerDelegateViews::DisplayModal() {
  modal_signin_widget_ = constrained_window::ShowWebModalDialogViews(
      this, browser_->tab_strip_model()->GetActiveWebContents());
  content_view_->RequestFocus();
}

// static
views::WebView* SigninViewControllerDelegateViews::CreateGaiaWebView(
    content::WebContentsDelegate* delegate,
    profiles::BubbleViewMode mode,
    Profile* profile,
    signin_metrics::AccessPoint access_point) {
  GURL url =
      signin::GetSigninURLFromBubbleViewMode(profile, mode, access_point);

  // Adds Gaia signin webview.
  const gfx::Size pref_size =
      switches::UsePasswordSeparatedSigninFlow()
          ? gfx::Size(kModalDialogWidth, kFixedGaiaViewHeight)
          : gfx::Size(kPasswordCombinedFixedGaiaViewWidth,
                      kPasswordCombinedFixedGaiaViewHeight);
  views::WebView* web_view = new views::WebView(profile);
  web_view->LoadInitialURL(url);

  if (delegate)
    web_view->GetWebContents()->SetDelegate(delegate);

  web_view->SetPreferredSize(pref_size);
  content::RenderWidgetHostView* rwhv =
      web_view->GetWebContents()->GetRenderWidgetHostView();
  if (rwhv)
    rwhv->SetBackgroundColor(profiles::kAvatarBubbleGaiaBackgroundColor);

  return web_view;
}

views::WebView*
SigninViewControllerDelegateViews::CreateSyncConfirmationWebView(
    Profile* profile) {
  views::WebView* web_view = new views::WebView(profile);
  web_view->LoadInitialURL(GURL(chrome::kChromeUISyncConfirmationURL));
  web_view->SetPreferredSize(
      gfx::Size(kModalDialogWidth, kSyncConfirmationDialogHeight));

  return web_view;
}

SigninViewControllerDelegate*
SigninViewControllerDelegate::CreateModalSigninDelegate(
    SigninViewController* signin_view_controller,
    profiles::BubbleViewMode mode,
    Browser* browser,
    signin_metrics::AccessPoint access_point) {
  return new SigninViewControllerDelegateViews(
      signin_view_controller,
      SigninViewControllerDelegateViews::CreateGaiaWebView(
          nullptr, mode, browser->profile(), access_point),
      browser, false);
}

SigninViewControllerDelegate*
SigninViewControllerDelegate::CreateSyncConfirmationDelegate(
    SigninViewController* signin_view_controller,
    Browser* browser) {
  return new SigninViewControllerDelegateViews(
      signin_view_controller,
      SigninViewControllerDelegateViews::CreateSyncConfirmationWebView(
          browser->profile()),
      browser, true);
}
