// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_COCOA_BROWSER_WINDOW_CONTROLLER_PRIVATE_H_
#define CHROME_BROWSER_UI_COCOA_BROWSER_WINDOW_CONTROLLER_PRIVATE_H_

#import "chrome/browser/ui/cocoa/browser_window_controller.h"
#import "chrome/browser/ui/cocoa/browser_window_layout.h"
#import "chrome/browser/ui/cocoa/fullscreen_toolbar_controller.h"

@class BrowserWindowLayout;
class PermissionRequestManager;

namespace content {
class WebContents;
}  // content.

// Private methods for the |BrowserWindowController|. This category should
// contain the private methods used by different parts of the BWC; private
// methods used only by single parts should be declared in their own file.
// TODO(viettrungluu): [crbug.com/35543] work on splitting out stuff from the
// BWC, and figuring out which methods belong here (need to unravel
// "dependencies").
@interface BrowserWindowController(Private)

// Create the appropriate tab strip controller based on whether or not side
// tabs are enabled. Replaces the current controller.
- (void)createTabStripController;

// Sets the window's collection behavior to the appropriate
// fullscreen behavior.
- (void)updateFullscreenCollectionBehavior;

// Saves the window's position in the local state preferences.
- (void)saveWindowPositionIfNeeded;

// We need to adjust where sheets come out of the window, as by default they
// erupt from the omnibox, which is rather weird.
- (NSRect)window:(NSWindow*)window
    willPositionSheet:(NSWindow*)sheet
            usingRect:(NSRect)defaultSheetRect;

// Repositions the window's subviews. From the top down: toolbar, normal
// bookmark bar (if shown), infobar, NTP detached bookmark bar (if shown),
// content area, download shelf (if any).
- (void)layoutSubviews;

// Shows the informational "how to exit fullscreen" bubble.
- (void)showFullscreenExitBubbleIfNecessary;

// Lays out the tab strip and avatar button.
- (void)applyTabStripLayout:(const chrome::TabStripLayout&)layout;

// Returns YES if the bookmark bar should be placed below the infobar, NO
// otherwise.
- (BOOL)placeBookmarkBarBelowInfoBar;

// Lays out the tab content area in the given frame. If the height changes,
// sends a message to the renderer to resize.
- (void)layoutTabContentArea:(NSRect)frame;

// Sets the toolbar's height to a value appropriate for the given compression.
// Also adjusts the bookmark bar's height by the opposite amount in order to
// keep the total height of the two views constant.
- (void)adjustToolbarAndBookmarkBarForCompression:(CGFloat)compression;

// Moves views between windows in preparation for fullscreen mode when not using
// Cocoa's System Fullscreen API.  (System Fullscreen reuses the original window
// for fullscreen mode, so there is no need to move views around.)  This method
// does not position views; callers must also call |-layoutSubviews:|.
- (void)moveViewsForImmersiveFullscreen:(BOOL)fullscreen
                          regularWindow:(NSWindow*)regularWindow
                       fullscreenWindow:(NSWindow*)fullscreenWindow;

// Updates the anchor position of the permission bubble.
- (void)updatePermissionBubbleAnchor;

// Enter or exit fullscreen without using Cocoa's System Fullscreen API.  These
// methods are internal implementations of |-setFullscreen:|.
- (void)enterImmersiveFullscreen;
- (void)exitImmersiveFullscreen;

// Register or deregister for content view resize notifications.  These
// notifications are used while transitioning into fullscreen mode using Cocoa's
// System Fullscreen API.
- (void)registerForContentViewResizeNotifications;
- (void)deregisterForContentViewResizeNotifications;

// Allows/prevents bar visibility locks and releases from updating the visual
// state. Enabling makes changes instantaneously; disabling cancels any
// timers/animation.
- (void)enableBarVisibilityUpdates;
- (void)disableBarVisibilityUpdates;

// If there are no visibility locks and bar visibity updates are enabled, hides
// the bar with |animation|.  Otherwise, does nothing.
- (void)hideOverlayIfPossibleWithAnimation:(BOOL)animation;

// The opacity for the toolbar divider; 0 means that it shouldn't be shown.
- (CGFloat)toolbarDividerOpacity;

// Update visibility of the infobar tip, depending on the state of the window.
- (void)updateInfoBarTipVisibility;

// The min Y of the bubble point in the coordinate space of the toolbar.
- (NSInteger)pageInfoBubblePointY;

// Configures the fullscreenToolbarController_ right after it is constructed.
- (void)configureFullscreenToolbarController;

// Allows the omnibox to slide. Also prepares UI for several fullscreen modes.
// This method gets called when entering AppKit fullscren, or when entering
// Immersive fullscreen. Expects fullscreenStyle_ to be set.
- (void)adjustUIForSlidingFullscreenStyle:(fullscreen_mac::SlidingStyle)style;

// This method gets called when exiting AppKit fullscreen, or when exiting
// Immersive fullscreen. It performs some common UI changes, and stops the
// omnibox from sliding.
- (void)adjustUIForExitingFullscreenAndStopOmniboxSliding;

// Exposed for testing.
// Creates a FullscreenToolbarController with the given style.
- (FullscreenToolbarController*)newFullscreenToolbarControllerWithStyle:
    (fullscreen_mac::SlidingStyle)style;

// Toggles the AppKit Fullscreen API. By default, doing so enters Canonical
// Fullscreen.
- (void)enterAppKitFullscreen;
- (void)exitAppKitFullscreen;

// Returns where the fullscreen button should be positioned in the window.
// Returns NSZeroRect if there is no fullscreen button (if currently in
// fullscreen, or if running 10.6 or 10.10+).
- (NSRect)fullscreenButtonFrame;

// Updates |layout| with the full set of parameters required to statelessly
// determine the layout of the views managed by this controller.
- (void)updateLayoutParameters:(BrowserWindowLayout*)layout;

// Applies a layout to the views managed by this controller.
- (void)applyLayout:(BrowserWindowLayout*)layout;

// Ensures that the window's content view's subviews have the correct
// z-ordering. Will add or remove subviews as necessary.
- (void)updateSubviewZOrder;

// Performs updateSubviewZOrder when this controller is not in fullscreen.
- (void)updateSubviewZOrderNormal;

// Performs updateSubviewZOrder when this controller is in fullscreen.
- (void)updateSubviewZOrderFullscreen;

// Sets the content view's subviews. Attempts to not touch the tabContentArea
// to prevent redraws.
- (void)setContentViewSubviews:(NSArray*)subviews;

// There is a bug in Mavericks for applications linked against OSX 10.8 and
// earlier. The bug requires Screens Have Separate Spaces to be enabled, and
// for the window to be on a secondary screen. When AppKit Fullscreen is
// invoked on the window, its final frame is 22pt too short. These methods
// detect when the relevant conditions have been met so that a hack can be
// applied to fix the size of the window.
// http://crbug.com/396980
+ (BOOL)systemSettingsRequireMavericksAppKitFullscreenHack;
- (BOOL)shouldUseMavericksAppKitFullscreenHack;

// Whether the instance should use a custom transition when animating into and
// out of AppKit Fullscreen.
- (BOOL)shouldUseCustomAppKitFullscreenTransition:(BOOL)enterFullScreen;

// Resets the variables that were set from using custom AppKit fullscreen
// animation.
- (void)resetCustomAppKitFullscreenVariables;

- (content::WebContents*)webContents;
- (PermissionRequestManager*)permissionRequestManager;

// Hides or unhides any displayed modal sheet for fullscreen transition.
// Modal sheets should be hidden at the beginning and then shown at the end.
- (void)setSheetHiddenForFullscreenTransition:(BOOL)shoudHide;

// Adjusts the UI and destroys the exit bubble when we are exiting fullscreen.
- (void)adjustUIForExitingFullscreen;

// Determines the appropriate sliding fullscreen style and adjusts the UI to
// it when we are entering fullscreen.
- (void)adjustUIForEnteringFullscreen;

// Returns YES if the fullscreen is for tab content or an extension.
- (BOOL)isFullscreenForTabContentOrExtension;

@end  // @interface BrowserWindowController(Private)

#endif  // CHROME_BROWSER_UI_COCOA_BROWSER_WINDOW_CONTROLLER_PRIVATE_H_
