// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_WEBUI_SETTINGS_PROTOCOL_HANDLERS_HANDLER_H_
#define CHROME_BROWSER_UI_WEBUI_SETTINGS_PROTOCOL_HANDLERS_HANDLER_H_

#include <string>

#include "base/macros.h"
#include "chrome/browser/custom_handlers/protocol_handler_registry.h"
#include "chrome/browser/ui/webui/settings/settings_page_ui_handler.h"
#include "chrome/common/custom_handlers/protocol_handler.h"
#include "content/public/browser/notification_observer.h"
#include "content/public/browser/notification_registrar.h"

////////////////////////////////////////////////////////////////////////////////
// ProtocolHandlersHandler

// Listen for changes to protocol handlers (i.e. registerProtocolHandler()).
// This get triggered whenever a user allows a specific website or application
// to handle clicks on a link with a specified protocol (i.e. mailto: -> Gmail).

namespace base {
class DictionaryValue;
}

namespace settings {

class ProtocolHandlersHandler : public SettingsPageUIHandler,
                              public content::NotificationObserver {
 public:
  ProtocolHandlersHandler();
  ~ProtocolHandlersHandler() override;

  // SettingsPageUIHandler:
  void OnJavascriptAllowed() override;
  void OnJavascriptDisallowed() override;
  void RegisterMessages() override;

  // content::NotificationObserver:
  void Observe(int type,
               const content::NotificationSource& source,
               const content::NotificationDetails& details) override;

 private:
  // Called to fetch the initial list of data to show.
  void InitializeProtocolHandlerList(const base::ListValue* args);

  // Notifies the JS side whether the handlers are enabled or not.
  void SendHandlersEnabledValue();

  // Called when the user toggles whether custom handlers are enabled.
  void SetHandlersEnabled(const base::ListValue* args);

  // Called when the user sets a new default handler for a protocol.
  void SetDefault(const base::ListValue* args);

  // Called when the user clears the default handler for a protocol.
  // |args| is the string name of the protocol to clear.
  void ClearDefault(const base::ListValue* args);

  // Parses a ProtocolHandler out of the arguments passed back from the view.
  // |args| is a list of [protocol, url, title].
  ProtocolHandler ParseHandlerFromArgs(const base::ListValue* args) const;

  // Returns a JSON object describing the set of protocol handlers for the
  // given protocol.
  void GetHandlersForProtocol(const std::string& protocol,
                              base::DictionaryValue* value);

  // Returns a JSON list of the ignored protocol handlers.
  void GetIgnoredHandlers(base::ListValue* handlers);

  // Called when the JS PasswordManager object is initialized.
  void UpdateHandlerList();

  // Remove a handler.
  // |args| is a list of [protocol, url, title].
  void RemoveHandler(const base::ListValue* args);

  // Remove an ignored handler.
  // |args| is a list of [protocol, url, title].
  void RemoveIgnoredHandler(const base::ListValue* args);

  ProtocolHandlerRegistry* GetProtocolHandlerRegistry();

  content::NotificationRegistrar notification_registrar_;

  DISALLOW_COPY_AND_ASSIGN(ProtocolHandlersHandler);
};

}  // namespace settings

#endif  // CHROME_BROWSER_UI_WEBUI_SETTINGS_PROTOCOL_HANDLERS_HANDLER_H_
