// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "blimp/client/core/contents/blimp_contents_impl.h"

#include "base/memory/ptr_util.h"
#include "base/supports_user_data.h"
#include "blimp/client/core/contents/blimp_contents_view.h"
#include "blimp/client/core/contents/tab_control_feature.h"
#include "blimp/client/public/contents/blimp_contents_observer.h"
#include "ui/gfx/native_widget_types.h"

#if defined(OS_ANDROID)
#include "blimp/client/core/contents/android/blimp_contents_impl_android.h"
#endif  // OS_ANDROID

namespace blimp {
namespace client {

namespace {

#if defined(OS_ANDROID)
const char kBlimpContentsImplAndroidKey[] = "blimp_contents_impl_android";
#endif  // OS_ANDROID
}

BlimpContentsImpl::BlimpContentsImpl(
    int id,
    gfx::NativeWindow window,
    BlimpCompositorDependencies* compositor_deps,
    ImeFeature* ime_feature,
    NavigationFeature* navigation_feature,
    RenderWidgetFeature* render_widget_feature,
    TabControlFeature* tab_control_feature)
    : navigation_controller_(id, this, navigation_feature),
      compositor_manager_(id, render_widget_feature, compositor_deps),
      id_(id),
      ime_feature_(ime_feature),
      window_(window),
      tab_control_feature_(tab_control_feature) {
  blimp_contents_view_ =
      BlimpContentsView::Create(this, compositor_manager_.layer());
  ime_feature_->set_delegate(blimp_contents_view_->GetImeDelegate());
}

BlimpContentsImpl::~BlimpContentsImpl() {
  FOR_EACH_OBSERVER(BlimpContentsObserver, observers_, BlimpContentsDying());
  ime_feature_->set_delegate(nullptr);
}

#if defined(OS_ANDROID)

base::android::ScopedJavaLocalRef<jobject> BlimpContentsImpl::GetJavaObject() {
  return GetBlimpContentsImplAndroid()->GetJavaObject();
}

BlimpContentsImplAndroid* BlimpContentsImpl::GetBlimpContentsImplAndroid() {
  BlimpContentsImplAndroid* blimp_contents_impl_android =
      static_cast<BlimpContentsImplAndroid*>(
          GetUserData(kBlimpContentsImplAndroidKey));
  if (!blimp_contents_impl_android) {
    blimp_contents_impl_android = new BlimpContentsImplAndroid(this);
    SetUserData(kBlimpContentsImplAndroidKey, blimp_contents_impl_android);
  }
  return blimp_contents_impl_android;
}

#endif  // defined(OS_ANDROID)

BlimpNavigationControllerImpl& BlimpContentsImpl::GetNavigationController() {
  return navigation_controller_;
}

gfx::NativeWindow BlimpContentsImpl::GetNativeWindow() {
  return window_;
}

void BlimpContentsImpl::AddObserver(BlimpContentsObserver* observer) {
  observers_.AddObserver(observer);
}

void BlimpContentsImpl::RemoveObserver(BlimpContentsObserver* observer) {
  observers_.RemoveObserver(observer);
}

gfx::NativeView BlimpContentsImpl::GetNativeView() {
  return blimp_contents_view_->GetNativeView();
}

void BlimpContentsImpl::Show() {
  compositor_manager_.SetVisible(true);
}

void BlimpContentsImpl::Hide() {
  compositor_manager_.SetVisible(false);
}

bool BlimpContentsImpl::HasObserver(BlimpContentsObserver* observer) {
  return observers_.HasObserver(observer);
}

void BlimpContentsImpl::OnNavigationStateChanged() {
  FOR_EACH_OBSERVER(BlimpContentsObserver, observers_,
                    OnNavigationStateChanged());
}

void BlimpContentsImpl::OnLoadingStateChanged(bool loading) {
  FOR_EACH_OBSERVER(BlimpContentsObserver, observers_,
                    OnLoadingStateChanged(loading));
}

void BlimpContentsImpl::SetSizeAndScale(const gfx::Size& size,
                                        float device_pixel_ratio) {
  tab_control_feature_->SetSizeAndScale(size, device_pixel_ratio);
}

BlimpContentsView* BlimpContentsImpl::GetBlimpContentsView() {
  return blimp_contents_view_.get();
}

}  // namespace client
}  // namespace blimp
