// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_SAFE_BROWSING_THREAT_DETAILS_CACHE_H_
#define CHROME_BROWSER_SAFE_BROWSING_THREAT_DETAILS_CACHE_H_

// A class that gets threat details from the HTTP Cache.
// An instance of this class is generated by ThreatDetails.

#include <string>
#include <vector>

#include "base/callback.h"
#include "base/containers/hash_tables.h"
#include "base/memory/linked_ptr.h"
#include "base/memory/ref_counted.h"
#include "chrome/common/safe_browsing/csd.pb.h"
#include "net/base/completion_callback.h"
#include "net/url_request/url_fetcher_delegate.h"

namespace net {
class URLFetcher;
class URLRequestContext;
}

namespace safe_browsing {

// Maps a URL to its Resource.
typedef base::hash_map<
    std::string,
    linked_ptr<ClientSafeBrowsingReportRequest::Resource>>
    ResourceMap;

class ThreatDetailsCacheCollector
    : public base::RefCountedThreadSafe<ThreatDetailsCacheCollector>,
      public net::URLFetcherDelegate {
 public:
  ThreatDetailsCacheCollector();

  // We use |request_context_getter|, we modify |resources| and
  // |result|, and we call |callback|, so they must all remain alive
  // for the lifetime of this object.
  void StartCacheCollection(
      net::URLRequestContextGetter* request_context_getter,
      ResourceMap* resources,
      bool* result,
      const base::Closure& callback);

  // Returns whether or not StartCacheCollection has been called.
  bool HasStarted();

 protected:
  // Implementation of URLFetcher::Delegate. Called after the request
  // completes (either successfully or with failure).
  void OnURLFetchComplete(const net::URLFetcher* source) override;

 private:
  friend class base::RefCountedThreadSafe<ThreatDetailsCacheCollector>;

  ~ThreatDetailsCacheCollector() override;

  // Points to the url for which we are fetching the HTTP cache entry or
  // redirect chain.
  ResourceMap::iterator resources_it_;

  // Points to the resources_ map in the ThreatDetails.
  ResourceMap* resources_;

  // Points to the cache_result_ in the ThreatDetails.
  bool* result_;

  // Method we call when we are done. The caller must be alive for the
  // whole time, we are modifying its state (see above).
  base::Closure callback_;

  // Set to true as soon as StartCacheCollection is called.
  bool has_started_;

  // Used to get a pointer to the current request context.
  scoped_refptr<net::URLRequestContextGetter> request_context_getter_;

  // The current URLFetcher.
  std::unique_ptr<net::URLFetcher> current_fetch_;

  // Returns the resource from resources_ that corresponds to |url|
  ClientSafeBrowsingReportRequest::Resource* GetResource(
      const GURL& url);

  // Creates a new URLFetcher and starts it.
  void OpenEntry();

  // Read the HTTP response from |source| and add it to |pb_resource|.
  void ReadResponse(
      ClientSafeBrowsingReportRequest::Resource* pb_resource,
      const net::URLFetcher* source);

  // Read the body |data| and add it to |pb_resource|.
  void ReadData(
      ClientSafeBrowsingReportRequest::Resource* pb_resource,
      const std::string& data);

  // Called when we are done.
  void AllDone(bool success);

  // Advances to the next entry in resources_it_.
  void AdvanceEntry();
};

}  // namespace safe_browsing

#endif  // CHROME_BROWSER_SAFE_BROWSING_THREAT_DETAILS_CACHE_H_
