// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.webapps;

import android.content.Intent;
import android.text.TextUtils;

import org.chromium.base.Log;
import org.chromium.chrome.browser.ShortcutHelper;
import org.chromium.chrome.browser.util.IntentUtils;

/**
 * Stores info for WebAPK.
 */
public class WebApkInfo extends WebappInfo {
    private static final String TAG = "WebApkInfo";

    private String mWebApkPackageName;

    public static WebApkInfo createEmpty() {
        return new WebApkInfo();
    }

    /**
     * Constructs a WebApkInfo from the passed in Intent and <meta-data> in the WebAPK's Android
     * manifest.
     * @param intent Intent containing info about the app.
     */
    public static WebApkInfo create(Intent intent) {
        String webApkPackageName =
                IntentUtils.safeGetStringExtra(intent, ShortcutHelper.EXTRA_WEBAPK_PACKAGE_NAME);
        if (TextUtils.isEmpty(webApkPackageName)) {
            return null;
        }

        String url = urlFromIntent(intent);
        int source = sourceFromIntent(intent);

        // Unlike non-WebAPK web apps, WebAPK ids are predictable. A malicious actor may send an
        // intent with a valid start URL and arbitrary other data. Only use the start URL, the
        // package name and the ShortcutSource from the launch intent and extract the remaining data
        // from the <meta-data> in the WebAPK's Android manifest.
        return WebApkMetaDataUtils.extractWebappInfoFromWebApk(webApkPackageName, url, source);
    }

    /**
     * Construct a {@link WebApkInfo} instance.
     *
     * @param id                ID for the webapp.
     * @param url               URL for the webapp.
     * @param scope             Scope for the webapp.
     * @param encodedIcon       Icon to show for the webapp.
     * @param name              Name of the webapp.
     * @param shortName         The short name of the webapp.
     * @param displayMode       Display mode of the webapp.
     * @param orientation       Orientation of the webapp.
     * @param source            Source where the webapp was added from.
     * @param themeColor        The theme color of the webapp.
     * @param backgroundColor   The background color of the webapp.
     * @param isIconGenerated   Whether the |icon| was generated by Chromium.
     * @param webApkPackageName The package of the WebAPK associated with the webapp.
     */
    public static WebApkInfo create(String id, String url, String scope, String encodedIcon,
            String name, String shortName, int displayMode, int orientation, int source,
            long themeColor, long backgroundColor, boolean isIconGenerated,
            String webApkPackageName) {
        if (id == null || url == null || webApkPackageName == null) {
            Log.e(TAG, "Incomplete data provided: " + id + ", " + url + ", " + webApkPackageName);
            return null;
        }

        return new WebApkInfo(id, url, scope, encodedIcon, name, shortName, displayMode,
                orientation, source, themeColor, backgroundColor, isIconGenerated,
                webApkPackageName);
    }

    protected WebApkInfo(String id, String url, String scope, String encodedIcon, String name,
            String shortName, int displayMode, int orientation, int source, long themeColor,
            long backgroundColor, boolean isIconGenerated, String webApkPackageName) {
        super(id, url, scope, encodedIcon, name, shortName, displayMode, orientation, source,
                themeColor, backgroundColor, isIconGenerated);
        mWebApkPackageName = webApkPackageName;
    }

    protected WebApkInfo() {}

    @Override
    public String webApkPackageName() {
        return mWebApkPackageName;
    }

    @Override
    public void setWebappIntentExtras(Intent intent) {
        // For launching a {@link WebApkActivity}.
        intent.putExtra(ShortcutHelper.EXTRA_URL, uri().toString());
        intent.putExtra(ShortcutHelper.EXTRA_SOURCE, source());
        intent.putExtra(ShortcutHelper.EXTRA_WEBAPK_PACKAGE_NAME, webApkPackageName());
    }

}
