// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/policy/profile_policy_connector_factory.h"

#include <utility>

#include "base/logging.h"
#include "base/memory/singleton.h"
#include "build/build_config.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/policy/profile_policy_connector.h"
#include "chrome/browser/policy/schema_registry_service.h"
#include "chrome/browser/policy/schema_registry_service_factory.h"
#include "chrome/browser/profiles/incognito_helpers.h"
#include "chrome/browser/profiles/profile.h"
#include "components/keyed_service/content/browser_context_dependency_manager.h"
#include "components/policy/core/common/policy_service.h"
#include "components/policy/core/common/policy_service_impl.h"

#if defined(OS_CHROMEOS)
#include "chrome/browser/chromeos/policy/browser_policy_connector_chromeos.h"
#include "chrome/browser/chromeos/policy/device_cloud_policy_manager_chromeos.h"
#include "chrome/browser/chromeos/policy/user_cloud_policy_manager_chromeos.h"
#include "chrome/browser/chromeos/policy/user_cloud_policy_manager_factory_chromeos.h"
#include "chrome/browser/chromeos/profiles/profile_helper.h"
#include "components/user_manager/user.h"
#else  // Non-ChromeOS.
#include "chrome/browser/policy/cloud/user_cloud_policy_manager_factory.h"
#include "components/policy/core/common/cloud/user_cloud_policy_manager.h"
#endif

namespace policy {

// static
ProfilePolicyConnectorFactory* ProfilePolicyConnectorFactory::GetInstance() {
  return base::Singleton<ProfilePolicyConnectorFactory>::get();
}

// static
ProfilePolicyConnector* ProfilePolicyConnectorFactory::GetForBrowserContext(
    content::BrowserContext* context) {
  return GetInstance()->GetForBrowserContextInternal(context);
}

// static
std::unique_ptr<ProfilePolicyConnector>
ProfilePolicyConnectorFactory::CreateForBrowserContext(
    content::BrowserContext* context,
    bool force_immediate_load) {
  return GetInstance()->CreateForBrowserContextInternal(context,
                                                        force_immediate_load);
}

void ProfilePolicyConnectorFactory::SetServiceForTesting(
    content::BrowserContext* context,
    ProfilePolicyConnector* connector) {
  ProfilePolicyConnector*& map_entry = connectors_[context];
  CHECK(!map_entry);
  map_entry = connector;
}

void ProfilePolicyConnectorFactory::PushProviderForTesting(
    ConfigurationPolicyProvider* provider) {
  test_providers_.push_back(provider);
}

ProfilePolicyConnectorFactory::ProfilePolicyConnectorFactory()
    : BrowserContextKeyedBaseFactory(
        "ProfilePolicyConnector",
        BrowserContextDependencyManager::GetInstance()) {
  DependsOn(SchemaRegistryServiceFactory::GetInstance());
#if defined(OS_CHROMEOS)
  DependsOn(UserCloudPolicyManagerFactoryChromeOS::GetInstance());
#else
  DependsOn(UserCloudPolicyManagerFactory::GetInstance());
#endif
}

ProfilePolicyConnectorFactory::~ProfilePolicyConnectorFactory() {
  DCHECK(connectors_.empty());
}

ProfilePolicyConnector*
ProfilePolicyConnectorFactory::GetForBrowserContextInternal(
    content::BrowserContext* context) {
  // Get the connector for the original Profile, so that the incognito Profile
  // gets managed settings from the same PolicyService.
  content::BrowserContext* const original_context =
      chrome::GetBrowserContextRedirectedInIncognito(context);
  const ConnectorMap::const_iterator it = connectors_.find(original_context);
  CHECK(it != connectors_.end());
  return it->second;
}

std::unique_ptr<ProfilePolicyConnector>
ProfilePolicyConnectorFactory::CreateForBrowserContextInternal(
    content::BrowserContext* context,
    bool force_immediate_load) {
  DCHECK(connectors_.find(context) == connectors_.end());

  SchemaRegistry* schema_registry = nullptr;
  CloudPolicyManager* user_cloud_policy_manager = nullptr;

  schema_registry =
      SchemaRegistryServiceFactory::GetForContext(context)->registry();

#if defined(OS_CHROMEOS)
  Profile* const profile = Profile::FromBrowserContext(context);
  const user_manager::User* user = nullptr;
  if (chromeos::ProfileHelper::IsSigninProfile(profile)) {
    policy::BrowserPolicyConnectorChromeOS* browser_policy_connector =
        g_browser_process->platform_part()->browser_policy_connector_chromeos();
    policy::DeviceCloudPolicyManagerChromeOS* device_cloud_policy_manager =
        browser_policy_connector->GetDeviceCloudPolicyManager();
    device_cloud_policy_manager->SetSigninProfileSchemaRegistry(
        schema_registry);
  } else {
    user = chromeos::ProfileHelper::Get()->GetUserByProfile(profile);
    CHECK(user);
  }
  user_cloud_policy_manager =
      UserCloudPolicyManagerFactoryChromeOS::GetForProfile(profile);
#else
  user_cloud_policy_manager =
      UserCloudPolicyManagerFactory::GetForBrowserContext(context);
#endif  // defined(OS_CHROMEOS)

  std::unique_ptr<ProfilePolicyConnector> connector(
      new ProfilePolicyConnector());

  if (test_providers_.empty()) {
    connector->Init(
#if defined(OS_CHROMEOS)
        user,
#endif
        schema_registry, user_cloud_policy_manager);
  } else {
    PolicyServiceImpl::Providers providers;
    providers.push_back(test_providers_.front());
    test_providers_.pop_front();
    std::unique_ptr<PolicyService> service(new PolicyServiceImpl(providers));
    connector->InitForTesting(std::move(service));
  }

  connectors_[context] = connector.get();
  return connector;
}

void ProfilePolicyConnectorFactory::BrowserContextShutdown(
    content::BrowserContext* context) {
  if (Profile::FromBrowserContext(context)->IsOffTheRecord())
    return;
  const ConnectorMap::const_iterator it = connectors_.find(context);
  if (it != connectors_.end())
    it->second->Shutdown();
}

void ProfilePolicyConnectorFactory::BrowserContextDestroyed(
    content::BrowserContext* context) {
  const ConnectorMap::iterator it = connectors_.find(context);
  if (it != connectors_.end())
    connectors_.erase(it);
  BrowserContextKeyedBaseFactory::BrowserContextDestroyed(context);
}

void ProfilePolicyConnectorFactory::SetEmptyTestingFactory(
    content::BrowserContext* context) {}

bool ProfilePolicyConnectorFactory::HasTestingFactory(
    content::BrowserContext* context) {
  return false;
}

void ProfilePolicyConnectorFactory::CreateServiceNow(
    content::BrowserContext* context) {}

}  // namespace policy
