// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_COMMON_CHROME_CONTENT_CLIENT_H_
#define CHROME_COMMON_CHROME_CONTENT_CLIENT_H_

#include <memory>
#include <set>
#include <string>
#include <vector>

#include "base/compiler_specific.h"
#include "base/files/file_path.h"
#include "build/build_config.h"
#include "chrome/common/origin_trials/chrome_origin_trial_policy.h"
#include "content/public/common/content_client.h"
#include "ppapi/features/features.h"

#if BUILDFLAG(ENABLE_PLUGINS)
#include "content/public/common/pepper_plugin_info.h"
#endif

#include "url/url_util.h"

// Returns the user agent of Chrome.
std::string GetUserAgent();

class ChromeContentClient : public content::ContentClient {
 public:
#if defined(GOOGLE_CHROME_BUILD)
  // kNotPresent is a placeholder plugin location for plugins that are not
  // currently present in this installation of Chrome, but which can be fetched
  // on-demand and therefore should still appear in navigator.plugins.
  static const char kNotPresent[];
#endif
  static const char kPDFPluginName[];
  static const char kPDFPluginPath[];
  static const char kRemotingViewerPluginPath[];

  ChromeContentClient();
  ~ChromeContentClient() override;

  // The methods below are called by child processes to set the function
  // pointers for built-in plugins. We avoid linking these plugins into
  // chrome_common because then on Windows we would ship them twice because of
  // the split DLL.
#if !defined(DISABLE_NACL)
  static void SetNaClEntryFunctions(
      content::PepperPluginInfo::GetInterfaceFunc get_interface,
      content::PepperPluginInfo::PPP_InitializeModuleFunc initialize_module,
      content::PepperPluginInfo::PPP_ShutdownModuleFunc shutdown_module);
#endif

#if BUILDFLAG(ENABLE_PLUGINS)
  static void SetPDFEntryFunctions(
      content::PepperPluginInfo::GetInterfaceFunc get_interface,
      content::PepperPluginInfo::PPP_InitializeModuleFunc initialize_module,
      content::PepperPluginInfo::PPP_ShutdownModuleFunc shutdown_module);

  // This returns the most recent plugin based on the plugin versions. In the
  // event of a tie, a debug plugin will be considered more recent than a
  // non-debug plugin.
  // It does not make sense to call this on a vector that contains more than one
  // plugin type. This function may return a nullptr if given an empty vector.
  // The method is only visible for testing purposes.
  static content::PepperPluginInfo* FindMostRecentPlugin(
      const std::vector<std::unique_ptr<content::PepperPluginInfo>>& plugins);
#endif

  void SetActiveURL(const GURL& url) override;
  void SetGpuInfo(const gpu::GPUInfo& gpu_info) override;
  void AddPepperPlugins(
      std::vector<content::PepperPluginInfo>* plugins) override;
  void AddContentDecryptionModules(
      std::vector<content::CdmInfo>* cdms) override;
  void AddAdditionalSchemes(std::vector<url::SchemeWithType>* standard_schemes,
                            std::vector<url::SchemeWithType>* referrer_schemes,
                            std::vector<std::string>* saveable_shemes) override;
  std::string GetProduct() const override;
  std::string GetUserAgent() const override;
  base::string16 GetLocalizedString(int message_id) const override;
  base::StringPiece GetDataResource(
      int resource_id,
      ui::ScaleFactor scale_factor) const override;
  base::RefCountedMemory* GetDataResourceBytes(
      int resource_id) const override;
  gfx::Image& GetNativeImageNamed(int resource_id) const override;
  std::string GetProcessTypeNameInEnglish(int type) override;

#if defined(OS_MACOSX)
  bool GetSandboxProfileForSandboxType(
      int sandbox_type,
      int* sandbox_profile_resource_id) const override;
#endif

  void AddSecureSchemesAndOrigins(std::set<std::string>* schemes,
                                  std::set<GURL>* origins) override;

  void AddServiceWorkerSchemes(std::set<std::string>* schemes) override;
  bool AllowScriptExtensionForServiceWorker(const GURL& script_url) override;

  bool IsSupplementarySiteIsolationModeEnabled() override;

  content::OriginTrialPolicy* GetOriginTrialPolicy() override;

#if defined(OS_ANDROID)
  media::MediaClientAndroid* GetMediaClientAndroid() override;
#endif  // OS_ANDROID

 private:
  std::unique_ptr<ChromeOriginTrialPolicy> origin_trial_policy_;
};

#endif  // CHROME_COMMON_CHROME_CONTENT_CLIENT_H_
