// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/files/file_util.h"
#include "base/files/scoped_temp_dir.h"
#include "base/message_loop/message_loop.h"
#include "base/run_loop.h"
#include "base/strings/string_number_conversions.h"
#include "base/strings/string_util.h"

#include "device/generic_sensor/generic_sensor_consts.h"
#include "device/generic_sensor/linux/sensor_data_linux.h"
#include "device/generic_sensor/linux/sensor_device_manager.h"
#include "device/generic_sensor/platform_sensor_provider_linux.h"

#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"

using ::testing::_;
using ::testing::Invoke;
using ::testing::IsNull;
using ::testing::NiceMock;
using ::testing::NotNull;
using ::testing::Return;

namespace device {

namespace {

using mojom::SensorType;

// Zero value can mean whether value is not being not used or zero value.
constexpr double kZero = 0.0;

constexpr double kAccelerometerFrequencyValue = 10.0;
constexpr double kAccelerometerOffsetValue = 1.0;
constexpr double kAccelerometerScalingValue = 0.009806;

constexpr double kGyroscopeFrequencyValue = 6.0;
constexpr double kGyroscopeOffsetValue = 2.0;
constexpr double kGyroscopeScalingValue = 0.000017;

constexpr double kMagnetometerFrequencyValue = 7.0;
constexpr double kMagnetometerOffsetValue = 3.0;
constexpr double kMagnetometerScalingValue = 0.000001;

void DeleteFile(const base::FilePath& file) {
  EXPECT_TRUE(base::DeleteFile(file, true));
}

void WriteValueToFile(const base::FilePath& path, double value) {
  const std::string str = base::DoubleToString(value);
  int bytes_written = base::WriteFile(path, str.data(), str.size());
  EXPECT_EQ(static_cast<size_t>(bytes_written), str.size());
}

std::string ReadValueFromFile(const base::FilePath& path,
                              const std::string& file) {
  base::FilePath file_path = base::FilePath(path).Append(file);
  std::string new_read_value;
  if (!base::ReadFileToString(file_path, &new_read_value))
    return std::string();
  return new_read_value;
}

}  // namespace

// Mock for SensorDeviceService that SensorDeviceManager owns.
// This mock is used to emulate udev events and send found sensor devices
// to SensorDeviceManager.
class MockSensorDeviceManager : public SensorDeviceManager {
 public:
  MockSensorDeviceManager() = default;
  ~MockSensorDeviceManager() override {}

  MOCK_METHOD1(GetUdevDeviceGetSubsystem, std::string(udev_device*));
  MOCK_METHOD1(GetUdevDeviceGetSyspath, std::string(udev_device*));
  MOCK_METHOD1(GetUdevDeviceGetDevnode, std::string(udev_device* dev));
  MOCK_METHOD2(GetUdevDeviceGetSysattrValue,
               std::string(udev_device*, const std::string&));
  MOCK_METHOD1(Start, void(Delegate*));

  void InitializeService(Delegate* delegate) { delegate_ = delegate; }

  void EnumerationReady() {
    bool success = task_runner_->PostTask(
        FROM_HERE,
        base::Bind(&SensorDeviceManager::Delegate::OnSensorNodesEnumerated,
                   base::Unretained(delegate_)));
    ASSERT_TRUE(success);
  }

  void DeviceAdded(udev_device* dev) {
    SensorDeviceManager::OnDeviceAdded(dev);
  }

  void DeviceRemoved(udev_device* dev) {
    SensorDeviceManager::OnDeviceRemoved(dev);
  }

 private:
  DISALLOW_COPY_AND_ASSIGN(MockSensorDeviceManager);
};

// Mock for PlatformSensor's client interface that is used to deliver
// error and data changes notifications.
class MockPlatformSensorClient : public PlatformSensor::Client {
 public:
  MockPlatformSensorClient() = default;
  explicit MockPlatformSensorClient(scoped_refptr<PlatformSensor> sensor)
      : sensor_(sensor) {
    if (sensor_)
      sensor_->AddClient(this);

    ON_CALL(*this, IsNotificationSuspended()).WillByDefault(Return(false));
  }

  ~MockPlatformSensorClient() override {
    if (sensor_)
      sensor_->RemoveClient(this);
  }

  // PlatformSensor::Client interface.
  MOCK_METHOD0(OnSensorReadingChanged, void());
  MOCK_METHOD0(OnSensorError, void());
  MOCK_METHOD0(IsNotificationSuspended, bool());

 private:
  scoped_refptr<PlatformSensor> sensor_;

  DISALLOW_COPY_AND_ASSIGN(MockPlatformSensorClient);
};

class PlatformSensorAndProviderLinuxTest : public ::testing::Test {
 public:
  void SetUp() override {
    provider_ = PlatformSensorProviderLinux::GetInstance();
    provider_->SetFileTaskRunnerForTesting(message_loop_.task_runner());

    auto manager = base::MakeUnique<NiceMock<MockSensorDeviceManager>>();
    manager_ = manager.get();
    provider_->SetSensorDeviceManagerForTesting(std::move(manager));

    ASSERT_TRUE(sensors_dir_.CreateUniqueTempDir());
  }

  void TearDown() override {
    provider_->SetSensorDeviceManagerForTesting(nullptr);
    ASSERT_TRUE(sensors_dir_.Delete());
    base::RunLoop().RunUntilIdle();
  }

 protected:
  void SensorsCreated(scoped_refptr<PlatformSensor> sensor) {
    platform_sensor_vector_.push_back(sensor);
  }

  void SensorCreated(scoped_refptr<PlatformSensor> sensor) {
    platform_sensor_ = sensor;
    run_loop_->Quit();
  }

  // Sensor creation is asynchronous, therefore inner loop is used to wait for
  // PlatformSensorProvider::CreateSensorCallback completion.
  scoped_refptr<PlatformSensor> CreateSensor(mojom::SensorType type) {
    run_loop_ = base::MakeUnique<base::RunLoop>();
    provider_->CreateSensor(
        type, base::Bind(&PlatformSensorAndProviderLinuxTest::SensorCreated,
                         base::Unretained(this)));
    run_loop_->Run();
    scoped_refptr<PlatformSensor> sensor;
    sensor.swap(platform_sensor_);
    run_loop_ = nullptr;
    return sensor;
  }

  // Creates sensor files according to SensorPathsLinux.
  // Existence of sensor read files mean existence of a sensor.
  void InitializeSupportedSensor(SensorType type,
                                 double frequency,
                                 double offset,
                                 double scaling,
                                 double values[3]) {
    SensorPathsLinux data;
    EXPECT_TRUE(InitSensorData(type, &data));

    base::FilePath sensor_dir = sensors_dir_.GetPath();
    if (!data.sensor_scale_name.empty()) {
      base::FilePath sensor_scale_file =
          base::FilePath(sensor_dir).Append(data.sensor_scale_name);
      WriteValueToFile(sensor_scale_file, scaling);
    }

    if (!data.sensor_offset_file_name.empty()) {
      base::FilePath sensor_offset_file =
          base::FilePath(sensor_dir).Append(data.sensor_offset_file_name);
      WriteValueToFile(sensor_offset_file, offset);
    }

    if (!data.sensor_frequency_file_name.empty()) {
      base::FilePath sensor_frequency_file =
          base::FilePath(sensor_dir).Append(data.sensor_frequency_file_name);
      WriteValueToFile(sensor_frequency_file, frequency);
    }

    uint32_t i = 0;
    for (const auto& file_names : data.sensor_file_names) {
      for (const auto& name : file_names) {
        base::FilePath sensor_file = base::FilePath(sensor_dir).Append(name);
        WriteValueToFile(sensor_file, values[i++]);
        break;
      }
    }
  }

  // Initializes mock udev methods that emulate system methods by
  // just reading values from files, which SensorDeviceService has specified
  // calling udev methods.
  void InitializeMockUdevMethods(const base::FilePath& sensor_dir) {
    ON_CALL(*manager_, GetUdevDeviceGetSubsystem(IsNull()))
        .WillByDefault(Invoke([this](udev_device* dev) { return "iio"; }));

    ON_CALL(*manager_, GetUdevDeviceGetSyspath(IsNull()))
        .WillByDefault(Invoke(
            [sensor_dir](udev_device* dev) { return sensor_dir.value(); }));

    ON_CALL(*manager_, GetUdevDeviceGetDevnode(IsNull()))
        .WillByDefault(
            Invoke([this](udev_device* dev) { return "/dev/test"; }));

    ON_CALL(*manager_, GetUdevDeviceGetSysattrValue(IsNull(), _))
        .WillByDefault(Invoke(
            [sensor_dir](udev_device* dev, const std::string& attribute) {
              return ReadValueFromFile(sensor_dir, attribute);
            }));
  }

  // Emulates device enumerations and initial udev events. Once all
  // devices are added, tells manager its ready.
  void SetServiceStart() {
    EXPECT_CALL(*manager_, Start(NotNull()))
        .WillOnce(Invoke([this](SensorDeviceManager::Delegate* delegate) {
          manager_->InitializeService(delegate);
          udev_device* dev = nullptr;
          manager_->DeviceAdded(dev /* not used */);
          manager_->EnumerationReady();
        }));
  }

  // Waits before OnSensorReadingChanged is called.
  void WaitOnSensorReadingChangedEvent(MockPlatformSensorClient* client) {
    run_loop_ = base::MakeUnique<base::RunLoop>();
    EXPECT_CALL(*client, OnSensorReadingChanged()).WillOnce(Invoke([this]() {
      run_loop_->Quit();
    }));
    run_loop_->Run();
    run_loop_ = nullptr;
  }

  // Waits before OnSensorError is called.
  void WaitOnSensorErrorEvent(MockPlatformSensorClient* client) {
    run_loop_ = base::MakeUnique<base::RunLoop>();
    EXPECT_CALL(*client, OnSensorError()).WillOnce(Invoke([this]() {
      run_loop_->Quit();
    }));
    run_loop_->Run();
    run_loop_ = nullptr;
  }

  // Generates a "remove device" event by removed sensors' directory and
  // notifies the mock service about "removed" event.
  void GenerateDeviceRemovedEvent(const base::FilePath& sensor_dir) {
    udev_device* dev = nullptr;
    DeleteFile(sensor_dir);
    bool success = base::ThreadTaskRunnerHandle::Get()->PostTask(
        FROM_HERE, base::Bind(&MockSensorDeviceManager::DeviceRemoved,
                              base::Unretained(manager_), dev /* not used */));
    ASSERT_TRUE(success);
  }

  MockSensorDeviceManager* manager_;
  scoped_refptr<PlatformSensor> platform_sensor_;
  std::vector<scoped_refptr<PlatformSensor>> platform_sensor_vector_;
  base::MessageLoop message_loop_;
  std::unique_ptr<base::RunLoop> run_loop_;
  PlatformSensorProviderLinux* provider_;
  // Holds base dir where a sensor dir is located.
  base::ScopedTempDir sensors_dir_;
};

// Tests sensor is not returned if not implemented.
TEST_F(PlatformSensorAndProviderLinuxTest, SensorIsNotImplemented) {
  double sensor_value[3] = {5};
  InitializeSupportedSensor(SensorType::AMBIENT_LIGHT, kZero, kZero, kZero,
                            sensor_value);
  SetServiceStart();
  EXPECT_FALSE(CreateSensor(SensorType::PROXIMITY));
}

// Tests sensor is not returned if not supported by hardware.
TEST_F(PlatformSensorAndProviderLinuxTest, SensorIsNotSupported) {
  double sensor_value[3] = {5};
  InitializeSupportedSensor(SensorType::AMBIENT_LIGHT, kZero, kZero, kZero,
                            sensor_value);
  SetServiceStart();
  EXPECT_FALSE(CreateSensor(SensorType::ACCELEROMETER));
}

// Tests sensor is returned if supported.
TEST_F(PlatformSensorAndProviderLinuxTest, SensorIsSupported) {
  double sensor_value[3] = {5};
  InitializeSupportedSensor(SensorType::AMBIENT_LIGHT, kZero, kZero, kZero,
                            sensor_value);
  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto sensor = CreateSensor(SensorType::AMBIENT_LIGHT);
  EXPECT_TRUE(sensor);
  EXPECT_EQ(SensorType::AMBIENT_LIGHT, sensor->GetType());
}

// Tests that PlatformSensor::StartListening fails when provided reporting
// frequency is above hardware capabilities.
TEST_F(PlatformSensorAndProviderLinuxTest, StartFails) {
  double sensor_value[3] = {5};
  InitializeSupportedSensor(SensorType::AMBIENT_LIGHT, kZero, kZero, kZero,
                            sensor_value);
  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto sensor = CreateSensor(SensorType::AMBIENT_LIGHT);
  EXPECT_TRUE(sensor);

  auto client = base::MakeUnique<NiceMock<MockPlatformSensorClient>>(sensor);
  PlatformSensorConfiguration configuration(10);
  EXPECT_FALSE(sensor->StartListening(client.get(), configuration));
}

// Tests that PlatformSensor::StartListening succeeds and notification about
// modified sensor reading is sent to the PlatformSensor::Client interface.
TEST_F(PlatformSensorAndProviderLinuxTest, SensorStarted) {
  double sensor_value[3] = {5};
  InitializeSupportedSensor(SensorType::AMBIENT_LIGHT, kZero, kZero, kZero,
                            sensor_value);
  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto sensor = CreateSensor(SensorType::AMBIENT_LIGHT);
  EXPECT_TRUE(sensor);

  auto client = base::MakeUnique<NiceMock<MockPlatformSensorClient>>(sensor);
  PlatformSensorConfiguration configuration(5);
  EXPECT_TRUE(sensor->StartListening(client.get(), configuration));
  WaitOnSensorReadingChangedEvent(client.get());
  EXPECT_TRUE(sensor->StopListening(client.get(), configuration));
}

// Tests that OnSensorError is called when sensor is disconnected.
TEST_F(PlatformSensorAndProviderLinuxTest, SensorRemoved) {
  double sensor_value[3] = {1};
  InitializeSupportedSensor(SensorType::AMBIENT_LIGHT, kZero, kZero, kZero,
                            sensor_value);
  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto sensor = CreateSensor(SensorType::AMBIENT_LIGHT);
  EXPECT_TRUE(sensor);

  auto client = base::MakeUnique<NiceMock<MockPlatformSensorClient>>(sensor);
  PlatformSensorConfiguration configuration(5);
  EXPECT_TRUE(sensor->StartListening(client.get(), configuration));
  GenerateDeviceRemovedEvent(sensors_dir_.GetPath());
  WaitOnSensorErrorEvent(client.get());
}

// Tests that sensor is not returned if not connected and
// is created after it has been added.
TEST_F(PlatformSensorAndProviderLinuxTest, SensorAddedAndRemoved) {
  double sensor_value[3] = {1, 2, 4};
  InitializeSupportedSensor(SensorType::AMBIENT_LIGHT, kZero, kZero, kZero,
                            sensor_value);
  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto als_sensor = CreateSensor(SensorType::AMBIENT_LIGHT);
  EXPECT_TRUE(als_sensor);
  auto gyro_sensor = CreateSensor(SensorType::GYROSCOPE);
  EXPECT_FALSE(gyro_sensor);

  InitializeSupportedSensor(SensorType::GYROSCOPE, kGyroscopeFrequencyValue,
                            kGyroscopeOffsetValue, kGyroscopeScalingValue,
                            sensor_value);
  udev_device* dev = nullptr;
  manager_->DeviceAdded(dev /* not used */);
  base::RunLoop().RunUntilIdle();
  gyro_sensor = CreateSensor(SensorType::GYROSCOPE);
  EXPECT_TRUE(gyro_sensor);
  EXPECT_EQ(gyro_sensor->GetType(), SensorType::GYROSCOPE);
}

// Checks the main fields of all sensors and initialized right.
TEST_F(PlatformSensorAndProviderLinuxTest, CheckAllSupportedSensors) {
  double sensor_value[3] = {1, 2, 3};
  InitializeSupportedSensor(SensorType::AMBIENT_LIGHT, kZero, kZero, kZero,
                            sensor_value);
  InitializeSupportedSensor(
      SensorType::ACCELEROMETER, kAccelerometerFrequencyValue,
      kAccelerometerOffsetValue, kAccelerometerScalingValue, sensor_value);
  InitializeSupportedSensor(SensorType::GYROSCOPE, kGyroscopeFrequencyValue,
                            kGyroscopeOffsetValue, kGyroscopeScalingValue,
                            sensor_value);
  InitializeSupportedSensor(
      SensorType::MAGNETOMETER, kMagnetometerFrequencyValue,
      kMagnetometerOffsetValue, kMagnetometerScalingValue, sensor_value);
  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto als_sensor = CreateSensor(SensorType::AMBIENT_LIGHT);
  EXPECT_TRUE(als_sensor);
  EXPECT_EQ(als_sensor->GetType(), SensorType::AMBIENT_LIGHT);
  EXPECT_THAT(als_sensor->GetDefaultConfiguration().frequency(),
              kDefaultAmbientLightFrequencyHz);

  auto accel_sensor = CreateSensor(SensorType::ACCELEROMETER);
  EXPECT_TRUE(accel_sensor);
  EXPECT_EQ(accel_sensor->GetType(), SensorType::ACCELEROMETER);
  EXPECT_THAT(accel_sensor->GetDefaultConfiguration().frequency(),
              kAccelerometerFrequencyValue);

  auto gyro_sensor = CreateSensor(SensorType::GYROSCOPE);
  EXPECT_TRUE(gyro_sensor);
  EXPECT_EQ(gyro_sensor->GetType(), SensorType::GYROSCOPE);
  EXPECT_THAT(gyro_sensor->GetDefaultConfiguration().frequency(),
              kGyroscopeFrequencyValue);

  auto magn_sensor = CreateSensor(SensorType::MAGNETOMETER);
  EXPECT_TRUE(magn_sensor);
  EXPECT_EQ(magn_sensor->GetType(), SensorType::MAGNETOMETER);
  EXPECT_THAT(magn_sensor->GetDefaultConfiguration().frequency(),
              kMagnetometerFrequencyValue);
}

// Tests that GetMaximumSupportedFrequency provides correct value.
TEST_F(PlatformSensorAndProviderLinuxTest, GetMaximumSupportedFrequency) {
  double sensor_value[3] = {5};
  InitializeSupportedSensor(
      SensorType::ACCELEROMETER, kAccelerometerFrequencyValue,
      kAccelerometerOffsetValue, kAccelerometerScalingValue, sensor_value);
  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto sensor = CreateSensor(SensorType::ACCELEROMETER);
  EXPECT_TRUE(sensor);
  EXPECT_THAT(sensor->GetMaximumSupportedFrequency(),
              kAccelerometerFrequencyValue);
}

// Tests that GetMaximumSupportedFrequency provides correct value when
// OS does not provide any information about frequency.
TEST_F(PlatformSensorAndProviderLinuxTest,
       GetMaximumSupportedFrequencyDefault) {
  double sensor_value[3] = {5};
  InitializeSupportedSensor(SensorType::AMBIENT_LIGHT, kZero, kZero, kZero,
                            sensor_value);
  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto sensor = CreateSensor(SensorType::AMBIENT_LIGHT);
  EXPECT_TRUE(sensor);
  EXPECT_EQ(SensorType::AMBIENT_LIGHT, sensor->GetType());
  EXPECT_THAT(sensor->GetMaximumSupportedFrequency(),
              kDefaultAmbientLightFrequencyHz);
}

// Tests that Ambient Light sensor is correctly read.
TEST_F(PlatformSensorAndProviderLinuxTest, CheckAmbientLightReadings) {
  mojo::ScopedSharedBufferHandle handle = provider_->CloneSharedBufferHandle();
  mojo::ScopedSharedBufferMapping mapping = handle->MapAtOffset(
      sizeof(SensorReadingSharedBuffer),
      SensorReadingSharedBuffer::GetOffset(SensorType::AMBIENT_LIGHT));

  double sensor_value[3] = {22};
  InitializeSupportedSensor(SensorType::AMBIENT_LIGHT, kZero, kZero, kZero,
                            sensor_value);

  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto sensor = CreateSensor(SensorType::AMBIENT_LIGHT);
  EXPECT_TRUE(sensor);
  EXPECT_EQ(sensor->GetReportingMode(), mojom::ReportingMode::ON_CHANGE);

  auto client = base::MakeUnique<NiceMock<MockPlatformSensorClient>>(sensor);
  PlatformSensorConfiguration configuration(
      sensor->GetMaximumSupportedFrequency());
  EXPECT_TRUE(sensor->StartListening(client.get(), configuration));
  WaitOnSensorReadingChangedEvent(client.get());
  EXPECT_TRUE(sensor->StopListening(client.get(), configuration));

  SensorReadingSharedBuffer* buffer =
      static_cast<SensorReadingSharedBuffer*>(mapping.get());
  EXPECT_THAT(buffer->reading.values[0], sensor_value[0]);
}

// Tests that Accelerometer readings are correctly converted.
TEST_F(PlatformSensorAndProviderLinuxTest,
       CheckAccelerometerReadingConversion) {
  mojo::ScopedSharedBufferHandle handle = provider_->CloneSharedBufferHandle();
  mojo::ScopedSharedBufferMapping mapping = handle->MapAtOffset(
      sizeof(SensorReadingSharedBuffer),
      SensorReadingSharedBuffer::GetOffset(SensorType::ACCELEROMETER));

  double sensor_values[3] = {4.5, -2.45, -3.29};
  InitializeSupportedSensor(
      SensorType::ACCELEROMETER, kAccelerometerFrequencyValue,
      kAccelerometerOffsetValue, kAccelerometerScalingValue, sensor_values);

  // As long as WaitOnSensorReadingChangedEvent() waits until client gets a
  // a notification about readings changed, the frequency file must be deleted
  // to make the sensor device manager identify this sensor with ON_CHANGE
  // reporting mode. This will allow the MockPlatformSensorClient to
  // receive a notification and test if reading values are right. Otherwise
  // the test will not know when data is ready.
  SensorPathsLinux data;
  EXPECT_TRUE(InitSensorData(SensorType::ACCELEROMETER, &data));
  base::FilePath frequency_file = base::FilePath(sensors_dir_.GetPath())
                                      .Append(data.sensor_frequency_file_name);
  DeleteFile(frequency_file);

  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto sensor = CreateSensor(SensorType::ACCELEROMETER);
  EXPECT_TRUE(sensor);
  // The reporting mode is ON_CHANGE only for this test.
  EXPECT_EQ(sensor->GetReportingMode(), mojom::ReportingMode::ON_CHANGE);

  auto client = base::MakeUnique<NiceMock<MockPlatformSensorClient>>(sensor);
  PlatformSensorConfiguration configuration(10);
  EXPECT_TRUE(sensor->StartListening(client.get(), configuration));
  WaitOnSensorReadingChangedEvent(client.get());
  EXPECT_TRUE(sensor->StopListening(client.get(), configuration));

  SensorReadingSharedBuffer* buffer =
      static_cast<SensorReadingSharedBuffer*>(mapping.get());
#if defined(OS_CHROMEOS)
  double scaling = kMeanGravity / kAccelerometerScalingValue;
  EXPECT_THAT(buffer->reading.values[0], scaling * sensor_values[0]);
  EXPECT_THAT(buffer->reading.values[1], scaling * sensor_values[1]);
  EXPECT_THAT(buffer->reading.values[2], scaling * sensor_values[2]);
#else
  double scaling = kAccelerometerScalingValue + kAccelerometerOffsetValue;
  EXPECT_THAT(buffer->reading.values[0], -scaling * sensor_values[0]);
  EXPECT_THAT(buffer->reading.values[1], -scaling * sensor_values[1]);
  EXPECT_THAT(buffer->reading.values[2], -scaling * sensor_values[2]);
#endif
}

// Tests that Gyroscope readings are correctly converted.
TEST_F(PlatformSensorAndProviderLinuxTest, CheckGyroscopeReadingConversion) {
  mojo::ScopedSharedBufferHandle handle = provider_->CloneSharedBufferHandle();
  mojo::ScopedSharedBufferMapping mapping = handle->MapAtOffset(
      sizeof(SensorReadingSharedBuffer),
      SensorReadingSharedBuffer::GetOffset(SensorType::GYROSCOPE));

  double sensor_values[3] = {2.2, -3.8, -108.7};
  InitializeSupportedSensor(SensorType::GYROSCOPE, kGyroscopeFrequencyValue,
                            kGyroscopeOffsetValue, kGyroscopeScalingValue,
                            sensor_values);

  // As long as WaitOnSensorReadingChangedEvent() waits until client gets a
  // a notification about readings changed, the frequency file must be deleted
  // to make the sensor device manager identify this sensor with ON_CHANGE
  // reporting mode. This will allow the MockPlatformSensorClient to
  // receive a notification and test if reading values are right. Otherwise
  // the test will not know when data is ready.
  SensorPathsLinux data;
  EXPECT_TRUE(InitSensorData(SensorType::GYROSCOPE, &data));
  base::FilePath frequency_file = base::FilePath(sensors_dir_.GetPath())
                                      .Append(data.sensor_frequency_file_name);
  DeleteFile(frequency_file);

  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto sensor = CreateSensor(SensorType::GYROSCOPE);
  EXPECT_TRUE(sensor);
  // The reporting mode is ON_CHANGE only for this test.
  EXPECT_EQ(sensor->GetReportingMode(), mojom::ReportingMode::ON_CHANGE);

  auto client = base::MakeUnique<NiceMock<MockPlatformSensorClient>>(sensor);
  PlatformSensorConfiguration configuration(10);
  EXPECT_TRUE(sensor->StartListening(client.get(), configuration));
  WaitOnSensorReadingChangedEvent(client.get());
  EXPECT_TRUE(sensor->StopListening(client.get(), configuration));

  SensorReadingSharedBuffer* buffer =
      static_cast<SensorReadingSharedBuffer*>(mapping.get());
#if defined(OS_CHROMEOS)
  double scaling =
      kMeanGravity * kRadiansInDegreesPerSecond / kGyroscopeScalingValue;
  EXPECT_THAT(buffer->reading.values[0], -scaling * sensor_values[0]);
  EXPECT_THAT(buffer->reading.values[1], -scaling * sensor_values[1]);
  EXPECT_THAT(buffer->reading.values[2], -scaling * sensor_values[2]);
#else
  double scaling = kGyroscopeScalingValue + kGyroscopeOffsetValue;
  EXPECT_THAT(buffer->reading.values[0], scaling * sensor_values[0]);
  EXPECT_THAT(buffer->reading.values[1], scaling * sensor_values[1]);
  EXPECT_THAT(buffer->reading.values[2], scaling * sensor_values[2]);
#endif
}

// Tests that Magnetometer readings are correctly converted.
TEST_F(PlatformSensorAndProviderLinuxTest, CheckMagnetometerReadingConversion) {
  mojo::ScopedSharedBufferHandle handle = provider_->CloneSharedBufferHandle();
  mojo::ScopedSharedBufferMapping mapping = handle->MapAtOffset(
      sizeof(SensorReadingSharedBuffer),
      SensorReadingSharedBuffer::GetOffset(SensorType::MAGNETOMETER));

  double sensor_values[3] = {2.2, -3.8, -108.7};
  InitializeSupportedSensor(
      SensorType::MAGNETOMETER, kMagnetometerFrequencyValue,
      kMagnetometerOffsetValue, kMagnetometerScalingValue, sensor_values);

  // As long as WaitOnSensorReadingChangedEvent() waits until client gets a
  // a notification about readings changed, the frequency file must be deleted
  // to make the sensor device manager identify this sensor with ON_CHANGE
  // reporting mode. This will allow the MockPlatformSensorClient to
  // receive a notification and test if reading values are right. Otherwise
  // the test will not know when data is ready.
  SensorPathsLinux data;
  EXPECT_TRUE(InitSensorData(SensorType::MAGNETOMETER, &data));
  base::FilePath frequency_file = base::FilePath(sensors_dir_.GetPath())
                                      .Append(data.sensor_frequency_file_name);
  DeleteFile(frequency_file);

  InitializeMockUdevMethods(sensors_dir_.GetPath());
  SetServiceStart();

  auto sensor = CreateSensor(SensorType::MAGNETOMETER);
  EXPECT_TRUE(sensor);
  // The reporting mode is ON_CHANGE only for this test.
  EXPECT_EQ(sensor->GetReportingMode(), mojom::ReportingMode::ON_CHANGE);

  auto client = base::MakeUnique<NiceMock<MockPlatformSensorClient>>(sensor);
  PlatformSensorConfiguration configuration(10);
  EXPECT_TRUE(sensor->StartListening(client.get(), configuration));
  WaitOnSensorReadingChangedEvent(client.get());
  EXPECT_TRUE(sensor->StopListening(client.get(), configuration));

  SensorReadingSharedBuffer* buffer =
      static_cast<SensorReadingSharedBuffer*>(mapping.get());
  double scaling = (kMagnetometerScalingValue + kMagnetometerOffsetValue) *
                   kMicroteslaInGauss;
  EXPECT_THAT(buffer->reading.values[0], scaling * sensor_values[0]);
  EXPECT_THAT(buffer->reading.values[1], scaling * sensor_values[1]);
  EXPECT_THAT(buffer->reading.values[2], scaling * sensor_values[2]);
}

}  // namespace device
