// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.download;

import android.annotation.TargetApi;
import android.app.DownloadManager;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Rect;
import android.graphics.drawable.shapes.OvalShape;
import android.os.Binder;
import android.os.Build;
import android.os.Bundle;
import android.os.IBinder;
import android.service.notification.StatusBarNotification;
import android.text.TextUtils;
import android.util.Pair;

import org.chromium.base.ApiCompatibilityUtils;
import org.chromium.base.ApplicationStatus;
import org.chromium.base.ContextUtils;
import org.chromium.base.Log;
import org.chromium.base.ObserverList;
import org.chromium.base.VisibleForTesting;
import org.chromium.base.library_loader.LibraryLoader;
import org.chromium.base.library_loader.ProcessInitException;
import org.chromium.chrome.R;
import org.chromium.chrome.browser.AppHooks;
import org.chromium.chrome.browser.ChromeApplication;
import org.chromium.chrome.browser.init.BrowserParts;
import org.chromium.chrome.browser.init.ChromeBrowserInitializer;
import org.chromium.chrome.browser.init.EmptyBrowserParts;
import org.chromium.chrome.browser.notifications.ChromeNotificationBuilder;
import org.chromium.chrome.browser.notifications.NotificationConstants;
import org.chromium.chrome.browser.notifications.NotificationUmaTracker;
import org.chromium.chrome.browser.offlinepages.downloads.OfflinePageDownloadBridge;
import org.chromium.chrome.browser.profiles.Profile;
import org.chromium.chrome.browser.util.IntentUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * Service responsible for creating and updating download notifications even after
 * Chrome gets killed.
 *
 * On O and above, this service will receive {@link Service#startForeground(int, Notification)}
 * calls when containing active downloads.  The foreground notification will be the summary
 * notification generated by {@link DownloadNotificationService#buildSummaryNotification(Context)}.
 * The service will receive a {@link Service#stopForeground(boolean)} call when all active downloads
 * are paused.  The summary notification will be hidden when there are no other notifications in the
 * {@link NotificationConstants#GROUP_DOWNLOADS} group.  This gets checked after every notification
 * gets removed from the {@link NotificationManager}.
 */
public class DownloadNotificationService extends Service {
    static final String EXTRA_DOWNLOAD_GUID = "DownloadGuid";
    static final String EXTRA_DOWNLOAD_FILE_PATH = "DownloadFilePath";
    static final String EXTRA_NOTIFICATION_DISMISSED = "NotificationDismissed";
    static final String EXTRA_IS_SUPPORTED_MIME_TYPE = "IsSupportedMimeType";
    static final String EXTRA_IS_OFF_THE_RECORD =
            "org.chromium.chrome.browser.download.IS_OFF_THE_RECORD";
    static final String EXTRA_IS_OFFLINE_PAGE =
            "org.chromium.chrome.browser.download.IS_OFFLINE_PAGE";

    public static final String ACTION_DOWNLOAD_CANCEL =
            "org.chromium.chrome.browser.download.DOWNLOAD_CANCEL";
    public static final String ACTION_DOWNLOAD_PAUSE =
            "org.chromium.chrome.browser.download.DOWNLOAD_PAUSE";
    public static final String ACTION_DOWNLOAD_RESUME =
            "org.chromium.chrome.browser.download.DOWNLOAD_RESUME";
    static final String ACTION_DOWNLOAD_RESUME_ALL =
            "org.chromium.chrome.browser.download.DOWNLOAD_RESUME_ALL";
    public static final String ACTION_DOWNLOAD_OPEN =
            "org.chromium.chrome.browser.download.DOWNLOAD_OPEN";
    public static final String ACTION_DOWNLOAD_UPDATE_SUMMARY_ICON =
            "org.chromium.chrome.browser.download.DOWNLOAD_UPDATE_SUMMARY_ICON";

    static final String NOTIFICATION_NAMESPACE = "DownloadNotificationService";
    private static final String TAG = "DownloadNotification";
    // Limit file name to 25 characters. TODO(qinmin): use different limit for different devices?
    private static final int MAX_FILE_NAME_LENGTH = 25;

    /** Notification Id starting value, to avoid conflicts from IDs used in prior versions. */

    private static final String EXTRA_NOTIFICATION_BUNDLE_ICON_ID =
            "Chrome.NotificationBundleIconIdExtra";
    private static final int STARTING_NOTIFICATION_ID = 1000000;
    private static final int MAX_RESUMPTION_ATTEMPT_LEFT = 5;
    @VisibleForTesting static final long SECONDS_PER_MINUTE = TimeUnit.MINUTES.toSeconds(1);
    @VisibleForTesting static final long SECONDS_PER_HOUR = TimeUnit.HOURS.toSeconds(1);
    @VisibleForTesting static final long SECONDS_PER_DAY = TimeUnit.DAYS.toSeconds(1);

    private static final String KEY_AUTO_RESUMPTION_ATTEMPT_LEFT = "ResumptionAttemptLeft";
    private static final String KEY_NEXT_DOWNLOAD_NOTIFICATION_ID = "NextDownloadNotificationId";

    /**
     * An Observer interface that allows other classes to know when this class wants to shut itself
     * down.  This lets them unbind if necessary.
     */
    public interface Observer {
        /** Called when this service is about to start attempting to stop itself. */
        void onServiceShutdownRequested();

        /**
         * Called when a download was canceled from the notification.  The implementer is not
         * responsible for canceling the actual download (that should be triggered internally from
         * this class).  The implementer is responsible for using this to do their own tracking
         * related to which downloads might be active in this service.  File downloads don't trigger
         * a cancel event when they are told to cancel downloads, so classes might have no idea that
         * a download stopped otherwise.
         * @param guid The guid of the download that was canceled.
         */
        void onDownloadCanceled(String guid);
    }

    private final ObserverList<Observer> mObservers = new ObserverList<>();
    private final IBinder mBinder = new LocalBinder();
    private final List<String> mDownloadsInProgress = new ArrayList<String>();

    private NotificationManager mNotificationManager;
    private SharedPreferences mSharedPrefs;
    private Context mContext;
    private int mNextNotificationId;
    private int mNumAutoResumptionAttemptLeft;
    private Bitmap mDownloadSuccessLargeIcon;
    private DownloadSharedPreferenceHelper mDownloadSharedPreferenceHelper;

    /**
     * @return Whether or not this service should be made a foreground service if there are active
     * downloads.
     */
    @VisibleForTesting
    static boolean useForegroundService() {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.N;
    }

    /**
     * Start this service with a summary {@link Notification}.  This will start the service in the
     * foreground.
     * @param context The context used to build the notification and to start the service.
     * @param source The {@link Intent} that should be used to build on to start the service.
     */
    public static void startDownloadNotificationService(Context context, Intent source) {
        Intent intent = source != null ? new Intent(source) : new Intent();
        intent.setComponent(new ComponentName(context, DownloadNotificationService.class));

        if (useForegroundService()) {
            NotificationManager manager =
                    (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
            // Attempt to update the notification summary icon without starting the service.
            if (ACTION_DOWNLOAD_UPDATE_SUMMARY_ICON.equals(intent.getAction())) {
                // updateSummaryIcon should be a noop if the notification isn't showing or if the
                // icon won't change anyway.
                updateSummaryIcon(context, manager, -1, null);
                return;
            }

            AppHooks.get().startServiceWithNotification(intent,
                    NotificationConstants.NOTIFICATION_ID_DOWNLOAD_SUMMARY,
                    buildSummaryNotification(context, manager));
        } else {
            context.startService(intent);
        }
    }

    /**
     * Updates the notification summary with a new icon, if necessary.
     * @param removedNotificationId The id of a notification that is currently closing and should be
     *                              ignored.  -1 if no notifications are being closed.
     * @param addedNotification     A {@link Pair} of <id, Notification> of a notification that is
     *                              currently being added and should be used in addition to or in
     *                              place of the existing icons.
     */
    private static void updateSummaryIcon(Context context, NotificationManager manager,
            int removedNotificationId, Pair<Integer, Notification> addedNotification) {
        if (!useForegroundService()) return;

        Pair<Boolean, Integer> icon =
                getSummaryIcon(context, manager, removedNotificationId, addedNotification);
        if (!icon.first || !hasDownloadNotifications(manager, removedNotificationId)) return;

        manager.notify(NotificationConstants.NOTIFICATION_ID_DOWNLOAD_SUMMARY,
                buildSummaryNotificationWithIcon(context, icon.second));
    }

    /**
     * Returns whether or not there are any download notifications showing that aren't the summary
     * notification.
     * @param notificationIdToIgnore If not -1, the id of a notification to ignore and
     *                               assume is closing or about to be closed.
     * @return Whether or not there are valid download notifications currently visible.
     */
    @TargetApi(Build.VERSION_CODES.M)
    private static boolean hasDownloadNotifications(
            NotificationManager manager, int notificationIdToIgnore) {
        if (!useForegroundService()) return false;

        StatusBarNotification[] notifications = manager.getActiveNotifications();
        for (StatusBarNotification notification : notifications) {
            boolean isDownloadsGroup = TextUtils.equals(notification.getNotification().getGroup(),
                    NotificationConstants.GROUP_DOWNLOADS);
            boolean isSummaryNotification =
                    notification.getId() == NotificationConstants.NOTIFICATION_ID_DOWNLOAD_SUMMARY;
            boolean isIgnoredNotification =
                    notificationIdToIgnore != -1 && notificationIdToIgnore == notification.getId();
            if (isDownloadsGroup && !isSummaryNotification && !isIgnoredNotification) return true;
        }

        return false;
    }

    /**
     * Calculates the suggested icon for the summary notification based on the other notifications
     * currently showing.
     * @param context A context to use to query Android-specific information (NotificationManager).
     * @param removedNotificationId The id of a notification that is currently closing and should be
     *                              ignored.  -1 if no notifications are being closed.
     * @param addedNotification     A {@link Pair} of <id, Notification> of a notification that is
     *                              currently being added and should be used in addition to or in
     *                              place of the existing icons.
     * @return                      A {@link Pair} that represents both whether or not the new icon
     *                              is different from the old one and the icon id itself.
     */
    @TargetApi(Build.VERSION_CODES.M)
    private static Pair<Boolean, Integer> getSummaryIcon(Context context,
            NotificationManager manager, int removedNotificationId,
            Pair<Integer, Notification> addedNotification) {
        if (!useForegroundService()) return new Pair<Boolean, Integer>(false, -1);
        boolean progress = false;
        boolean paused = false;
        boolean pending = false;
        boolean completed = false;
        boolean failed = false;

        final int progressIcon = android.R.drawable.stat_sys_download;
        final int pausedIcon = R.drawable.ic_download_pause;
        final int pendingIcon = R.drawable.ic_download_pending;
        final int completedIcon = R.drawable.offline_pin;
        final int failedIcon = android.R.drawable.stat_sys_download_done;

        StatusBarNotification[] notifications = manager.getActiveNotifications();

        int oldIcon = -1;
        for (StatusBarNotification notification : notifications) {
            boolean isDownloadsGroup = TextUtils.equals(notification.getNotification().getGroup(),
                    NotificationConstants.GROUP_DOWNLOADS);
            if (!isDownloadsGroup) continue;
            if (notification.getId() == removedNotificationId) continue;

            boolean isSummaryNotification =
                    notification.getId() == NotificationConstants.NOTIFICATION_ID_DOWNLOAD_SUMMARY;

            if (addedNotification != null && addedNotification.first == notification.getId())
                continue;

            int icon =
                    notification.getNotification().extras.getInt(EXTRA_NOTIFICATION_BUNDLE_ICON_ID);
            if (isSummaryNotification) {
                oldIcon = icon;
                continue;
            }

            progress |= icon == progressIcon;
            paused |= icon == pausedIcon;
            pending |= icon == pendingIcon;
            completed |= icon == completedIcon;
            failed |= icon == failedIcon;
        }

        if (addedNotification != null) {
            int icon = addedNotification.second.extras.getInt(EXTRA_NOTIFICATION_BUNDLE_ICON_ID);

            progress |= icon == progressIcon;
            paused |= icon == pausedIcon;
            pending |= icon == pendingIcon;
            completed |= icon == completedIcon;
            failed |= icon == failedIcon;
        }

        int newIcon = android.R.drawable.stat_sys_download_done;
        if (progress) {
            newIcon = android.R.drawable.stat_sys_download;
        } else if (pending) {
            newIcon = R.drawable.ic_download_pending;
        } else if (failed) {
            newIcon = android.R.drawable.stat_sys_download_done;
        } else if (paused) {
            newIcon = R.drawable.ic_download_pause;
        } else if (completed) {
            newIcon = R.drawable.offline_pin;
        }

        return new Pair<Boolean, Integer>(newIcon != oldIcon, newIcon);
    }

    /**
     * Builds a summary notification that represents all downloads.
     * {@see #buildSummaryNotification(Context)}.
     * @param context A context used to query Android strings and resources.
     * @param iconId  The id of an icon to use for the notification.
     * @return        a {@link Notification} that represents the summary icon for all downloads.
     */
    private static Notification buildSummaryNotificationWithIcon(Context context, int iconId) {
        ChromeNotificationBuilder builder =
                AppHooks.get()
                        .createChromeNotificationBuilder(true /* preferCompat */,
                                NotificationConstants.CATEGORY_ID_BROWSER,
                                context.getString(R.string.notification_category_browser),
                                NotificationConstants.CATEGORY_GROUP_ID_GENERAL,
                                context.getString(R.string.notification_category_group_general))
                        .setContentTitle(
                                context.getString(R.string.download_notification_summary_title))
                        .setSubText(context.getString(R.string.menu_downloads))
                        .setSmallIcon(iconId)
                        .setLocalOnly(true)
                        .setGroup(NotificationConstants.GROUP_DOWNLOADS)
                        .setGroupSummary(true);
        Bundle extras = new Bundle();
        extras.putInt(EXTRA_NOTIFICATION_BUNDLE_ICON_ID, iconId);
        builder.addExtras(extras);

        // This notification should not actually be shown.  But if it is, set the click intent to
        // open downloads home.
        Intent downloadHomeIntent = buildActionIntent(
                context, DownloadManager.ACTION_NOTIFICATION_CLICKED, null, false, false);
        builder.setContentIntent(PendingIntent.getBroadcast(context,
                NotificationConstants.NOTIFICATION_ID_DOWNLOAD_SUMMARY, downloadHomeIntent,
                PendingIntent.FLAG_UPDATE_CURRENT));

        return builder.build();
    }

    /**
     * Builds a summary notification that represents downloads.  This is the notification passed to
     * {@link #startForeground(int, Notification)}, which keeps this service in the foreground.
     * @param context The context used to build the notification and pull specific resources.
     * @return The {@link Notification} to show for the summary.  Meant to be used by
     *         {@link NotificationManager#notify(int, Notification)}.
     */
    private static Notification buildSummaryNotification(
            Context context, NotificationManager manager) {
        Pair<Boolean, Integer> icon = getSummaryIcon(context, manager, -1, null);
        return buildSummaryNotificationWithIcon(context, icon.second);
    }

    /**
     * @return Whether or not there are any current resumable downloads being tracked.  These
     *         tracked downloads may not currently be showing notifications.
     */
    public static boolean isTrackingResumableDownloads(Context context) {
        List<DownloadSharedPreferenceEntry> entries =
                DownloadSharedPreferenceHelper.getInstance().getEntries();
        for (DownloadSharedPreferenceEntry entry : entries) {
            if (canResumeDownload(context, entry)) return true;
        }
        return false;
    }

    /**
     * Class for clients to access.
     */
    public class LocalBinder extends Binder {
        DownloadNotificationService getService() {
            return DownloadNotificationService.this;
        }
    }

    @Override
    public void onTaskRemoved(Intent rootIntent) {
        super.onTaskRemoved(rootIntent);
        // If we've lost all Activities, cancel the off the record downloads and validate that we
        // should still be showing any download notifications at all.
        if (ApplicationStatus.isEveryActivityDestroyed()) {
            cancelOffTheRecordDownloads();
            hideSummaryNotificationIfNecessary(-1);
        }
    }

    @Override
    public void onCreate() {
        mContext = ContextUtils.getApplicationContext();
        mNotificationManager = (NotificationManager) mContext.getSystemService(
                Context.NOTIFICATION_SERVICE);
        mSharedPrefs = ContextUtils.getAppSharedPreferences();
        mNumAutoResumptionAttemptLeft = mSharedPrefs.getInt(KEY_AUTO_RESUMPTION_ATTEMPT_LEFT,
                MAX_RESUMPTION_ATTEMPT_LEFT);
        mDownloadSharedPreferenceHelper = DownloadSharedPreferenceHelper.getInstance();
        mNextNotificationId = mSharedPrefs.getInt(
                KEY_NEXT_DOWNLOAD_NOTIFICATION_ID, STARTING_NOTIFICATION_ID);
    }

    @Override
    public void onDestroy() {
        updateNotificationsForShutdown();
        rescheduleDownloads();
        super.onDestroy();
    }

    @Override
    public int onStartCommand(final Intent intent, int flags, int startId) {
        if (intent == null) {
            // Intent is only null during a process restart because of returning START_STICKY.  In
            // this case cancel the off the record notifications and put the normal notifications
            // into a pending state, then try to restart.  Finally validate that we are actually
            // showing something.
            updateNotificationsForShutdown();
            handleDownloadOperation(
                    new Intent(DownloadNotificationService.ACTION_DOWNLOAD_RESUME_ALL));
            hideSummaryNotificationIfNecessary(-1);
        } else if (isDownloadOperationIntent(intent)) {
            handleDownloadOperation(intent);
            DownloadResumptionScheduler.getDownloadResumptionScheduler(mContext).cancelTask();
            // Limit the number of auto resumption attempts in case Chrome falls into a vicious
            // cycle.
            if (ACTION_DOWNLOAD_RESUME_ALL.equals(intent.getAction())) {
                if (mNumAutoResumptionAttemptLeft > 0) {
                    mNumAutoResumptionAttemptLeft--;
                    updateResumptionAttemptLeft();
                }
            } else {
                // Reset number of attempts left if the action is triggered by user.
                mNumAutoResumptionAttemptLeft = MAX_RESUMPTION_ATTEMPT_LEFT;
                clearResumptionAttemptLeft();
            }
        }
        // This should restart the service after Chrome gets killed. However, this
        // doesn't work on Android 4.4.2.
        return START_STICKY;
    }

    /**
     * Adds an {@link Observer}, which will be notified when this service attempts to
     * start stopping itself.
     */
    public void addObserver(Observer observer) {
        mObservers.addObserver(observer);
    }

    /**
     * Removes {@code observer}, which will no longer be notified when this class decides to start
     * stopping itself.
     */
    public void removeObserver(Observer observer) {
        mObservers.removeObserver(observer);
    }

    /**
     * On >= O Android releases, puts this service into a background state.
     * @param killNotification Whether or not this call should kill the summary notification or not.
     *                         Not killing it puts the service into the background, but leaves the
     *                         download notifications visible.
     */
    @VisibleForTesting
    @TargetApi(Build.VERSION_CODES.N)
    void stopForegroundInternal(boolean killNotification) {
        if (!useForegroundService()) return;
        stopForeground(killNotification ? STOP_FOREGROUND_REMOVE : STOP_FOREGROUND_DETACH);
    }

    /**
     * On >= O Android releases, puts this service into a foreground state, binding it to the
     * {@link Notification} generated by {@link #buildSummaryNotification(Context)}.
     */
    @VisibleForTesting
    void startForegroundInternal() {
        if (!useForegroundService()) return;
        Notification notification =
                buildSummaryNotification(getApplicationContext(), mNotificationManager);
        startForeground(NotificationConstants.NOTIFICATION_ID_DOWNLOAD_SUMMARY, notification);
    }

    @VisibleForTesting
    boolean hasDownloadNotificationsInternal(int notificationIdToIgnore) {
        return hasDownloadNotifications(mNotificationManager, notificationIdToIgnore);
    }

    @VisibleForTesting
    void updateSummaryIconInternal(
            int removedNotificationId, Pair<Integer, Notification> addedNotification) {
        updateSummaryIcon(mContext, mNotificationManager, removedNotificationId, addedNotification);
    }

    private void rescheduleDownloads() {
        List<DownloadSharedPreferenceEntry> entries = mDownloadSharedPreferenceHelper.getEntries();
        if (entries.isEmpty()) return;

        boolean scheduleAutoResumption = false;
        boolean allowMeteredConnection = false;
        for (int i = 0; i < entries.size(); ++i) {
            DownloadSharedPreferenceEntry entry = entries.get(i);
            if (entry.isAutoResumable) {
                scheduleAutoResumption = true;
                if (entry.canDownloadWhileMetered) {
                    allowMeteredConnection = true;
                    break;
                }
            }
        }
        if (scheduleAutoResumption && mNumAutoResumptionAttemptLeft > 0) {
            DownloadResumptionScheduler.getDownloadResumptionScheduler(mContext).schedule(
                    allowMeteredConnection);
        }
    }

    @VisibleForTesting
    void updateNotificationsForShutdown() {
        cancelOffTheRecordDownloads();
        List<DownloadSharedPreferenceEntry> entries = mDownloadSharedPreferenceHelper.getEntries();
        for (DownloadSharedPreferenceEntry entry : entries) {
            if (entry.isOffTheRecord) continue;
            // Move all regular downloads to pending.  Don't propagate the pause because
            // if native is still working and it triggers an update, then the service will be
            // restarted.
            notifyDownloadPaused(entry.downloadGuid, !entry.isOffTheRecord, true);
        }
    }

    @VisibleForTesting
    void cancelOffTheRecordDownloads() {
        boolean cancelActualDownload = LibraryLoader.isInitialized()
                && Profile.getLastUsedProfile().hasOffTheRecordProfile();

        List<DownloadSharedPreferenceEntry> entries = mDownloadSharedPreferenceHelper.getEntries();
        List<DownloadSharedPreferenceEntry> copies =
                new ArrayList<DownloadSharedPreferenceEntry>(entries);
        for (DownloadSharedPreferenceEntry entry : copies) {
            if (!entry.isOffTheRecord) continue;
            notifyDownloadCanceled(entry.downloadGuid);
            if (cancelActualDownload) {
                DownloadServiceDelegate delegate = getServiceDelegate(entry.itemType);
                delegate.cancelDownload(entry.downloadGuid, true);
                delegate.destroyServiceDelegate();
            }
            for (Observer observer : mObservers) observer.onDownloadCanceled(entry.downloadGuid);
        }
    }

    /**
     * Track in-progress downloads here and, if on an Android version >= O, make
     * this a foreground service.
     * @param downloadGuid The guid of the download that has been started and should be tracked.
     */
    private void startTrackingInProgressDownload(String downloadGuid) {
        if (mDownloadsInProgress.size() == 0) startForegroundInternal();
        if (!mDownloadsInProgress.contains(downloadGuid)) mDownloadsInProgress.add(downloadGuid);
    }

    /**
     * Stop tracking the download represented by {@code downloadGuid}.  If on an Android version >=
     * O, stop making this a foreground service.
     * @param downloadGuid The guid of the download that has been paused or canceled and shouldn't
     *                     be tracked.
     * @param allowStopForeground Whether or not this should check internal state and stop the
     *                            foreground notification from showing.  This could be false if we
     *                            plan on removing the notification in the near future.  We don't
     *                            want to just detach here, because that will put us in a
     *                            potentially bad state where we cannot dismiss the notification.
     */
    private void stopTrackingInProgressDownload(String downloadGuid, boolean allowStopForeground) {
        mDownloadsInProgress.remove(downloadGuid);
        if (allowStopForeground && mDownloadsInProgress.size() == 0) stopForegroundInternal(false);
    }

    /**
     * @return The summary {@link StatusBarNotification} if one is showing.
     */
    @TargetApi(Build.VERSION_CODES.M)
    private StatusBarNotification getSummaryNotification() {
        if (!useForegroundService()) return null;

        StatusBarNotification[] notifications = mNotificationManager.getActiveNotifications();
        for (StatusBarNotification notification : notifications) {
            boolean isDownloadsGroup = TextUtils.equals(notification.getNotification().getGroup(),
                    NotificationConstants.GROUP_DOWNLOADS);
            boolean isSummaryNotification =
                    notification.getId() == NotificationConstants.NOTIFICATION_ID_DOWNLOAD_SUMMARY;
            if (isDownloadsGroup && isSummaryNotification) return notification;
        }

        return null;
    }

    /**
     * Cancels the existing summary notification.  Moved to a helper method for test mocking.
     */
    @VisibleForTesting
    void cancelSummaryNotification() {
        mNotificationManager.cancel(NotificationConstants.NOTIFICATION_ID_DOWNLOAD_SUMMARY);
    }

    /**
     * Check all current notifications and hide the summary notification if we have no downloads
     * notifications left.  On Android if the user swipes away the last download notification the
     * summary will be dismissed.  But if the last downloads notification is dismissed via
     * {@link NotificationManager#cancel(int)}, the summary will remain, so we need to check and
     * manually remove it ourselves.
     * @param notificationIdToIgnore Canceling a notification and querying for the current list of
     *                               active notifications isn't synchronous.  Pass a notification id
     *                               here if there is a notification that should be assumed gone.
     *                               Or pass -1 if no notification fits that criteria.
     */
    @TargetApi(Build.VERSION_CODES.M)
    boolean hideSummaryNotificationIfNecessary(int notificationIdToIgnore) {
        if (!useForegroundService()) return false;
        if (mDownloadsInProgress.size() > 0) return false;

        if (hasDownloadNotificationsInternal(notificationIdToIgnore)) return false;

        StatusBarNotification notification = getSummaryNotification();
        if (notification != null) {
            // We have a valid summary notification, but how we dismiss it depends on whether or not
            // it is currently bound to this service via startForeground(...).
            if ((notification.getNotification().flags & Notification.FLAG_FOREGROUND_SERVICE)
                    != 0) {
                // If we are a foreground service and we are hiding the notification, we have no
                // other downloads notifications showing, so we need to remove the notification and
                // unregister it from this service at the same time.
                stopForegroundInternal(true);
            } else {
                // If we are not a foreground service, remove the notification via the
                // NotificationManager.  The notification is not bound to this service, so any call
                // to stopForeground() won't affect the notification.
                cancelSummaryNotification();
            }
        } else {
            // If we don't have a valid summary, just guarantee that we aren't in the foreground for
            // safety.
            stopForegroundInternal(false);
        }

        // Notify all observers that we are requesting a chance to shut down.  This will let any
        // observers unbind from us if necessary.
        for (Observer observer : mObservers) observer.onServiceShutdownRequested();

        // Stop the service which should start the destruction process.  At this point we should be
        // (1) a background service and (2) unbound from any clients.
        stopSelf();
        return true;
    }

    @Override
    public IBinder onBind(Intent intent) {
        return mBinder;
    }

    /**
     * Helper method to update the remaining number of background resumption attempts left.
     */
    private void updateResumptionAttemptLeft() {
        SharedPreferences.Editor editor = mSharedPrefs.edit();
        editor.putInt(KEY_AUTO_RESUMPTION_ATTEMPT_LEFT, mNumAutoResumptionAttemptLeft);
        editor.apply();
    }

    /**
     * Helper method to clear the remaining number of background resumption attempts left.
     */
    static void clearResumptionAttemptLeft() {
        SharedPreferences SharedPrefs = ContextUtils.getAppSharedPreferences();
        SharedPreferences.Editor editor = SharedPrefs.edit();
        editor.remove(KEY_AUTO_RESUMPTION_ATTEMPT_LEFT);
        editor.apply();
    }

    /**
     * Adds or updates an in-progress download notification.
     * @param downloadGuid GUID of the download.
     * @param fileName File name of the download.
     * @param percentage Percentage completed. Value should be between 0 to 100 if
     *        the percentage can be determined, or -1 if it is unknown.
     * @param bytesReceived Total number of bytes received.
     * @param timeRemainingInMillis Remaining download time in milliseconds.
     * @param startTime Time when download started.
     * @param isOffTheRecord Whether the download is off the record.
     * @param canDownloadWhileMetered Whether the download can happen in metered network.
     * @param isOfflinePage Whether the download is for offline page.
     */
    @VisibleForTesting
    public void notifyDownloadProgress(String downloadGuid, String fileName, int percentage,
            long bytesReceived, long timeRemainingInMillis, long startTime, boolean isOffTheRecord,
            boolean canDownloadWhileMetered, boolean isOfflinePage) {
        updateActiveDownloadNotification(downloadGuid, fileName, percentage, bytesReceived,
                timeRemainingInMillis, startTime, isOffTheRecord, canDownloadWhileMetered,
                isOfflinePage, false);
    }

    /**
     * Adds or updates a pending download notification.
     * @param downloadGuid GUID of the download.
     * @param fileName File name of the download.
     * @param isOffTheRecord Whether the download is off the record.
     * @param canDownloadWhileMetered Whether the download can happen in metered network.
     * @param isOfflinePage Whether the download is for offline page.
     */
    private void notifyDownloadPending(String downloadGuid, String fileName, boolean isOffTheRecord,
            boolean canDownloadWhileMetered, boolean isOfflinePage) {
        updateActiveDownloadNotification(downloadGuid, fileName,
                DownloadItem.INDETERMINATE_DOWNLOAD_PERCENTAGE, 0, 0, 0, isOffTheRecord,
                canDownloadWhileMetered, isOfflinePage, true);
    }

    /**
     * Helper method to update the notification for an active download, the download is either in
     * progress or pending.
     * @param downloadGuid GUID of the download.
     * @param fileName File name of the download.
     * @param percentage Percentage completed. Value should be between 0 to 100 if
     *        the percentage can be determined, or -1 if it is unknown.
     * @param bytesReceived Total number of bytes received.
     * @param timeRemainingInMillis Remaining download time in milliseconds or -1 if it is unknown.
     * @param startTime Time when download started.
     * @param isOffTheRecord Whether the download is off the record.
     * @param canDownloadWhileMetered Whether the download can happen in metered network.
     * @param isOfflinePage Whether the download is for offline page.
     * @param isDownloadPending Whether the download is pending.
     */
    private void updateActiveDownloadNotification(String downloadGuid, String fileName,
            int percentage, long bytesReceived, long timeRemainingInMillis, long startTime,
            boolean isOffTheRecord, boolean canDownloadWhileMetered, boolean isOfflinePage,
            boolean isDownloadPending) {
        boolean indeterminate =
                (percentage == DownloadItem.INDETERMINATE_DOWNLOAD_PERCENTAGE) || isDownloadPending;
        String contentText = null;
        if (isDownloadPending) {
            contentText = mContext.getResources().getString(R.string.download_notification_pending);
        } else if (indeterminate) {
            contentText = DownloadUtils.getStringForBytes(
                    mContext, DownloadUtils.BYTES_DOWNLOADED_STRINGS, bytesReceived);
        } else {
            contentText = timeRemainingInMillis < 0
                    ? mContext.getResources().getString(R.string.download_started)
                    : formatRemainingTime(mContext, timeRemainingInMillis);
        }
        int resId = isDownloadPending ? R.drawable.ic_download_pending
                : android.R.drawable.stat_sys_download;
        ChromeNotificationBuilder builder = buildNotification(resId, fileName, contentText);
        builder.setOngoing(true);
        builder.setPriority(Notification.PRIORITY_HIGH);

        // Avoid animations while the download isn't progressing.
        if (!isDownloadPending) {
            builder.setProgress(100, percentage, indeterminate);
        }

        if (!indeterminate && !isOfflinePage) {
            String percentText = DownloadUtils.getPercentageString(percentage);
            if (Build.VERSION.CODENAME.equals("N")
                    || Build.VERSION.SDK_INT > Build.VERSION_CODES.M) {
                builder.setSubText(percentText);
            } else {
                builder.setContentInfo(percentText);
            }
        }
        int notificationId = getNotificationId(downloadGuid);
        if (startTime > 0) builder.setWhen(startTime);

        // Clicking on an in-progress download sends the user to see all their downloads.
        Intent downloadHomeIntent = buildActionIntent(mContext,
                DownloadManager.ACTION_NOTIFICATION_CLICKED, null, isOffTheRecord, isOfflinePage);
        builder.setContentIntent(PendingIntent.getBroadcast(
                mContext, notificationId, downloadHomeIntent, PendingIntent.FLAG_UPDATE_CURRENT));
        builder.setAutoCancel(false);

        Intent pauseIntent = buildActionIntent(
                mContext, ACTION_DOWNLOAD_PAUSE, downloadGuid, isOffTheRecord, isOfflinePage);
        builder.addAction(R.drawable.ic_pause_white_24dp,
                mContext.getResources().getString(R.string.download_notification_pause_button),
                buildPendingIntent(pauseIntent, notificationId));

        Intent cancelIntent = buildActionIntent(
                mContext, ACTION_DOWNLOAD_CANCEL, downloadGuid, isOffTheRecord, isOfflinePage);
        builder.addAction(R.drawable.btn_close_white,
                mContext.getResources().getString(R.string.download_notification_cancel_button),
                buildPendingIntent(cancelIntent, notificationId));

        int itemType = isOfflinePage ? DownloadSharedPreferenceEntry.ITEM_TYPE_OFFLINE_PAGE
                                     : DownloadSharedPreferenceEntry.ITEM_TYPE_DOWNLOAD;
        updateNotification(notificationId, builder.build(), downloadGuid, isOfflinePage,
                new DownloadSharedPreferenceEntry(notificationId, isOffTheRecord,
                        canDownloadWhileMetered, downloadGuid, fileName, itemType, true));
        startTrackingInProgressDownload(downloadGuid);
    }

    /**
     * Cancel a download notification.
     * @param notificationId Notification ID of the download
     * @param downloadGuid GUID of the download.
     */
    @VisibleForTesting
    void cancelNotification(int notificationId, String downloadGuid) {
        mNotificationManager.cancel(NOTIFICATION_NAMESPACE, notificationId);
        mDownloadSharedPreferenceHelper.removeSharedPreferenceEntry(downloadGuid);

        // Since we are about to go through the process of validating whether or not we can shut
        // down, don't stop foreground if we have no download notifications left to show.  Hiding
        // the summary will take care of that for us.
        stopTrackingInProgressDownload(
                downloadGuid, hasDownloadNotificationsInternal(notificationId));
        if (!hideSummaryNotificationIfNecessary(notificationId)) {
            updateSummaryIcon(mContext, mNotificationManager, notificationId, null);
        }
    }

    /**
     * Called when a download is canceled.
     * @param downloadGuid GUID of the download.
     */
    @VisibleForTesting
    public void notifyDownloadCanceled(String downloadGuid) {
        DownloadSharedPreferenceEntry entry =
                mDownloadSharedPreferenceHelper.getDownloadSharedPreferenceEntry(downloadGuid);
        if (entry == null) return;
        cancelNotification(entry.notificationId, downloadGuid);
    }

    /**
     * Change a download notification to paused state.
     * @param downloadGuid GUID of the download.
     * @param isResumable Whether download can be resumed.
     * @param isAutoResumable whether download is can be resumed automatically.
     */
    public void notifyDownloadPaused(String downloadGuid, boolean isResumable,
            boolean isAutoResumable) {
        DownloadSharedPreferenceEntry entry =
                mDownloadSharedPreferenceHelper.getDownloadSharedPreferenceEntry(downloadGuid);
        if (entry == null) return;
        if (!isResumable) {
            notifyDownloadFailed(entry.isOfflinePage(), downloadGuid, entry.fileName);
            return;
        }
        // If download is already paused, do nothing.
        if (!entry.isAutoResumable) return;
        // If download is interrupted due to network disconnection, show download pending state.
        if (isAutoResumable) {
            notifyDownloadPending(entry.downloadGuid, entry.fileName, entry.isOffTheRecord,
                    entry.canDownloadWhileMetered, entry.isOfflinePage());
            stopTrackingInProgressDownload(entry.downloadGuid, true);
            return;
        }

        String contentText = mContext.getResources().getString(
                R.string.download_notification_paused);
        ChromeNotificationBuilder builder =
                buildNotification(R.drawable.ic_download_pause, entry.fileName, contentText);

        // Clicking on an in-progress download sends the user to see all their downloads.
        Intent downloadHomeIntent = buildActionIntent(
                mContext, DownloadManager.ACTION_NOTIFICATION_CLICKED, null, false, false);
        builder.setContentIntent(PendingIntent.getBroadcast(mContext, entry.notificationId,
                downloadHomeIntent, PendingIntent.FLAG_UPDATE_CURRENT));
        builder.setAutoCancel(false);

        Intent resumeIntent = buildActionIntent(mContext, ACTION_DOWNLOAD_RESUME,
                entry.downloadGuid, entry.isOffTheRecord, entry.isOfflinePage());
        builder.addAction(R.drawable.ic_get_app_white_24dp,
                mContext.getResources().getString(R.string.download_notification_resume_button),
                buildPendingIntent(resumeIntent, entry.notificationId));

        Intent cancelIntent = buildActionIntent(mContext, ACTION_DOWNLOAD_CANCEL,
                entry.downloadGuid, entry.isOffTheRecord, entry.isOfflinePage());
        builder.addAction(R.drawable.btn_close_white,
                mContext.getResources().getString(R.string.download_notification_cancel_button),
                buildPendingIntent(cancelIntent, entry.notificationId));
        builder.setDeleteIntent(buildSummaryIconIntent(entry.notificationId));

        updateNotification(entry.notificationId, builder.build(), downloadGuid,
                entry.isOfflinePage(),
                new DownloadSharedPreferenceEntry(entry.notificationId, entry.isOffTheRecord,
                        entry.canDownloadWhileMetered, entry.downloadGuid, entry.fileName,
                        entry.itemType, isAutoResumable));
        stopTrackingInProgressDownload(downloadGuid, true);
    }

    /**
     * Add a download successful notification.
     * @param downloadGuid GUID of the download.
     * @param filePath Full path to the download.
     * @param fileName Filename of the download.
     * @param systemDownloadId Download ID assigned by system DownloadManager.
     * @param isOfflinePage Whether the download is for offline page.
     * @param isSupportedMimeType Whether the MIME type can be viewed inside browser.
     * @return ID of the successful download notification. Used for removing the notification when
     *         user click on the snackbar.
     */
    @VisibleForTesting
    public int notifyDownloadSuccessful(
            String downloadGuid, String filePath, String fileName, long systemDownloadId,
            boolean isOfflinePage, boolean isSupportedMimeType) {
        int notificationId = getNotificationId(downloadGuid);
        ChromeNotificationBuilder builder = buildNotification(R.drawable.offline_pin, fileName,
                mContext.getResources().getString(R.string.download_notification_completed));
        ComponentName component = new ComponentName(
                mContext.getPackageName(), DownloadBroadcastReceiver.class.getName());
        Intent intent;
        if (isOfflinePage) {
            intent = buildActionIntent(mContext, ACTION_DOWNLOAD_OPEN, downloadGuid, false, true);
        } else {
            intent = new Intent(DownloadManager.ACTION_NOTIFICATION_CLICKED);
            long[] idArray = {systemDownloadId};
            intent.putExtra(DownloadManager.EXTRA_NOTIFICATION_CLICK_DOWNLOAD_IDS, idArray);
            intent.putExtra(EXTRA_DOWNLOAD_FILE_PATH, filePath);
            intent.putExtra(EXTRA_IS_SUPPORTED_MIME_TYPE, isSupportedMimeType);
        }
        intent.setComponent(component);
        builder.setContentIntent(PendingIntent.getBroadcast(
                mContext, notificationId, intent, PendingIntent.FLAG_UPDATE_CURRENT));
        if (mDownloadSuccessLargeIcon == null) {
            Bitmap bitmap = BitmapFactory.decodeResource(
                    mContext.getResources(), R.drawable.offline_pin);
            mDownloadSuccessLargeIcon = getLargeNotificationIcon(bitmap);
        }
        builder.setDeleteIntent(buildSummaryIconIntent(notificationId));
        builder.setLargeIcon(mDownloadSuccessLargeIcon);
        updateNotification(notificationId, builder.build(), downloadGuid, isOfflinePage, null);
        stopTrackingInProgressDownload(downloadGuid, true);
        return notificationId;
    }

    /**
     * Add a download failed notification.
     * @param isOfflinePage Whether or not the download was for an offline page.
     * @param downloadGuid GUID of the download.
     * @param fileName GUID of the download.
     */
    @VisibleForTesting
    public void notifyDownloadFailed(boolean isOfflinePage, String downloadGuid, String fileName) {
        // If the download is not in history db, fileName could be empty. Get it from
        // SharedPreferences.
        if (TextUtils.isEmpty(fileName)) {
            DownloadSharedPreferenceEntry entry =
                    mDownloadSharedPreferenceHelper.getDownloadSharedPreferenceEntry(downloadGuid);
            if (entry == null) return;
            fileName = entry.fileName;
        }

        int notificationId = getNotificationId(downloadGuid);
        ChromeNotificationBuilder builder =
                buildNotification(android.R.drawable.stat_sys_download_done, fileName,
                        mContext.getResources().getString(R.string.download_notification_failed));
        builder.setDeleteIntent(buildSummaryIconIntent(notificationId));
        updateNotification(notificationId, builder.build(), downloadGuid, isOfflinePage, null);
        stopTrackingInProgressDownload(downloadGuid, true);
    }

    /**
     * Helper method to build a PendingIntent from the provided intent.
     * @param intent Intent to broadcast.
     * @param notificationId ID of the notification.
     */
    private PendingIntent buildPendingIntent(Intent intent, int notificationId) {
        return PendingIntent.getBroadcast(
                mContext, notificationId, intent, PendingIntent.FLAG_UPDATE_CURRENT);
    }

    private PendingIntent buildSummaryIconIntent(int notificationId) {
        Intent intent = new Intent(mContext, DownloadBroadcastReceiver.class);
        intent.setAction(ACTION_DOWNLOAD_UPDATE_SUMMARY_ICON);
        return buildPendingIntent(intent, notificationId);
    }

    /**
     * Helper method to build an download action Intent from the provided information.
     * @param context {@link Context} to pull resources from.
     * @param action Download action to perform.
     * @param downloadGuid GUID of the download.
     * @param isOffTheRecord Whether the download is incognito.
     * @param isOfflinePage Whether the download represents an Offline Page.
     */
    static Intent buildActionIntent(Context context, String action, String downloadGuid,
            boolean isOffTheRecord, boolean isOfflinePage) {
        ComponentName component = new ComponentName(
                context.getPackageName(), DownloadBroadcastReceiver.class.getName());
        Intent intent = new Intent(action);
        intent.setComponent(component);
        intent.putExtra(EXTRA_DOWNLOAD_GUID, downloadGuid);
        intent.putExtra(EXTRA_IS_OFF_THE_RECORD, isOffTheRecord);
        intent.putExtra(EXTRA_IS_OFFLINE_PAGE, isOfflinePage);
        return intent;
    }

    /**
     * Builds a notification to be displayed.
     * @param iconId Id of the notification icon.
     * @param title Title of the notification.
     * @param contentText Notification content text to be displayed.
     * @return notification builder that builds the notification to be displayed
     */
    private ChromeNotificationBuilder buildNotification(
            int iconId, String title, String contentText) {
        Bundle extras = new Bundle();
        extras.putInt(EXTRA_NOTIFICATION_BUNDLE_ICON_ID, iconId);

        ChromeNotificationBuilder builder =
                AppHooks.get()
                        .createChromeNotificationBuilder(true /* preferCompat */,
                                NotificationConstants.CATEGORY_ID_BROWSER,
                                mContext.getString(R.string.notification_category_browser),
                                NotificationConstants.CATEGORY_GROUP_ID_GENERAL,
                                mContext.getString(R.string.notification_category_group_general))
                        .setContentTitle(
                                DownloadUtils.getAbbreviatedFileName(title, MAX_FILE_NAME_LENGTH))
                        .setSmallIcon(iconId)
                        .setLocalOnly(true)
                        .setAutoCancel(true)
                        .setContentText(contentText)
                        .setGroup(NotificationConstants.GROUP_DOWNLOADS)
                        .addExtras(extras);
        return builder;
    }

    private Bitmap getLargeNotificationIcon(Bitmap bitmap) {
        Resources resources = mContext.getResources();
        int height = (int) resources.getDimension(android.R.dimen.notification_large_icon_height);
        int width = (int) resources.getDimension(android.R.dimen.notification_large_icon_width);
        final OvalShape circle = new OvalShape();
        circle.resize(width, height);
        final Paint paint = new Paint();
        paint.setColor(ApiCompatibilityUtils.getColor(resources, R.color.google_blue_grey_500));

        final Bitmap result = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(result);
        circle.draw(canvas, paint);
        float leftOffset = (width - bitmap.getWidth()) / 2f;
        float topOffset = (height - bitmap.getHeight()) / 2f;
        if (leftOffset >= 0 && topOffset >= 0) {
            canvas.drawBitmap(bitmap, leftOffset, topOffset, null);
        } else {
            // Scale down the icon into the notification icon dimensions
            canvas.drawBitmap(bitmap,
                    new Rect(0, 0, bitmap.getWidth(), bitmap.getHeight()),
                    new Rect(0, 0, width, height),
                    null);
        }
        return result;
    }

    /**
     * Retrieves DownloadSharedPreferenceEntry from a download action intent.
     * @param intent Intent that contains the download action.
     */
    private DownloadSharedPreferenceEntry getDownloadEntryFromIntent(Intent intent) {
        if (ACTION_DOWNLOAD_RESUME_ALL.equals(intent.getAction())) return null;
        String guid = IntentUtils.safeGetStringExtra(intent, EXTRA_DOWNLOAD_GUID);
        return mDownloadSharedPreferenceHelper.getDownloadSharedPreferenceEntry(guid);
    }

    /**
     * Helper method to launch the browser process and handle a download operation that is included
     * in the given intent.
     * @param intent Intent with the download operation.
     */
    private void handleDownloadOperation(final Intent intent) {
        // Process updating the summary notification first.  This has no impact on a specific
        // download.
        if (ACTION_DOWNLOAD_UPDATE_SUMMARY_ICON.equals(intent.getAction())) {
            updateSummaryIcon(mContext, mNotificationManager, -1, null);
            hideSummaryNotificationIfNecessary(-1);
            return;
        }

        // TODO(qinmin): Figure out how to properly handle this case.
        boolean isOfflinePage =
                IntentUtils.safeGetBooleanExtra(intent, EXTRA_IS_OFFLINE_PAGE, false);
        final DownloadSharedPreferenceEntry entry = getDownloadEntryFromIntent(intent);
        if (entry == null
                && !(isOfflinePage && TextUtils.equals(intent.getAction(), ACTION_DOWNLOAD_OPEN))) {
            handleDownloadOperationForMissingNotification(intent);
            hideSummaryNotificationIfNecessary(-1);
            return;
        }

        if (ACTION_DOWNLOAD_PAUSE.equals(intent.getAction())) {
            // If browser process already goes away, the download should have already paused. Do
            // nothing in that case.
            if (!DownloadManagerService.hasDownloadManagerService()) {
                notifyDownloadPaused(entry.downloadGuid, !entry.isOffTheRecord, false);
                hideSummaryNotificationIfNecessary(-1);
                return;
            }
        } else if (ACTION_DOWNLOAD_RESUME.equals(intent.getAction())) {
            // If user manually resumes a download, update the network type if it
            // is not metered previously.
            boolean canDownloadWhileMetered = entry.canDownloadWhileMetered
                    || DownloadManagerService.isActiveNetworkMetered(mContext);
            // Update the SharedPreference entry.
            mDownloadSharedPreferenceHelper.addOrReplaceSharedPreferenceEntry(
                    new DownloadSharedPreferenceEntry(entry.notificationId, entry.isOffTheRecord,
                            canDownloadWhileMetered, entry.downloadGuid, entry.fileName,
                            entry.itemType, true));
        } else if (ACTION_DOWNLOAD_RESUME_ALL.equals(intent.getAction())
                && (mDownloadSharedPreferenceHelper.getEntries().isEmpty()
                        || DownloadManagerService.hasDownloadManagerService())) {
            hideSummaryNotificationIfNecessary(-1);
            return;
        } else if (ACTION_DOWNLOAD_OPEN.equals(intent.getAction())) {
            // TODO(fgorski): Do we even need to do anything special here, before we launch Chrome?
        } else if (ACTION_DOWNLOAD_CANCEL.equals(intent.getAction())
                && IntentUtils.safeGetBooleanExtra(intent, EXTRA_NOTIFICATION_DISMISSED, false)) {
            // User canceled a download by dismissing its notification from earlier versions, ignore
            // it. TODO(qinmin): remove this else-if block after M60.
            return;
        }

        BrowserParts parts = new EmptyBrowserParts() {
            @Override
            public boolean shouldStartGpuProcess() {
                return false;
            }

            @Override
            public void finishNativeInitialization() {
                int itemType = entry != null ? entry.itemType
                        : (ACTION_DOWNLOAD_OPEN.equals(intent.getAction())
                                ? DownloadSharedPreferenceEntry.ITEM_TYPE_OFFLINE_PAGE
                                : DownloadSharedPreferenceEntry.ITEM_TYPE_DOWNLOAD);
                DownloadServiceDelegate downloadServiceDelegate =
                        ACTION_DOWNLOAD_OPEN.equals(intent.getAction()) ? null
                                : getServiceDelegate(itemType);
                if (ACTION_DOWNLOAD_CANCEL.equals(intent.getAction())) {
                        // TODO(qinmin): Alternatively, we can delete the downloaded content on
                        // SD card, and remove the download ID from the SharedPreferences so we
                        // don't need to restart the browser process. http://crbug.com/579643.
                        cancelNotification(entry.notificationId, entry.downloadGuid);
                        downloadServiceDelegate.cancelDownload(entry.downloadGuid,
                                entry.isOffTheRecord);
                        for (Observer observer : mObservers) {
                            observer.onDownloadCanceled(entry.downloadGuid);
                        }
                } else if (ACTION_DOWNLOAD_PAUSE.equals(intent.getAction())) {
                        downloadServiceDelegate.pauseDownload(entry.downloadGuid,
                                entry.isOffTheRecord);
                } else if (ACTION_DOWNLOAD_RESUME.equals(intent.getAction())) {
                        notifyDownloadPending(entry.downloadGuid, entry.fileName,
                                entry.isOffTheRecord, entry.canDownloadWhileMetered,
                                entry.isOfflinePage());
                        downloadServiceDelegate.resumeDownload(entry.buildDownloadItem(), true);
                } else if (ACTION_DOWNLOAD_RESUME_ALL.equals(intent.getAction())) {
                        assert entry == null;
                        resumeAllPendingDownloads();
                } else if (ACTION_DOWNLOAD_OPEN.equals(intent.getAction())) {
                        OfflinePageDownloadBridge.openDownloadedPage(
                                IntentUtils.safeGetStringExtra(intent, EXTRA_DOWNLOAD_GUID));
                } else {
                        Log.e(TAG, "Unrecognized intent action.", intent);
                }
                if (!ACTION_DOWNLOAD_OPEN.equals(intent.getAction())) {
                    downloadServiceDelegate.destroyServiceDelegate();
                }

                hideSummaryNotificationIfNecessary(ACTION_DOWNLOAD_CANCEL.equals(intent.getAction())
                                ? entry.notificationId
                                : -1);
            }
        };
        try {
            ChromeBrowserInitializer.getInstance(mContext).handlePreNativeStartup(parts);
            ChromeBrowserInitializer.getInstance(mContext).handlePostNativeStartup(true, parts);
        } catch (ProcessInitException e) {
            Log.e(TAG, "Unable to load native library.", e);
            ChromeApplication.reportStartupErrorAndExit(e);
        }
    }

    /**
     * Handles operations for downloads that the DownloadNotificationService is unaware of.
     *
     * This can happen because the DownloadNotificationService learn about downloads later than
     * Download Home does, and may not yet have a DownloadSharedPreferenceEntry for the item.
     *
     * TODO(qinmin): Figure out how to fix the SharedPreferences so that it properly tracks entries.
     */
    private void handleDownloadOperationForMissingNotification(Intent intent) {
        // This function should only be called via Download Home, but catch this case to be safe.
        if (!DownloadManagerService.hasDownloadManagerService()) return;

        String action = intent.getAction();
        String downloadGuid = IntentUtils.safeGetStringExtra(intent, EXTRA_DOWNLOAD_GUID);
        boolean isOffTheRecord =
                IntentUtils.safeGetBooleanExtra(intent, EXTRA_IS_OFF_THE_RECORD, false);
        int itemType = IntentUtils.safeGetBooleanExtra(intent, EXTRA_IS_OFFLINE_PAGE, false)
                ? DownloadSharedPreferenceEntry.ITEM_TYPE_OFFLINE_PAGE
                : DownloadSharedPreferenceEntry.ITEM_TYPE_DOWNLOAD;
        if (itemType != DownloadSharedPreferenceEntry.ITEM_TYPE_DOWNLOAD) return;

        // Pass information directly to the DownloadManagerService.
        if (TextUtils.equals(action, ACTION_DOWNLOAD_CANCEL)) {
            getServiceDelegate(itemType).cancelDownload(downloadGuid, isOffTheRecord);
        } else if (TextUtils.equals(action, ACTION_DOWNLOAD_PAUSE)) {
            getServiceDelegate(itemType).pauseDownload(downloadGuid, isOffTheRecord);
        } else if (TextUtils.equals(action, ACTION_DOWNLOAD_RESUME)) {
            DownloadInfo info = new DownloadInfo.Builder()
                                        .setDownloadGuid(downloadGuid)
                                        .setIsOffTheRecord(isOffTheRecord)
                                        .build();
            getServiceDelegate(itemType).resumeDownload(new DownloadItem(false, info), true);
        }
    }

    /**
     * Gets appropriate download delegate that can handle interactions with download item referred
     * to by the entry.
     * @param forOfflinePage Whether the service should deal with offline pages or downloads.
     * @return delegate for interactions with the entry
     */
    DownloadServiceDelegate getServiceDelegate(int downloadItemType) {
        if (downloadItemType == DownloadSharedPreferenceEntry.ITEM_TYPE_OFFLINE_PAGE) {
            return OfflinePageDownloadBridge.getDownloadServiceDelegate();
        }
        if (downloadItemType != DownloadSharedPreferenceEntry.ITEM_TYPE_DOWNLOAD) {
            Log.e(TAG, "Unrecognized intent type.", downloadItemType);
        }
        return DownloadManagerService.getDownloadManagerService(getApplicationContext());
    }
    @VisibleForTesting
    void updateNotification(int id, Notification notification) {
        mNotificationManager.notify(NOTIFICATION_NAMESPACE, id, notification);
    }

    private void updateNotification(int id, Notification notification, String downloadGuid,
            boolean isOfflinePage, DownloadSharedPreferenceEntry entry) {
        updateNotification(id, notification);
        trackNotificationUma(isOfflinePage, downloadGuid);

        if (entry != null) {
            mDownloadSharedPreferenceHelper.addOrReplaceSharedPreferenceEntry(entry);
        } else {
            mDownloadSharedPreferenceHelper.removeSharedPreferenceEntry(downloadGuid);
        }
        updateSummaryIcon(mContext, mNotificationManager, -1,
                new Pair<Integer, Notification>(id, notification));
    }

    private void trackNotificationUma(boolean isOfflinePage, String downloadGuid) {
        // Check if we already have an entry in the DownloadSharedPreferenceHelper.  This is a
        // reasonable indicator for whether or not a notification is already showing (or at least if
        // we had built one for this download before.
        if (mDownloadSharedPreferenceHelper.hasEntry(downloadGuid)) return;
        NotificationUmaTracker.getInstance().onNotificationShown(isOfflinePage
                        ? NotificationUmaTracker.DOWNLOAD_PAGES
                        : NotificationUmaTracker.DOWNLOAD_FILES);
    }

    /**
     * Checks if an intent requires operations on a download.
     * @param intent An intent to validate.
     * @return true if the intent requires actions, or false otherwise.
     */
    static boolean isDownloadOperationIntent(Intent intent) {
        if (intent == null) return false;
        if (ACTION_DOWNLOAD_UPDATE_SUMMARY_ICON.equals(intent.getAction())) return true;
        if (ACTION_DOWNLOAD_RESUME_ALL.equals(intent.getAction())) return true;
        if (!ACTION_DOWNLOAD_CANCEL.equals(intent.getAction())
                && !ACTION_DOWNLOAD_RESUME.equals(intent.getAction())
                && !ACTION_DOWNLOAD_PAUSE.equals(intent.getAction())
                && !ACTION_DOWNLOAD_OPEN.equals(intent.getAction())) {
            return false;
        }
        if (!intent.hasExtra(EXTRA_DOWNLOAD_GUID)) return false;
        final String guid = IntentUtils.safeGetStringExtra(intent, EXTRA_DOWNLOAD_GUID);
        if (!DownloadSharedPreferenceEntry.isValidGUID(guid)) return false;
        return true;
    }

    private static boolean canResumeDownload(Context context, DownloadSharedPreferenceEntry entry) {
        if (entry == null) return false;
        if (!entry.isAutoResumable) return false;

        boolean isNetworkMetered = DownloadManagerService.isActiveNetworkMetered(context);
        return entry.canDownloadWhileMetered || !isNetworkMetered;
    }

    /**
     * Resumes all pending downloads from SharedPreferences. If a download is
     * already in progress, do nothing.
     */
    public void resumeAllPendingDownloads() {
        if (!DownloadManagerService.hasDownloadManagerService()) return;
        List<DownloadSharedPreferenceEntry> entries = mDownloadSharedPreferenceHelper.getEntries();
        for (int i = 0; i < entries.size(); ++i) {
            DownloadSharedPreferenceEntry entry = entries.get(i);
            if (!canResumeDownload(mContext, entry)) continue;
            if (mDownloadsInProgress.contains(entry.downloadGuid)) continue;

            notifyDownloadPending(entry.downloadGuid, entry.fileName, false,
                    entry.canDownloadWhileMetered, entry.isOfflinePage());
            DownloadServiceDelegate downloadServiceDelegate = getServiceDelegate(entry.itemType);
            downloadServiceDelegate.resumeDownload(entry.buildDownloadItem(), false);
            downloadServiceDelegate.destroyServiceDelegate();
        }
    }

    /**
     * Return the notification ID for the given download GUID.
     * @return notification ID to be used.
     */
    private int getNotificationId(String downloadGuid) {
        DownloadSharedPreferenceEntry entry =
                mDownloadSharedPreferenceHelper.getDownloadSharedPreferenceEntry(downloadGuid);
        if (entry != null) return entry.notificationId;
        int notificationId = mNextNotificationId;
        mNextNotificationId = mNextNotificationId == Integer.MAX_VALUE
                ? STARTING_NOTIFICATION_ID : mNextNotificationId + 1;
        SharedPreferences.Editor editor = mSharedPrefs.edit();
        editor.putInt(KEY_NEXT_DOWNLOAD_NOTIFICATION_ID, mNextNotificationId);
        editor.apply();
        return notificationId;
    }

    /**
     * Format remaining time for the given millis, in the following format:
     * 5 hours; will include 1 unit, can go down to seconds precision.
     * This is similar to what android.java.text.Formatter.formatShortElapsedTime() does. Don't use
     * ui::TimeFormat::Simple() as it is very expensive.
     *
     * @param context the application context.
     * @param millis the remaining time in milli seconds.
     * @return the formatted remaining time.
     */
    public static String formatRemainingTime(Context context, long millis) {
        long secondsLong = millis / 1000;

        int days = 0;
        int hours = 0;
        int minutes = 0;
        if (secondsLong >= SECONDS_PER_DAY) {
            days = (int) (secondsLong / SECONDS_PER_DAY);
            secondsLong -= days * SECONDS_PER_DAY;
        }
        if (secondsLong >= SECONDS_PER_HOUR) {
            hours = (int) (secondsLong / SECONDS_PER_HOUR);
            secondsLong -= hours * SECONDS_PER_HOUR;
        }
        if (secondsLong >= SECONDS_PER_MINUTE) {
            minutes = (int) (secondsLong / SECONDS_PER_MINUTE);
            secondsLong -= minutes * SECONDS_PER_MINUTE;
        }
        int seconds = (int) secondsLong;

        if (days >= 2) {
            days += (hours + 12) / 24;
            return context.getString(R.string.remaining_duration_days, days);
        } else if (days > 0) {
            return context.getString(R.string.remaining_duration_one_day);
        } else if (hours >= 2) {
            hours += (minutes + 30) / 60;
            return context.getString(R.string.remaining_duration_hours, hours);
        } else if (hours > 0) {
            return context.getString(R.string.remaining_duration_one_hour);
        } else if (minutes >= 2) {
            minutes += (seconds + 30) / 60;
            return context.getString(R.string.remaining_duration_minutes, minutes);
        } else if (minutes > 0) {
            return context.getString(R.string.remaining_duration_one_minute);
        } else if (seconds == 1) {
            return context.getString(R.string.remaining_duration_one_second);
        } else {
            return context.getString(R.string.remaining_duration_seconds, seconds);
        }
    }
}
