// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_ENGAGEMENT_IMPORTANT_SITES_UTIL_H_
#define CHROME_BROWSER_ENGAGEMENT_IMPORTANT_SITES_UTIL_H_

#include <string>
#include <vector>

#include "base/macros.h"
#include "url/gurl.h"

class Profile;

namespace user_prefs {
class PrefRegistrySyncable;
}

class ImportantSitesUtil {
 public:
  struct ImportantDomainInfo {
    std::string registerable_domain;
    GURL example_origin;
    double engagement_score = 0;
    int32_t reason_bitfield = 0;
  };

  static bool IsDialogDisabled(Profile* profile);

  static void RegisterProfilePrefs(user_prefs::PrefRegistrySyncable* registry);

  // This returns the top |<=max_results| important registrable domains. This
  // uses site engagement and notifications to generate the list. |max_results|
  // is assumed to be small.
  // See net/base/registry_controlled_domains/registry_controlled_domain.h for
  // more details on registrable domains and the current list of effective
  // eTLDs.
  static std::vector<ImportantDomainInfo> GetImportantRegisterableDomains(
      Profile* profile,
      size_t max_results);

  // Record the sites that the user chose to blacklist from clearing (in the
  // Clear Browsing Dialog) and the sites they ignored. The blacklisted sites
  // are NOT cleared as they are 'blacklisted' from the clear operation.
  // This records metrics for blacklisted and ignored sites and removes any
  // 'ignored' sites from our important sites list if they were ignored 3 times
  // in a row.
  static void RecordBlacklistedAndIgnoredImportantSites(
      Profile* profile,
      const std::vector<std::string>& blacklisted_sites,
      const std::vector<int32_t>& blacklisted_sites_reason_bitfield,
      const std::vector<std::string>& ignored_sites,
      const std::vector<int32_t>& ignored_sites_reason_bitfield);

  // This marks the given origin as important for testing. Note: This changes
  // the score requirements for the Site Engagement Service, so ONLY call for
  // testing.
  static void MarkOriginAsImportantForTesting(Profile* profile,
                                              const GURL& origin);

 private:
  DISALLOW_IMPLICIT_CONSTRUCTORS(ImportantSitesUtil);
};

#endif  // CHROME_BROWSER_ENGAGEMENT_IMPORTANT_SITES_UTIL_H_
