// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_EXTENSIONS_API_DIAL_DIAL_DEVICE_DATA_H_
#define CHROME_BROWSER_EXTENSIONS_API_DIAL_DIAL_DEVICE_DATA_H_

#include <string>
#include <vector>

#include "base/time/time.h"
#include "base/values.h"
#include "url/gurl.h"

namespace extensions {
namespace api {
namespace dial {

struct DialDevice;

// Dial device information that is used within the DialService and Registry on
// the IO thread. It is updated as new information arrives and a list of
// DialDeviceData is copied and sent to event listeners on the UI thread.
class DialDeviceData {
 public:
  DialDeviceData();
  DialDeviceData(const std::string& device_id,
                 const GURL& device_description_url,
                 const base::Time& response_time);
  DialDeviceData(const DialDeviceData& other);
  ~DialDeviceData();

  bool operator==(const DialDeviceData& other_data) const {
    return device_id_ == other_data.device_id_;
  }

  const std::string& device_id() const { return device_id_; }
  void set_device_id(const std::string& id) {
    device_id_ = id;
  }

  const std::string& label() const { return label_; }
  void set_label(const std::string& label) {
    label_ = label;
  }

  const GURL& device_description_url() const;
  void set_device_description_url(const GURL& url);

  const base::Time& response_time() const { return response_time_; }
  void set_response_time(const base::Time& response_time) {
    response_time_ = response_time;
  }

  int max_age() const { return max_age_; }
  void set_max_age(int max_age) { max_age_ = max_age; }
  bool has_max_age() const { return max_age_ >= 0; }

  int config_id() const { return config_id_; }
  void set_config_id(int config_id) { config_id_ = config_id; }
  bool has_config_id() const { return config_id_ >= 0; }

  // Fills the |device| API struct from this instance.
  void FillDialDevice(api::dial::DialDevice* device) const;

  // Updates this DeviceData based on information from a new response in
  // |new_data|.  Returns |true| if a field was updated that is visible through
  // the DIAL API.
  bool UpdateFrom(const DialDeviceData& new_data);

  // Validates that the URL is valid for the device description.
  static bool IsDeviceDescriptionUrl(const GURL& url);

 private:
  // Hardware identifier from the DIAL response.  Not exposed to API clients.
  std::string device_id_;

  // Identifies this device to clients of the API as a proxy for the hardware
  // identifier.  Automatically generated by the DIAL registry.
  std::string label_;

  // The device description URL.
  GURL device_description_url_;

  // The time that the most recent response was received.
  base::Time response_time_;

  // Optional (-1 means unset).
  int max_age_;

  // Optional (-1 means unset).
  int config_id_;
};

struct DialDeviceDescriptionData {
 public:
  DialDeviceDescriptionData() = default;
  DialDeviceDescriptionData(const std::string& device_description,
                            const GURL& app_url);

  std::string device_description;
  GURL app_url;
};

}  // namespace dial
}  // namespace api
}  // namespace extensions

#endif  // CHROME_BROWSER_EXTENSIONS_API_DIAL_DIAL_DEVICE_DATA_H_
