#!/usr/bin/env python
# Copyright 2016 The LUCI Authors. All rights reserved.
# Use of this source code is governed under the Apache License, Version 2.0
# that can be found in the LICENSE file.

"""A low-level blob storage/retrieval interface to the Isolate server"""

import base64
import binascii
import collections
import logging
import re
import sys
import threading
import time
import types

from utils import file_path
from utils import net

import isolated_format

# gRPC may not be installed on the worker machine. This is fine, as long as
# the bot doesn't attempt to use gRPC (checked in IsolateServerGrpc.__init__).
try:
  import grpc
  from proto import isolate_bot_pb2
except ImportError:
  grpc = None
  isolate_bot_pb2 = None


# Chunk size to use when reading from network stream.
NET_IO_FILE_CHUNK = 16 * 1024


# Read timeout in seconds for downloads from isolate storage. If there's no
# response from the server within this timeout whole download will be aborted.
DOWNLOAD_READ_TIMEOUT = 60


# A class to use to communicate with the server by default. Can be changed by
# 'set_storage_api_class'. Default is IsolateServer.
_storage_api_cls = None


class Item(object):
  """An item to push to Storage.

  Its digest and size may be provided in advance, if known. Otherwise they will
  be derived from content(). If digest is provided, it MUST correspond to
  hash algorithm used by Storage.

  When used with Storage, Item starts its life in a main thread, travels
  to 'contains' thread, then to 'push' thread and then finally back to
  the main thread. It is never used concurrently from multiple threads.
  """

  def __init__(self, digest=None, size=None, high_priority=False):
    self.digest = digest
    self.size = size
    self.high_priority = high_priority
    self.compression_level = 6

  def content(self):
    """Iterable with content of this item as byte string (str) chunks."""
    raise NotImplementedError()

  def prepare(self, hash_algo):
    """Ensures self.digest and self.size are set.

    Uses content() as a source of data to calculate them. Does nothing if digest
    and size is already known.

    Arguments:
      hash_algo: hash algorithm to use to calculate digest.
    """
    if self.digest is None or self.size is None:
      digest = hash_algo()
      total = 0
      for chunk in self.content():
        digest.update(chunk)
        total += len(chunk)
      self.digest = digest.hexdigest()
      self.size = total


class StorageApi(object):
  """Interface for classes that implement low-level storage operations.

  StorageApi is oblivious of compression and hashing scheme used. This details
  are handled in higher level Storage class.

  Clients should generally not use StorageApi directly. Storage class is
  preferred since it implements compression and upload optimizations.
  """

  @property
  def location(self):
    """URL of the backing store that this class is using."""
    raise NotImplementedError()

  @property
  def namespace(self):
    """Isolate namespace used by this storage.

    Indirectly defines hashing scheme and compression method used.
    """
    raise NotImplementedError()

  def fetch(self, digest, offset=0):
    """Fetches an object and yields its content.

    Arguments:
      digest: hash digest of item to download.
      offset: offset (in bytes) from the start of the file to resume fetch from.

    Yields:
      Chunks of downloaded item (as str objects).
    """
    raise NotImplementedError()

  def push(self, item, push_state, content=None):
    """Uploads an |item| with content generated by |content| generator.

    |item| MUST go through 'contains' call to get |push_state| before it can
    be pushed to the storage.

    To be clear, here is one possible usage:
      all_items = [... all items to push as Item subclasses ...]
      for missing_item, push_state in storage_api.contains(all_items).items():
        storage_api.push(missing_item, push_state)

    When pushing to a namespace with compression, data that should be pushed
    and data provided by the item is not the same. In that case |content| is
    not None and it yields chunks of compressed data (using item.content() as
    a source of original uncompressed data). This is implemented by Storage
    class.

    Arguments:
      item: Item object that holds information about an item being pushed.
      push_state: push state object as returned by 'contains' call.
      content: a generator that yields chunks to push, item.content() if None.

    Returns:
      None.
    """
    raise NotImplementedError()

  def contains(self, items):
    """Checks for |items| on the server, prepares missing ones for upload.

    Arguments:
      items: list of Item objects to check for presence.

    Returns:
      A dict missing Item -> opaque push state object to be passed to 'push'.
      See doc string for 'push'.
    """
    raise NotImplementedError()


class _IsolateServerPushState(object):
  """Per-item state passed from IsolateServer.contains to IsolateServer.push.

  Note this needs to be a global class to support pickling.
  """

  def __init__(self, preupload_status, size):
    self.preupload_status = preupload_status
    gs_upload_url = preupload_status.get('gs_upload_url') or None
    if gs_upload_url:
      self.upload_url = gs_upload_url
      self.finalize_url = 'api/isolateservice/v1/finalize_gs_upload'
    else:
      self.upload_url = 'api/isolateservice/v1/store_inline'
      self.finalize_url = None
    self.uploaded = False
    self.finalized = False
    self.size = size


def guard_memory_use(server, content, size):
  """Guards a server against using excessive memory while uploading.

  The server needs to contain a _memory_use int and a _lock mutex
  (both IsolateServer and IsolateServerGrpc qualify); this function
  then uses those values to track memory usage in a thread-safe way.

  If a request would cause the memory usage to exceed a safe maximum,
  this function sleeps in 0.1s increments until memory usage falls
  below the maximum.
  """
  if isinstance(content, (basestring, list)):
    # Memory is already used, too late.
    with server._lock:
      server._memory_use += size
  else:
    # TODO(vadimsh): Do not read from |content| generator when retrying push.
    # If |content| is indeed a generator, it can not be re-winded back to the
    # beginning of the stream. A retry will find it exhausted. A possible
    # solution is to wrap |content| generator with some sort of caching
    # restartable generator. It should be done alongside streaming support
    # implementation.
    #
    # In theory, we should keep the generator, so that it is not serialized in
    # memory. Sadly net.HttpService.request() requires the body to be
    # serialized.
    assert isinstance(content, types.GeneratorType), repr(content)
    slept = False
    # HACK HACK HACK. Please forgive me for my sins but OMG, it works!
    # One byte less than 512mb. This is to cope with incompressible content.
    max_size = int(sys.maxsize * 0.25)
    while True:
      with server._lock:
        # This is due to 32 bits python when uploading very large files. The
        # problem is that it's comparing uncompressed sizes, while we care
        # about compressed sizes since it's what is serialized in memory.
        # The first check assumes large files are compressible and that by
        # throttling one upload at once, we can survive. Otherwise, kaboom.
        memory_use = server._memory_use
        if ((size >= max_size and not memory_use) or
            (memory_use + size <= max_size)):
          server._memory_use += size
          memory_use = server._memory_use
          break
      time.sleep(0.1)
      slept = True
    if slept:
      logging.info('Unblocked: %d %d', memory_use, size)


class IsolateServer(StorageApi):
  """StorageApi implementation that downloads and uploads to Isolate Server.

  It uploads and downloads directly from Google Storage whenever appropriate.
  Works only within single namespace.
  """

  def __init__(self, base_url, namespace):
    super(IsolateServer, self).__init__()
    assert file_path.is_url(base_url), base_url
    self._base_url = base_url.rstrip('/')
    self._namespace = namespace
    self._namespace_dict = {
        'compression': 'flate' if namespace.endswith(
            ('-gzip', '-flate')) else '',
        'digest_hash': 'sha-1',
        'namespace': namespace,
    }
    self._lock = threading.Lock()
    self._server_caps = None
    self._memory_use = 0

  @property
  def _server_capabilities(self):
    """Gets server details.

    Returns:
      Server capabilities dictionary as returned by /server_details endpoint.
    """
    # TODO(maruel): Make this request much earlier asynchronously while the
    # files are being enumerated.

    # TODO(vadimsh): Put |namespace| in the URL so that server can apply
    # namespace-level ACLs to this call.

    with self._lock:
      if self._server_caps is None:
        self._server_caps = net.url_read_json(
            url='%s/api/isolateservice/v1/server_details' % self._base_url,
            data={})
      return self._server_caps

  @property
  def location(self):
    return self._base_url

  @property
  def namespace(self):
    return self._namespace

  def fetch(self, digest, offset=0):
    assert offset >= 0
    source_url = '%s/api/isolateservice/v1/retrieve' % (
        self._base_url)
    logging.debug('download_file(%s, %d)', source_url, offset)
    response = self._do_fetch(source_url, digest, offset)

    if not response:
      raise IOError(
          'Attempted to fetch from %s; no data exist: %s / %s.' % (
            source_url, self._namespace, digest))

    # for DB uploads
    content = response.get('content')
    if content is not None:
      yield base64.b64decode(content)
      return

    # for GS entities
    connection = net.url_open(response['url'])
    if not connection:
      raise IOError('Failed to download %s / %s' % (self._namespace, digest))

    # If |offset|, verify server respects it by checking Content-Range.
    if offset:
      content_range = connection.get_header('Content-Range')
      if not content_range:
        raise IOError('Missing Content-Range header')

      # 'Content-Range' format is 'bytes <offset>-<last_byte_index>/<size>'.
      # According to a spec, <size> can be '*' meaning "Total size of the file
      # is not known in advance".
      try:
        match = re.match(r'bytes (\d+)-(\d+)/(\d+|\*)', content_range)
        if not match:
          raise ValueError()
        content_offset = int(match.group(1))
        last_byte_index = int(match.group(2))
        size = None if match.group(3) == '*' else int(match.group(3))
      except ValueError:
        raise IOError('Invalid Content-Range header: %s' % content_range)

      # Ensure returned offset equals requested one.
      if offset != content_offset:
        raise IOError('Expecting offset %d, got %d (Content-Range is %s)' % (
            offset, content_offset, content_range))

      # Ensure entire tail of the file is returned.
      if size is not None and last_byte_index + 1 != size:
        raise IOError('Incomplete response. Content-Range: %s' % content_range)

    for data in connection.iter_content(NET_IO_FILE_CHUNK):
      yield data

  def push(self, item, push_state, content=None):
    assert isinstance(item, Item)
    assert item.digest is not None
    assert item.size is not None
    assert isinstance(push_state, _IsolateServerPushState)
    assert not push_state.finalized

    # Default to item.content().
    content = item.content() if content is None else content
    logging.info('Push state size: %d', push_state.size)
    guard_memory_use(self, content, push_state.size)

    try:
      # This push operation may be a retry after failed finalization call below,
      # no need to reupload contents in that case.
      if not push_state.uploaded:
        # PUT file to |upload_url|.
        success = self._do_push(push_state, content)
        if not success:
          raise IOError('Failed to upload file with hash %s to URL %s' % (
              item.digest, push_state.upload_url))
        push_state.uploaded = True
      else:
        logging.info(
            'A file %s already uploaded, retrying finalization only',
            item.digest)

      # Optionally notify the server that it's done.
      if push_state.finalize_url:
        # TODO(vadimsh): Calculate MD5 or CRC32C sum while uploading a file and
        # send it to isolated server. That way isolate server can verify that
        # the data safely reached Google Storage (GS provides MD5 and CRC32C of
        # stored files).
        # TODO(maruel): Fix the server to accept properly data={} so
        # url_read_json() can be used.
        response = net.url_read_json(
            url='%s/%s' % (self._base_url, push_state.finalize_url),
            data={
                'upload_ticket': push_state.preupload_status['upload_ticket'],
            })
        if not response or not response['ok']:
          raise IOError('Failed to finalize file with hash %s.' % item.digest)
      push_state.finalized = True
    finally:
      with self._lock:
        self._memory_use -= push_state.size

  def contains(self, items):
    # Ensure all items were initialized with 'prepare' call. Storage does that.
    assert all(i.digest is not None and i.size is not None for i in items)

    # Request body is a json encoded list of dicts.
    body = {
        'items': [
          {
            'digest': item.digest,
            'is_isolated': bool(item.high_priority),
            'size': item.size,
          } for item in items
        ],
        'namespace': self._namespace_dict,
    }

    query_url = '%s/api/isolateservice/v1/preupload' % self._base_url

    # Response body is a list of push_urls (or null if file is already present).
    response = None
    try:
      response = net.url_read_json(url=query_url, data=body)
      if response is None:
        raise isolated_format.MappingError(
            'Failed to execute preupload query')
    except ValueError as err:
      raise isolated_format.MappingError(
          'Invalid response from server: %s, body is %s' % (err, response))

    # Pick Items that are missing, attach _PushState to them.
    missing_items = {}
    for preupload_status in response.get('items', []):
      assert 'upload_ticket' in preupload_status, (
          preupload_status, '/preupload did not generate an upload ticket')
      index = int(preupload_status['index'])
      missing_items[items[index]] = _IsolateServerPushState(
          preupload_status, items[index].size)
    logging.info('Queried %d files, %d cache hit',
        len(items), len(items) - len(missing_items))
    return missing_items

  def _do_fetch(self, url, digest, offset):
    """Fetches isolated data from the URL.

    Used only for fetching files, not for API calls. Can be overridden in
    subclasses.

    Args:
      url: URL to fetch the data from, can possibly return http redirect.
      offset: byte offset inside the file to start fetching from.

    Returns:
      net.HttpResponse compatible object, with 'read' and 'get_header' calls.
    """
    assert isinstance(offset, int)
    data = {
        'digest': digest.encode('utf-8'),
        'namespace': self._namespace_dict,
        'offset': offset,
    }
    # TODO(maruel): url + '?' + urllib.urlencode(data) once a HTTP GET endpoint
    # is added.
    return net.url_read_json(
        url=url,
        data=data,
        read_timeout=DOWNLOAD_READ_TIMEOUT)

  def _do_push(self, push_state, content):
    """Uploads isolated file to the URL.

    Used only for storing files, not for API calls. Can be overridden in
    subclasses.

    Args:
      url: URL to upload the data to.
      push_state: an _IsolateServicePushState instance
      item: the original Item to be uploaded
      content: an iterable that yields 'str' chunks.
    """
    # A cheezy way to avoid memcpy of (possibly huge) file, until streaming
    # upload support is implemented.
    if isinstance(content, list) and len(content) == 1:
      content = content[0]
    else:
      content = ''.join(content)

    # DB upload
    if not push_state.finalize_url:
      url = '%s/%s' % (self._base_url, push_state.upload_url)
      content = base64.b64encode(content)
      data = {
          'upload_ticket': push_state.preupload_status['upload_ticket'],
          'content': content,
      }
      response = net.url_read_json(url=url, data=data)
      return response is not None and response['ok']

    # upload to GS
    url = push_state.upload_url
    response = net.url_read(
        content_type='application/octet-stream',
        data=content,
        method='PUT',
        headers={'Cache-Control': 'public, max-age=31536000'},
        url=url)
    return response is not None


class _IsolateServerGrpcPushState(object):
  """Empty class, just to present same interface as IsolateServer  """

  def __init__(self):
    pass


class IsolateServerGrpc(StorageApi):
  """StorageApi implementation that downloads and uploads to a gRPC service.

  Limitations: only works for the default-gzip namespace, and with zero offsets
  while fetching.
  """

  def __init__(self, server, namespace):
    super(IsolateServerGrpc, self).__init__()
    logging.info('Using gRPC for Isolate')

    # Make sure grpc was successfully imported
    assert grpc
    assert isolate_bot_pb2

    # Proxies only support the default-gzip namespace for now.
    # TODO(aludwin): support other namespaces
    assert namespace == 'default-gzip'
    self._server = server
    self._channel = grpc.insecure_channel(server)
    self._stub = isolate_bot_pb2.FileServiceStub(self._channel)
    self._lock = threading.Lock()
    self._memory_use = 0
    logging.info('...gRPC successfully initialized')

  @property
  def location(self):
    return self._server

  @property
  def namespace(self):
    # This is used to determine if the data is compressed, but gRPC proxies
    # don't have concepts of 'namespaces' and natively compress all messages
    # before transmission. So return an unlikely-to-be-used name so that
    # isolateserver doesn't try to compress anything.
    return 'grpc-proxy'

  def fetch(self, digest, offset=0):
    # The gRPC APIs only work with an offset of 0
    assert offset == 0
    request = isolate_bot_pb2.FetchBlobsRequest()
    req_digest = request.digest.add()
    # Convert the utf-8 encoded hexidecimal string (like '012abc') to a byte
    # array (like [0x01, 0x2a, 0xbc]).
    req_digest.digest = binascii.unhexlify(digest)
    expected_offset = 0
    try:
      for response in self._stub.FetchBlobs(request,
                                            timeout=DOWNLOAD_READ_TIMEOUT):
        if not response.status.succeeded:
          raise IOError(
              'Error while fetching %s: %s' % (digest, response.status))
        if not expected_offset == response.data.offset:
          raise IOError(
              'Error while fetching %s: expected offset %d, got %d' % (
                  digest, expected_offset, response.data.offset))
        expected_offset += len(response.data.data)
        yield response.data.data
    except grpc.RpcError as g:
      logging.error('gRPC error during fetch: re-throwing as IOError (%s)' % g)
      raise IOError(g)

  def push(self, item, push_state, content=None):
    assert isinstance(item, Item)
    assert item.digest is not None
    assert item.size is not None
    assert isinstance(push_state, _IsolateServerGrpcPushState)

    # Default to item.content().
    content = item.content() if content is None else content
    guard_memory_use(self, content, item.size)

    try:
      def chunker():
        # Returns one bit of content at a time
        if (isinstance(content, str)
            or not isinstance(content, collections.Iterable)):
          yield content
        else:
          for chunk in content:
            yield chunk
      def slicer():
        # Ensures every bit of content is under the gRPC max size; yields
        # proto messages to send via gRPC.
        request = isolate_bot_pb2.PushBlobsRequest()
        request.data.digest.digest = binascii.unhexlify(item.digest)
        request.data.digest.size_bytes = item.size
        request.data.offset = 0
        for chunk in chunker():
          # Make sure we send at least one chunk for zero-length blobs
          has_sent_anything = False
          while chunk or not has_sent_anything:
            slice_len = min(len(chunk), NET_IO_FILE_CHUNK)
            request.data.data = chunk[:slice_len]
            yield request
            has_sent_anything = True
            request.data.offset += slice_len
            # The proxy only expects the first chunk to have the digest
            request.data.ClearField("digest")
            chunk = chunk[slice_len:]

      # TODO(aludwin): batch up several requests to reuse TCP connections
      try:
        response = self._stub.PushBlobs(slicer())
      except grpc.RpcError as g:
        logging.error('gRPC error during push: re-throwing as IOError (%s)' % g)
        raise IOError(g)

      if not response.status.succeeded:
        raise IOError(
            'Error while uploading %s: %s' % (
                item.digest, response.status.error_detail))

    finally:
      with self._lock:
        self._memory_use -= item.size

  def contains(self, items):
    """Returns the set of all missing items."""
    # Ensure all items were initialized with 'prepare' call. Storage does that.
    assert all(i.digest is not None and i.size is not None for i in items)
    request = isolate_bot_pb2.ContainsRequest()
    items_by_digest = {}
    for item in items:
      cd = request.digest.add()
      cd.digest = binascii.unhexlify(item.digest)
      items_by_digest[cd.digest] = item
    try:
      response = self._stub.Contains(request)
    except grpc.RpcError as g:
      logging.error('gRPC error during contains: re-throwing as IOError (%s)'
                    % g)
      raise IOError(g)

    # If everything's present, return the empty set.
    if response.status.succeeded:
      return {}

    if not response.status.error == isolate_bot_pb2.BlobStatus.MISSING_DIGEST:
      raise IOError('Unknown response during lookup: %s' % response.status)

    # Pick Items that are missing, attach _PushState to them. The gRPC
    # implementation doesn't actually have a push state, we just attach
    # empty objects to satisfy the StorageApi interface.
    missing_items = {}
    for missing in response.status.missing_digest:
      item = items_by_digest[missing.digest]
      missing_items[item] = _IsolateServerGrpcPushState()

    logging.info('Queried %d files, %d cache hit',
        len(items), len(items) - len(missing_items))
    return missing_items


def set_storage_api_class(cls):
  """Replaces StorageApi implementation used by default."""
  global _storage_api_cls
  assert _storage_api_cls is None
  assert issubclass(cls, StorageApi)
  _storage_api_cls = cls


def get_storage_api(url, namespace):
  """Returns an object that implements low-level StorageApi interface.

  It is used by Storage to work with single isolate |namespace|. It should
  rarely be used directly by clients, see 'get_storage' for
  a better alternative.

  Arguments:
    url: URL of isolate service to use shared cloud based storage.
    namespace: isolate namespace to operate in, also defines hashing and
        compression scheme used, i.e. namespace names that end with '-gzip'
        store compressed data.

  Returns:
    Instance of StorageApi subclass.
  """
  cls = _storage_api_cls or IsolateServer
  return cls(url, namespace)
