// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.compositor;

import android.annotation.TargetApi;
import android.graphics.Rect;
import android.support.test.annotation.UiThreadTest;
import android.support.test.filters.SmallTest;
import android.view.SurfaceView;
import android.view.View;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import org.chromium.chrome.browser.ChromeActivityTest;
import org.chromium.chrome.browser.compositor.layouts.LayoutRenderHost;
import org.chromium.ui.resources.ResourceManager;

/**
 * Integration tests for {@link org.chromium.chrome.browser.compositor.CompositorView}.
 */
@TargetApi(21)
public class CompositorVisibilityTest extends ChromeActivityTest {
    private CompositorView mCompositorView;

    private LayoutRenderHost mRenderHost = new LayoutRenderHost() {
        @Override
        public void requestRender() {}

        @Override
        public void onCompositorLayout() {}

        @Override
        public void didSwapFrame(int pendingFrameCount) {}

        @Override
        public void onSurfaceCreated() {}

        @Override
        public void onPhysicalBackingSizeChanged(int width, int height) {}

        @Override
        public void pushDebugRect(Rect rect, int color) {}

        @Override
        public void loadPersitentTextureDataIfNeeded() {}

        @Override
        public int getBrowserControlsBackgroundColor() {
            return 0;
        }

        @Override
        public float getBrowserControlsUrlBarAlpha() {
            return 0;
        }

        @Override
        public boolean areBrowserControlsPermanentlyHidden() {
            return false;
        }

        @Override
        public ResourceManager getResourceManager() {
            return null;
        }

        @Override
        public void invalidateAccessibilityProvider() {}
    };

    // Verify that setVisibility on |mCompositorView| is transferred to its children.  Otherwise,
    // the underlying surface is not destroyed.  This can interfere with VR, which hides the
    // CompositorView and creates its own surfaces.  The compositor surfaces can show up when the VR
    // surfaces are supposed to be visible.
    @SmallTest
    @UiThreadTest
    public void testSetVisibilityHidesSurfaces() {
        mCompositorView = new CompositorView(getActivity(), mRenderHost);
        mCompositorView.setVisibility(View.VISIBLE);
        assertEquals(View.VISIBLE, mCompositorView.getChildAt(0).getVisibility());
        mCompositorView.setVisibility(View.INVISIBLE);
        assertEquals(View.INVISIBLE, mCompositorView.getChildAt(0).getVisibility());
    }

    // The surfaceview should be attached during construction, so that the application window knows
    // to set the blending hint correctly on the surface.  Otherwise, it will have to setFormat()
    // when the SurfaceView is attached to the CompositorView, which causes visual artifacts when
    // the surface is torn down and re-created (crbug.com/704866).
    @SmallTest
    @UiThreadTest
    public void testSurfaceViewIsAttachedImmediately() {
        mCompositorView = new CompositorView(getActivity(), mRenderHost);
        assertEquals(mCompositorView.getChildCount(), 1);
        assertTrue(mCompositorView.getChildAt(0) instanceof SurfaceView);
    }
}
