// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * @fileoverview 'password-edit-dialog' is the dialog that allows showing a
 *     saved password.
 */

(function() {
'use strict';

Polymer({
  is: 'password-edit-dialog',

  properties: {
    /**
     * The password that is being displayed.
     * @type {!chrome.passwordsPrivate.PasswordUiEntry}
     */
    item: Object,

    /** Holds the plaintext password when requested. */
    password: String,
  },

  /** @override */
  attached: function() {
    this.password = '';
    this.$.dialog.showModal();
  },

  /** Closes the dialog. */
  close: function() {
    this.$.dialog.close();
  },

  /**
   * Gets the password input's type. Should be 'text' when password is visible
   * and 'password' when it's not.
   * @param {string} password
   * @private
   */
  getPasswordInputType_: function(password) {
    return password ? 'text' : 'password';
  },

  /**
   * Gets the title text for the show/hide icon.
   * @param {string} password
   * @param {string} hide The i18n text to use for 'Hide'
   * @param {string} show The i18n text to use for 'Show'
   * @private
   */
  showPasswordTitle_: function(password, hide, show) {
    return password ? hide : show;
  },

  /**
   * Gets the text of the password. Will use the value of |password| unless it
   * cannot be shown, in which case it will be spaces.
   * @param {!chrome.passwordsPrivate.PasswordUiEntry} item
   * @param {string} password
   * @private
   */
  getPassword_: function(item, password) {
    if (password)
      return password;
    return item ? ' '.repeat(item.numCharactersInPassword) : '';
  },

  /**
   * Handler for tapping the show/hide button. Will fire an event to request the
   * password for this login pair.
   * @private
   */
  onShowPasswordButtonTap_: function() {
    if (this.password)
      this.password = '';
    else
      this.fire('show-password', this.item.loginPair);  // Request the password.
  },

  /**
   * Handler for tapping the 'done' button. Should just dismiss the dialog.
   * @private
   */
  onActionButtonTap_: function() {
    this.close();
  },

  /**
   * @param {!Event} event
   * @private
   */
  onReadonlyInputTap_: function(event) {
    /** @type {!PaperInputElement} */ (Polymer.dom(event).localTarget)
        .inputElement.select();
  }
});
})();
