// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/android/offline_pages/downloads/resource_throttle.h"

#include "base/logging.h"
#include "chrome/browser/android/offline_pages/offline_page_utils.h"
#include "components/offline_pages/core/client_namespace_constants.h"
#include "content/public/browser/browser_thread.h"
#include "content/public/browser/resource_request_info.h"
#include "content/public/browser/web_contents.h"
#include "net/base/mime_util.h"

namespace {
// Mime type of download resource that should trigger handoff to OfflinePages
// backend for full page load and snapshot.
bool CanDownloadAsOfflinePage(const std::string& contents_mime_type) {
  return net::MatchesMimeType(contents_mime_type, "text/html") ||
         net::MatchesMimeType(contents_mime_type, "application/xhtml+xml");
}

void WillStartOfflineRequestOnUIThread(
    const GURL& url,
    const content::ResourceRequestInfo::WebContentsGetter& contents_getter) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);
  content::WebContents* web_contents = contents_getter.Run();
  if (!web_contents)
    return;
  offline_pages::OfflinePageUtils::ScheduleDownload(
      web_contents, offline_pages::kDownloadNamespace, url,
      offline_pages::OfflinePageUtils::DownloadUIActionFlags::ALL);
}
}  // namespace

namespace offline_pages {
namespace downloads {

ResourceThrottle::ResourceThrottle(const net::URLRequest* request)
    : request_(request) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::IO);
}

ResourceThrottle::~ResourceThrottle() {
  DCHECK_CURRENTLY_ON(content::BrowserThread::IO);
}

void ResourceThrottle::WillProcessResponse(bool* defer) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::IO);
  std::string mime_type;
  request_->GetMimeType(&mime_type);
  if (CanDownloadAsOfflinePage(mime_type)) {
    const content::ResourceRequestInfo* info =
        content::ResourceRequestInfo::ForRequest(request_);
    if (!info)
      return;
    content::BrowserThread::PostTask(
        content::BrowserThread::UI, FROM_HERE,
        base::Bind(&WillStartOfflineRequestOnUIThread, request_->url(),
                   info->GetWebContentsGetterForRequest()));
    Cancel();
  }
}

const char* ResourceThrottle::GetNameForLogging() const {
  return "offline_pages::downloads::ResourceThrottle";
}

}  // namespace downloads
}  // namespace offline_pages
