// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.infobar;

import android.content.Context;
import android.content.res.Resources;
import android.view.Gravity;
import android.view.View;
import android.widget.FrameLayout;

import org.chromium.chrome.R;

/**
 * Layout that holds an infobar's contents and provides a background color and a top shadow.
 */
class InfoBarWrapper extends FrameLayout {

    private final InfoBarContainerLayout.Item mItem;

    /** Whether or not the height of the layout should be restricted for animations. */
    private boolean mRestrictHeightForAnimation;

    /**
     * The height in px that this view will be restricted to if
     * {@link #mRestrictHeightForAnimation} is set.
     */
    private int mHeightForAnimationPx;

    /**
     * Constructor for inflating from Java.
     */
    InfoBarWrapper(Context context, InfoBarContainerLayout.Item item) {
        super(context);
        mItem = item;
        Resources res = context.getResources();
        int peekingHeight = res.getDimensionPixelSize(R.dimen.infobar_peeking_height);
        int shadowHeight = res.getDimensionPixelSize(R.dimen.infobar_shadow_height);
        setMinimumHeight(peekingHeight + shadowHeight);

        // setBackgroundResource() changes the padding, so call setPadding() second.
        setBackgroundResource(R.drawable.infobar_wrapper_bg);
        setPadding(0, shadowHeight, 0, 0);
        setClipChildren(true);
    }

    /**
     * @param restrict Whether or not the height of this view should be restricted for animations.
     */
    public void setRestrictHeightForAnimation(boolean restrict) {
        mRestrictHeightForAnimation = restrict;
    }

    /**
     * @param heightPx The restricted height in px that will be used if
     *                 {@link #mRestrictHeightForAnimation} is set.
     */
    public void setHeightForAnimation(int heightPx) {
        mHeightForAnimationPx = heightPx;
    }

    InfoBarContainerLayout.Item getItem() {
        return mItem;
    }

    @Override
    public void onMeasure(int widthSpec, int heightSpec) {
        if (mRestrictHeightForAnimation) {
            int heightPx = Math.min(mHeightForAnimationPx, MeasureSpec.getSize(heightSpec));
            heightSpec = MeasureSpec.makeMeasureSpec(heightPx, MeasureSpec.getMode(heightSpec));
        }

        super.onMeasure(widthSpec, heightSpec);
    }

    @Override
    public void onViewAdded(View child) {
        child.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT,
                Gravity.TOP));
    }
}
