// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.customtabs;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.support.test.InstrumentationRegistry;
import android.support.test.filters.LargeTest;
import android.support.test.filters.MediumTest;
import android.util.Base64;
import android.view.Menu;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;

import org.chromium.base.ActivityState;
import org.chromium.base.ApplicationStatus;
import org.chromium.base.ThreadUtils;
import org.chromium.base.test.util.CommandLineFlags;
import org.chromium.base.test.util.Feature;
import org.chromium.chrome.R;
import org.chromium.chrome.browser.ChromeSwitches;
import org.chromium.chrome.browser.appmenu.AppMenuHandler;
import org.chromium.chrome.browser.customtabs.CustomTabDelegateFactory.CustomTabNavigationDelegate;
import org.chromium.chrome.browser.document.ChromeLauncherActivity;
import org.chromium.chrome.browser.externalnav.ExternalNavigationHandler.OverrideUrlLoadingResult;
import org.chromium.chrome.browser.preferences.ChromePreferenceManager;
import org.chromium.chrome.browser.tab.InterceptNavigationDelegateImpl;
import org.chromium.chrome.browser.tab.Tab;
import org.chromium.chrome.browser.tab.TabDelegateFactory;
import org.chromium.chrome.test.ChromeActivityTestRule;
import org.chromium.chrome.test.ChromeJUnit4ClassRunner;
import org.chromium.content.browser.test.util.Criteria;
import org.chromium.content.browser.test.util.CriteriaHelper;
import org.chromium.net.test.EmbeddedTestServer;

import java.util.concurrent.Callable;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Tests for external navigation handling of Custom Tabs generated by Chrome.
 */
@RunWith(ChromeJUnit4ClassRunner.class)
@CommandLineFlags.Add({ChromeSwitches.DISABLE_FIRST_RUN_EXPERIENCE,
        ChromeActivityTestRule.DISABLE_NETWORK_PREDICTION_FLAG,
        ChromeSwitches.HERB_FLAVOR_ELDERBERRY_SWITCH})
public class CustomTabFromChromeExternalNavigationTest {
    @Rule
    public CustomTabActivityTestRule mCustomTabActivityTestRule = new CustomTabActivityTestRule();

    private EmbeddedTestServer mTestServer;

    @Before
    public void setUp() throws Exception {
        mTestServer = EmbeddedTestServer.createAndStartServer(
                InstrumentationRegistry.getInstrumentation().getContext());

        ChromePreferenceManager.getInstance().setCachedHerbFlavor(
                ChromeSwitches.HERB_FLAVOR_ELDERBERRY);
    }

    @After
    public void tearDown() throws Exception {
        mTestServer.stopAndDestroyServer();
    }

    private void startActivityCompletely(Intent intent) {
        Activity activity = InstrumentationRegistry.getInstrumentation().startActivitySync(intent);
        Assert.assertTrue(activity instanceof CustomTabActivity);
        mCustomTabActivityTestRule.setActivity((CustomTabActivity) activity);
    }

    private Intent getCustomTabFromChromeIntent(final String url) {
        return ThreadUtils.runOnUiThreadBlockingNoException(new Callable<Intent>() {
            @Override
            public Intent call() throws Exception {
                Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                return ChromeLauncherActivity.createCustomTabActivityIntent(
                        InstrumentationRegistry.getInstrumentation().getTargetContext(), intent,
                        true);
            }
        });

    }

    private void startCustomTabFromChrome(String url) throws InterruptedException {
        mCustomTabActivityTestRule.startCustomTabActivityWithIntent(
                getCustomTabFromChromeIntent(url));
    }

    private void startPaymentRequestUIFromChrome(String url) throws InterruptedException {
        Intent intent = getCustomTabFromChromeIntent(url);
        CustomTabIntentDataProvider.addPaymentRequestUIExtras(intent);

        mCustomTabActivityTestRule.startCustomTabActivityWithIntent(intent);
        InstrumentationRegistry.getInstrumentation().waitForIdleSync();
    }

    @Test
    @Feature("CustomTabFromChrome")
    @MediumTest
    public void testUsingStandardExternalNavigationHandler() throws Exception {
        startCustomTabFromChrome("about:blank");

        Tab tab = mCustomTabActivityTestRule.getActivity().getActivityTab();
        TabDelegateFactory delegateFactory = tab.getDelegateFactory();
        Assert.assertTrue(delegateFactory instanceof CustomTabDelegateFactory);
        CustomTabDelegateFactory customTabDelegateFactory =
                ((CustomTabDelegateFactory) delegateFactory);
        Assert.assertFalse(customTabDelegateFactory.getExternalNavigationDelegate()
                                   instanceof CustomTabNavigationDelegate);
    }

    @Test
    @Feature("CustomTabFromChrome")
    @LargeTest
    public void testIntentWithRedirectToApp() throws Exception {
        final String redirectUrl = "https://maps.google.com/maps?q=1600+amphitheatre+parkway";
        final String initialUrl = mTestServer.getURL(
                "/chrome/test/data/android/redirect/js_redirect.html"
                + "?replace_text="
                + Base64.encodeToString("PARAM_URL".getBytes("utf-8"), Base64.URL_SAFE) + ":"
                + Base64.encodeToString(redirectUrl.getBytes("utf-8"), Base64.URL_SAFE));

        startActivityCompletely(getCustomTabFromChromeIntent(initialUrl));

        final AtomicReference<InterceptNavigationDelegateImpl> navigationDelegate =
                new AtomicReference<>();
        CriteriaHelper.pollUiThread(new Criteria("Tab never selected/initialized.") {
            @Override
            public boolean isSatisfied() {
                Tab tab = mCustomTabActivityTestRule.getActivity().getActivityTab();
                if (tab == null || tab.getInterceptNavigationDelegate() == null) return false;
                navigationDelegate.set(tab.getInterceptNavigationDelegate());
                return true;
            }
        });

        CriteriaHelper.pollUiThread(Criteria.equals(
                OverrideUrlLoadingResult.OVERRIDE_WITH_EXTERNAL_INTENT,
                new Callable<OverrideUrlLoadingResult>() {
                    @Override
                    public OverrideUrlLoadingResult call() throws Exception {
                        return navigationDelegate.get().getLastOverrideUrlLoadingResultForTests();
                    }
                }));

        CriteriaHelper.pollUiThread(new Criteria() {
            @Override
            public boolean isSatisfied() {
                int activityState = ApplicationStatus.getStateForActivity(
                        mCustomTabActivityTestRule.getActivity());
                return activityState == ActivityState.STOPPED
                        || activityState == ActivityState.DESTROYED;
            }
        });
    }

    @Test
    @Feature("CustomTabFromChrome")
    @MediumTest
    public void testIntentToOpenPaymentRequestUI() throws Exception {
        startPaymentRequestUIFromChrome("about:blank");

        Tab tab = mCustomTabActivityTestRule.getActivity().getActivityTab();
        TabDelegateFactory delegateFactory = tab.getDelegateFactory();
        Assert.assertTrue(delegateFactory instanceof CustomTabDelegateFactory);
        CustomTabDelegateFactory customTabDelegateFactory =
                ((CustomTabDelegateFactory) delegateFactory);
        Assert.assertFalse(customTabDelegateFactory.getExternalNavigationDelegate()
                                   instanceof CustomTabNavigationDelegate);

        showAppMenuAndAssertMenuShown(mCustomTabActivityTestRule.getActivity().getAppMenuHandler());
        Menu menu =
                mCustomTabActivityTestRule.getActivity().getAppMenuHandler().getAppMenu().getMenu();

        Assert.assertTrue(menu.findItem(R.id.icon_row_menu_id).isVisible());
        Assert.assertTrue(menu.findItem(R.id.find_in_page_id).isVisible());
        Assert.assertFalse(menu.findItem(R.id.open_in_browser_id).isVisible());
        Assert.assertFalse(menu.findItem(R.id.bookmark_this_page_id).isVisible());
        Assert.assertFalse(menu.findItem(R.id.offline_page_id).isVisible());
        Assert.assertFalse(menu.findItem(R.id.request_desktop_site_row_menu_id).isVisible());
        Assert.assertFalse(menu.findItem(R.id.add_to_homescreen_id).isVisible());
        Assert.assertFalse(menu.findItem(R.id.open_webapk_id).isVisible());
    }

    private void showAppMenuAndAssertMenuShown(final AppMenuHandler appMenuHandler) {
        ThreadUtils.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                appMenuHandler.showAppMenu(null, false);
            }
        });
        CriteriaHelper.pollUiThread(new Criteria("AppMenu did not show") {
            @Override
            public boolean isSatisfied() {
                return appMenuHandler.isAppMenuShowing();
            }
        });
    }
}
