// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.ntp.cards;

import android.support.annotation.Nullable;
import android.support.v7.widget.RecyclerView;
import android.support.v7.widget.RecyclerView.Adapter;
import android.view.View;
import android.view.ViewGroup;

import org.chromium.base.Callback;
import org.chromium.base.VisibleForTesting;
import org.chromium.chrome.browser.ChromeFeatureList;
import org.chromium.chrome.browser.ntp.ContextMenuManager;
import org.chromium.chrome.browser.ntp.cards.NewTabPageViewHolder.PartialBindCallback;
import org.chromium.chrome.browser.ntp.snippets.CategoryInt;
import org.chromium.chrome.browser.ntp.snippets.CategoryStatus;
import org.chromium.chrome.browser.ntp.snippets.KnownCategories;
import org.chromium.chrome.browser.ntp.snippets.SectionHeaderViewHolder;
import org.chromium.chrome.browser.ntp.snippets.SnippetArticleViewHolder;
import org.chromium.chrome.browser.ntp.snippets.SnippetsBridge;
import org.chromium.chrome.browser.ntp.snippets.SuggestionsSource;
import org.chromium.chrome.browser.offlinepages.OfflinePageBridge;
import org.chromium.chrome.browser.suggestions.DestructionObserver;
import org.chromium.chrome.browser.suggestions.SiteSection;
import org.chromium.chrome.browser.suggestions.SuggestionsCarousel;
import org.chromium.chrome.browser.suggestions.SuggestionsConfig;
import org.chromium.chrome.browser.suggestions.SuggestionsRecyclerView;
import org.chromium.chrome.browser.suggestions.SuggestionsUiDelegate;
import org.chromium.chrome.browser.suggestions.TileGroup;
import org.chromium.chrome.browser.util.FeatureUtilities;
import org.chromium.chrome.browser.widget.displaystyle.UiConfig;

import java.util.List;
import java.util.Set;

/**
 * A class that handles merging above the fold elements and below the fold cards into an adapter
 * that will be used to back the NTP RecyclerView. The first element in the adapter should always be
 * the above-the-fold view (containing the logo, search box, and most visited tiles) and subsequent
 * elements will be the cards shown to the user
 */
public class NewTabPageAdapter extends Adapter<NewTabPageViewHolder> implements NodeParent {
    private final SuggestionsUiDelegate mUiDelegate;
    private final ContextMenuManager mContextMenuManager;

    @Nullable
    private final View mAboveTheFoldView;
    private final UiConfig mUiConfig;
    private SuggestionsRecyclerView mRecyclerView;

    private final InnerNode mRoot;

    @Nullable
    private final AboveTheFoldItem mAboveTheFold;
    @Nullable
    private final SiteSection mSiteSection;
    private final SuggestionsCarousel mSuggestionsCarousel;
    private final SectionList mSections;
    private final SignInPromo mSigninPromo;
    private final AllDismissedItem mAllDismissed;
    private final Footer mFooter;
    private final SpacingItem mBottomSpacer;

    /**
     * Creates the adapter that will manage all the cards to display on the NTP.
     * @param uiDelegate used to interact with the rest of the system.
     * @param aboveTheFoldView the layout encapsulating all the above-the-fold elements
     *         (logo, search box, most visited tiles), or null if only suggestions should
     *         be displayed.
     * @param uiConfig the NTP UI configuration, to be passed to created views.
     * @param offlinePageBridge used to determine if articles are available.
     * @param contextMenuManager used to build context menus.
     * @param tileGroupDelegate if not null this is used to build a {@link SiteSection}.
     * @param suggestionsCarousel if not null this is used to build a carousel showing contextual
     *         suggestions.
     */
    public NewTabPageAdapter(SuggestionsUiDelegate uiDelegate, @Nullable View aboveTheFoldView,
            UiConfig uiConfig, OfflinePageBridge offlinePageBridge,
            ContextMenuManager contextMenuManager, @Nullable TileGroup.Delegate tileGroupDelegate,
            @Nullable SuggestionsCarousel suggestionsCarousel) {
        mUiDelegate = uiDelegate;
        mContextMenuManager = contextMenuManager;

        mAboveTheFoldView = aboveTheFoldView;
        mUiConfig = uiConfig;
        mRoot = new InnerNode();
        mSections = new SectionList(mUiDelegate, offlinePageBridge);
        mSigninPromo = new SignInPromo(mUiDelegate);
        mAllDismissed = new AllDismissedItem();

        if (mAboveTheFoldView == null) {
            mAboveTheFold = null;
        } else {
            mAboveTheFold = new AboveTheFoldItem();
            mRoot.addChild(mAboveTheFold);
        }

        mSuggestionsCarousel = suggestionsCarousel;
        if (suggestionsCarousel != null) {
            assert ChromeFeatureList.isEnabled(ChromeFeatureList.CONTEXTUAL_SUGGESTIONS_CAROUSEL);
            mRoot.addChild(mSuggestionsCarousel);
        }

        if (tileGroupDelegate == null) {
            mSiteSection = null;
        } else {
            mSiteSection = new SiteSection(uiDelegate, mContextMenuManager, tileGroupDelegate,
                    offlinePageBridge, uiConfig);
            mRoot.addChild(mSiteSection);
        }

        if (FeatureUtilities.isChromeHomeModernEnabled()) {
            mRoot.addChildren(mSigninPromo, mAllDismissed, mSections);
        } else {
            mRoot.addChildren(mSections, mSigninPromo, mAllDismissed);
        }

        mFooter = new Footer();
        mRoot.addChild(mFooter);

        if (mAboveTheFoldView == null
                || ChromeFeatureList.isEnabled(ChromeFeatureList.NTP_CONDENSED_LAYOUT)) {
            mBottomSpacer = null;
        } else {
            mBottomSpacer = new SpacingItem();
            mRoot.addChild(mBottomSpacer);
        }

        RemoteSuggestionsStatusObserver suggestionsObserver = new RemoteSuggestionsStatusObserver();
        mUiDelegate.addDestructionObserver(suggestionsObserver);

        updateAllDismissedVisibility();
        mRoot.setParent(this);
    }

    @Override
    @ItemViewType
    public int getItemViewType(int position) {
        return mRoot.getItemViewType(position);
    }

    @Override
    public NewTabPageViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
        assert parent == mRecyclerView;

        switch (viewType) {
            case ItemViewType.ABOVE_THE_FOLD:
                return new NewTabPageViewHolder(mAboveTheFoldView);

            case ItemViewType.SITE_SECTION:
                return SiteSection.createViewHolder(SiteSection.inflateSiteSection(parent));

            case ItemViewType.HEADER:
                return new SectionHeaderViewHolder(mRecyclerView, mUiConfig);

            case ItemViewType.SNIPPET:
                return new SnippetArticleViewHolder(
                        mRecyclerView, mContextMenuManager, mUiDelegate, mUiConfig);

            case ItemViewType.SPACING:
                return new NewTabPageViewHolder(SpacingItem.createView(parent));

            case ItemViewType.STATUS:
                return new StatusCardViewHolder(mRecyclerView, mContextMenuManager, mUiConfig);

            case ItemViewType.PROGRESS:
                return new ProgressViewHolder(mRecyclerView);

            case ItemViewType.ACTION:
                return new ActionItem.ViewHolder(
                        mRecyclerView, mContextMenuManager, mUiDelegate, mUiConfig);

            case ItemViewType.PROMO:
                return mSigninPromo.createViewHolder(mRecyclerView, mContextMenuManager, mUiConfig);

            case ItemViewType.FOOTER:
                return new Footer.ViewHolder(mRecyclerView, mUiDelegate.getNavigationDelegate());

            case ItemViewType.ALL_DISMISSED:
                return new AllDismissedItem.ViewHolder(mRecyclerView, mSections);

            case ItemViewType.CAROUSEL:
                return mSuggestionsCarousel.createViewHolder(parent);
        }

        assert false : viewType;
        return null;
    }

    @Override
    public void onBindViewHolder(NewTabPageViewHolder holder, int position, List<Object> payloads) {
        if (payloads.isEmpty()) {
            mRoot.onBindViewHolder(holder, position);
            return;
        }

        for (Object payload : payloads) {
            ((PartialBindCallback) payload).onResult(holder);
        }
    }

    @Override
    public void onBindViewHolder(NewTabPageViewHolder holder, final int position) {
        mRoot.onBindViewHolder(holder, position);
    }

    @Override
    public int getItemCount() {
        return mRoot.getItemCount();
    }

    /** Resets suggestions, pulling the current state as known by the backend. */
    public void refreshSuggestions() {
        if (FeatureUtilities.isChromeHomeEnabled()) {
            mSections.synchroniseWithSource();
        } else {
            mSections.refreshSuggestions();
        }

        if (mSiteSection != null) {
            mSiteSection.getTileGroup().onSwitchToForeground(/* trackLoadTasks = */ true);
        }
    }

    public int getAboveTheFoldPosition() {
        if (mAboveTheFoldView == null) return RecyclerView.NO_POSITION;

        return getChildPositionOffset(mAboveTheFold);
    }

    public int getFirstHeaderPosition() {
        return getFirstPositionForType(ItemViewType.HEADER);
    }

    public int getFirstCardPosition() {
        for (int i = 0; i < getItemCount(); ++i) {
            if (CardViewHolder.isCard(getItemViewType(i))) return i;
        }
        return RecyclerView.NO_POSITION;
    }

    int getLastContentItemPosition() {
        int bottomSpacerPosition = getChildPositionOffset(mBottomSpacer);
        assert bottomSpacerPosition > 0;
        return bottomSpacerPosition - 1;
    }

    int getBottomSpacerPosition() {
        if (mBottomSpacer == null) return RecyclerView.NO_POSITION;

        return getChildPositionOffset(mBottomSpacer);
    }

    private void updateAllDismissedVisibility() {
        boolean areRemoteSuggestionsEnabled =
                mUiDelegate.getSuggestionsSource().areRemoteSuggestionsEnabled();
        boolean hasAllBeenDismissed = hasAllBeenDismissed();

        mAllDismissed.setVisible(areRemoteSuggestionsEnabled && hasAllBeenDismissed);
        mFooter.setVisible(!SuggestionsConfig.scrollToLoad() && areRemoteSuggestionsEnabled
                && !hasAllBeenDismissed);

        if (mBottomSpacer != null) {
            mBottomSpacer.setVisible(areRemoteSuggestionsEnabled || !hasAllBeenDismissed);
        }
    }

    @Override
    public void onItemRangeChanged(TreeNode child, int itemPosition, int itemCount,
            @Nullable PartialBindCallback callback) {
        assert child == mRoot;
        notifyItemRangeChanged(itemPosition, itemCount, callback);
    }

    @Override
    public void onItemRangeInserted(TreeNode child, int itemPosition, int itemCount) {
        assert child == mRoot;
        notifyItemRangeInserted(itemPosition, itemCount);
        if (mBottomSpacer != null) mBottomSpacer.refresh();
        if (mRecyclerView != null && FeatureUtilities.isChromeHomeModernEnabled()
                && mSections.hasRecentlyInsertedContent()) {
            mRecyclerView.highlightContentLength();
        }

        updateAllDismissedVisibility();
    }

    @Override
    public void onItemRangeRemoved(TreeNode child, int itemPosition, int itemCount) {
        assert child == mRoot;
        notifyItemRangeRemoved(itemPosition, itemCount);
        if (mBottomSpacer != null) mBottomSpacer.refresh();

        updateAllDismissedVisibility();
    }

    @Override
    public void onAttachedToRecyclerView(RecyclerView recyclerView) {
        super.onAttachedToRecyclerView(recyclerView);

        if (mRecyclerView == recyclerView) return;

        // We are assuming for now that the adapter is used with a single RecyclerView.
        // Getting the reference as we are doing here is going to be broken if that changes.
        assert mRecyclerView == null;

        mRecyclerView = (SuggestionsRecyclerView) recyclerView;

        if (SuggestionsConfig.scrollToLoad()) {
            mRecyclerView.addOnScrollListener(new ScrollToLoadListener(
                    this, mRecyclerView.getLinearLayoutManager(), mSections));
        }
    }

    @Override
    public void onViewRecycled(NewTabPageViewHolder holder) {
        holder.recycle();
    }

    /**
     * @return the set of item positions that should be dismissed simultaneously when dismissing the
     *         item at the given {@code position} (including the position itself), or an empty set
     *         if the item can't be dismissed.
     */
    public Set<Integer> getItemDismissalGroup(int position) {
        return mRoot.getItemDismissalGroup(position);
    }

    /**
     * Dismisses the item at the provided adapter position. Can also cause the dismissal of other
     * items or even entire sections.
     * @param position the position of an item to be dismissed.
     * @param itemRemovedCallback
     */
    public void dismissItem(int position, Callback<String> itemRemovedCallback) {
        mRoot.dismissItem(position, itemRemovedCallback);
    }

    private boolean hasAllBeenDismissed() {
        if (mSigninPromo.isVisible()) return false;

        if (!FeatureUtilities.isChromeHomeModernEnabled()) return mSections.isEmpty();

        // In the modern layout, we only consider articles.
        SuggestionsSection suggestions = mSections.getSection(KnownCategories.ARTICLES);
        return suggestions == null || !suggestions.hasSuggestions();
    }

    private int getChildPositionOffset(TreeNode child) {
        return mRoot.getStartingOffsetForChild(child);
    }

    @VisibleForTesting
    public int getFirstPositionForType(@ItemViewType int viewType) {
        int count = getItemCount();
        for (int i = 0; i < count; i++) {
            if (getItemViewType(i) == viewType) return i;
        }
        return RecyclerView.NO_POSITION;
    }

    SectionList getSectionListForTesting() {
        return mSections;
    }

    public InnerNode getRootForTesting() {
        return mRoot;
    }

    private class RemoteSuggestionsStatusObserver
            extends SuggestionsSource.EmptyObserver implements DestructionObserver {
        public RemoteSuggestionsStatusObserver() {
            mUiDelegate.getSuggestionsSource().addObserver(this);
        }

        @Override
        public void onCategoryStatusChanged(
                @CategoryInt int category, @CategoryStatus int newStatus) {
            if (!SnippetsBridge.isCategoryRemote(category)) return;

            updateAllDismissedVisibility();
        }

        @Override
        public void onDestroy() {
            mUiDelegate.getSuggestionsSource().removeObserver(this);
        }
    }
}
