// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_COMMON_MEDIA_ROUTER_MOJO_MEDIA_ROUTER_STRUCT_TRAITS_H_
#define CHROME_COMMON_MEDIA_ROUTER_MOJO_MEDIA_ROUTER_STRUCT_TRAITS_H_

#include <string>

#include "chrome/common/media_router/discovery/media_sink_internal.h"
#include "chrome/common/media_router/issue.h"
#include "chrome/common/media_router/mojo/media_router.mojom.h"
#include "chrome/common/media_router/route_request_result.h"
#include "mojo/common/common_custom_types_struct_traits.h"
#include "net/base/ip_endpoint.h"

namespace mojo {

// Issue

template <>
struct EnumTraits<media_router::mojom::Issue::ActionType,
                  media_router::IssueInfo::Action> {
  static media_router::mojom::Issue::ActionType ToMojom(
      media_router::IssueInfo::Action action) {
    switch (action) {
      case media_router::IssueInfo::Action::DISMISS:
        return media_router::mojom::Issue::ActionType::DISMISS;
      case media_router::IssueInfo::Action::LEARN_MORE:
        return media_router::mojom::Issue::ActionType::LEARN_MORE;
    }
    NOTREACHED() << "Unknown issue action type " << static_cast<int>(action);
    return media_router::mojom::Issue::ActionType::DISMISS;
  }

  static bool FromMojom(media_router::mojom::Issue::ActionType input,
                        media_router::IssueInfo::Action* output) {
    switch (input) {
      case media_router::mojom::Issue::ActionType::DISMISS:
        *output = media_router::IssueInfo::Action::DISMISS;
        return true;
      case media_router::mojom::Issue::ActionType::LEARN_MORE:
        *output = media_router::IssueInfo::Action::LEARN_MORE;
        return true;
    }
    return false;
  }
};

template <>
struct EnumTraits<media_router::mojom::Issue::Severity,
                  media_router::IssueInfo::Severity> {
  static media_router::mojom::Issue::Severity ToMojom(
      media_router::IssueInfo::Severity severity) {
    switch (severity) {
      case media_router::IssueInfo::Severity::FATAL:
        return media_router::mojom::Issue::Severity::FATAL;
      case media_router::IssueInfo::Severity::WARNING:
        return media_router::mojom::Issue::Severity::WARNING;
      case media_router::IssueInfo::Severity::NOTIFICATION:
        return media_router::mojom::Issue::Severity::NOTIFICATION;
    }
    NOTREACHED() << "Unknown issue severity " << static_cast<int>(severity);
    return media_router::mojom::Issue::Severity::WARNING;
  }

  static bool FromMojom(media_router::mojom::Issue::Severity input,
                        media_router::IssueInfo::Severity* output) {
    switch (input) {
      case media_router::mojom::Issue::Severity::FATAL:
        *output = media_router::IssueInfo::Severity::FATAL;
        return true;
      case media_router::mojom::Issue::Severity::WARNING:
        *output = media_router::IssueInfo::Severity::WARNING;
        return true;
      case media_router::mojom::Issue::Severity::NOTIFICATION:
        *output = media_router::IssueInfo::Severity::NOTIFICATION;
        return true;
    }
    return false;
  }
};

template <>
struct UnionTraits<media_router::mojom::MediaSinkExtraDataDataView,
                   media_router::MediaSinkInternal> {
  static media_router::mojom::MediaSinkExtraDataDataView::Tag GetTag(
      const media_router::MediaSinkInternal& sink);

  static bool IsNull(const media_router::MediaSinkInternal& sink) {
    return !sink.is_cast_sink() && !sink.is_dial_sink();
  }

  static void SetToNull(media_router::MediaSinkInternal* out) {}

  static const media_router::DialSinkExtraData& dial_media_sink(
      const media_router::MediaSinkInternal& sink) {
    return sink.dial_data();
  }

  static const media_router::CastSinkExtraData& cast_media_sink(
      const media_router::MediaSinkInternal& sink) {
    return sink.cast_data();
  }

  static bool Read(media_router::mojom::MediaSinkExtraDataDataView data,
                   media_router::MediaSinkInternal* out);
};

template <>
struct StructTraits<media_router::mojom::DialMediaSinkDataView,
                    media_router::DialSinkExtraData> {
  static const std::string& model_name(
      const media_router::DialSinkExtraData& extra_data) {
    return extra_data.model_name;
  }

  static const net::IPAddress& ip_address(
      const media_router::DialSinkExtraData& extra_data) {
    return extra_data.ip_address;
  }

  static const GURL& app_url(
      const media_router::DialSinkExtraData& extra_data) {
    return extra_data.app_url;
  }

  static bool Read(media_router::mojom::DialMediaSinkDataView data,
                   media_router::DialSinkExtraData* out);
};

template <>
struct StructTraits<media_router::mojom::CastMediaSinkDataView,
                    media_router::CastSinkExtraData> {
  static const std::string& model_name(
      const media_router::CastSinkExtraData& extra_data) {
    return extra_data.model_name;
  }

  static const net::IPEndPoint& ip_endpoint(
      const media_router::CastSinkExtraData& extra_data) {
    return extra_data.ip_endpoint;
  }

  static uint8_t capabilities(
      const media_router::CastSinkExtraData& extra_data) {
    return extra_data.capabilities;
  }

  static int32_t cast_channel_id(
      const media_router::CastSinkExtraData& extra_data) {
    return extra_data.cast_channel_id;
  }

  static bool Read(media_router::mojom::CastMediaSinkDataView data,
                   media_router::CastSinkExtraData* out);
};

template <>
struct StructTraits<media_router::mojom::RouteMessageDataView,
                    content::PresentationConnectionMessage> {
  static media_router::mojom::RouteMessage::Type type(
      const content::PresentationConnectionMessage& msg) {
    if (msg.message)
      return media_router::mojom::RouteMessage::Type::TEXT;
    else if (msg.data)
      return media_router::mojom::RouteMessage::Type::BINARY;
    NOTREACHED();
    return media_router::mojom::RouteMessage::Type::TEXT;
  }

  static const base::Optional<std::string>& message(
      const content::PresentationConnectionMessage& msg) {
    return msg.message;
  }

  static const base::Optional<std::vector<uint8_t>>& data(
      const content::PresentationConnectionMessage& msg) {
    return msg.data;
  }

  static bool Read(media_router::mojom::RouteMessageDataView data,
                   content::PresentationConnectionMessage* out) {
    media_router::mojom::RouteMessage::Type type;
    if (!data.ReadType(&type))
      return false;
    switch (type) {
      case media_router::mojom::RouteMessage::Type::TEXT: {
        if (!data.ReadMessage(&out->message) || !out->message)
          return false;
        break;
      }
      case media_router::mojom::RouteMessage::Type::BINARY: {
        if (!data.ReadData(&out->data) || !out->data)
          return false;
        break;
      }
    }
    return true;
  }
};

template <>
struct StructTraits<media_router::mojom::IssueDataView,
                    media_router::IssueInfo> {
  static bool Read(media_router::mojom::IssueDataView data,
                   media_router::IssueInfo* out);

  static base::Optional<std::string> route_id(
      const media_router::IssueInfo& issue) {
    return issue.route_id.empty() ? base::Optional<std::string>()
                                  : base::make_optional(issue.route_id);
  }

  static media_router::IssueInfo::Severity severity(
      const media_router::IssueInfo& issue) {
    return issue.severity;
  }

  static bool is_blocking(const media_router::IssueInfo& issue) {
    return issue.is_blocking;
  }

  static std::string title(const media_router::IssueInfo& issue) {
    return issue.title;
  }

  static base::Optional<std::string> message(
      const media_router::IssueInfo& issue) {
    return issue.message.empty() ? base::Optional<std::string>()
                                 : base::make_optional(issue.message);
  }

  static media_router::IssueInfo::Action default_action(
      const media_router::IssueInfo& issue) {
    return issue.default_action;
  }

  static base::Optional<std::vector<media_router::IssueInfo::Action>>
  secondary_actions(const media_router::IssueInfo& issue) {
    return issue.secondary_actions;
  }

  static int32_t help_page_id(const media_router::IssueInfo& issue) {
    return issue.help_page_id;
  }
};

// MediaSink

template <>
struct EnumTraits<media_router::mojom::SinkIconType,
                  media_router::SinkIconType> {
  static media_router::mojom::SinkIconType ToMojom(
      media_router::SinkIconType icon_type) {
    switch (icon_type) {
      case media_router::SinkIconType::CAST:
        return media_router::mojom::SinkIconType::CAST;
      case media_router::SinkIconType::CAST_AUDIO_GROUP:
        return media_router::mojom::SinkIconType::CAST_AUDIO_GROUP;
      case media_router::SinkIconType::CAST_AUDIO:
        return media_router::mojom::SinkIconType::CAST_AUDIO;
      case media_router::SinkIconType::MEETING:
        return media_router::mojom::SinkIconType::MEETING;
      case media_router::SinkIconType::HANGOUT:
        return media_router::mojom::SinkIconType::HANGOUT;
      case media_router::SinkIconType::EDUCATION:
        return media_router::mojom::SinkIconType::EDUCATION;
      case media_router::SinkIconType::GENERIC:
        return media_router::mojom::SinkIconType::GENERIC;
    }
    NOTREACHED() << "Unknown sink icon type " << static_cast<int>(icon_type);
    return media_router::mojom::SinkIconType::GENERIC;
  }

  static bool FromMojom(media_router::mojom::SinkIconType input,
                        media_router::SinkIconType* output) {
    switch (input) {
      case media_router::mojom::SinkIconType::CAST:
        *output = media_router::SinkIconType::CAST;
        return true;
      case media_router::mojom::SinkIconType::CAST_AUDIO_GROUP:
        *output = media_router::SinkIconType::CAST_AUDIO_GROUP;
        return true;
      case media_router::mojom::SinkIconType::CAST_AUDIO:
        *output = media_router::SinkIconType::CAST_AUDIO;
        return true;
      case media_router::mojom::SinkIconType::MEETING:
        *output = media_router::SinkIconType::MEETING;
        return true;
      case media_router::mojom::SinkIconType::HANGOUT:
        *output = media_router::SinkIconType::HANGOUT;
        return true;
      case media_router::mojom::SinkIconType::EDUCATION:
        *output = media_router::SinkIconType::EDUCATION;
        return true;
      case media_router::mojom::SinkIconType::GENERIC:
        *output = media_router::SinkIconType::GENERIC;
        return true;
    }
    return false;
  }
};

template <>
struct StructTraits<media_router::mojom::MediaSinkDataView,
                    media_router::MediaSinkInternal> {
  static bool Read(media_router::mojom::MediaSinkDataView data,
                   media_router::MediaSinkInternal* out);

  static std::string sink_id(
      const media_router::MediaSinkInternal& sink_internal) {
    return sink_internal.sink().id();
  }

  static std::string name(
      const media_router::MediaSinkInternal& sink_internal) {
    return sink_internal.sink().name();
  }

  static base::Optional<std::string> description(
      const media_router::MediaSinkInternal& sink_internal) {
    return sink_internal.sink().description();
  }

  static base::Optional<std::string> domain(
      const media_router::MediaSinkInternal& sink_internal) {
    return sink_internal.sink().domain();
  }

  static media_router::SinkIconType icon_type(
      const media_router::MediaSinkInternal& sink_internal) {
    return sink_internal.sink().icon_type();
  }

  static const media_router::MediaSinkInternal& extra_data(
      const media_router::MediaSinkInternal& sink_internal) {
    return sink_internal;
  }
};

// MediaRoute

template <>
struct EnumTraits<media_router::mojom::RouteControllerType,
                  media_router::RouteControllerType> {
  static media_router::mojom::RouteControllerType ToMojom(
      media_router::RouteControllerType controller_type) {
    switch (controller_type) {
      case media_router::RouteControllerType::kNone:
        return media_router::mojom::RouteControllerType::kNone;
      case media_router::RouteControllerType::kGeneric:
        return media_router::mojom::RouteControllerType::kGeneric;
      case media_router::RouteControllerType::kHangouts:
        return media_router::mojom::RouteControllerType::kHangouts;
    }
    NOTREACHED() << "Unknown controller type "
                 << static_cast<int>(controller_type);
    return media_router::mojom::RouteControllerType::kNone;
  }

  static bool FromMojom(media_router::mojom::RouteControllerType input,
                        media_router::RouteControllerType* output) {
    switch (input) {
      case media_router::mojom::RouteControllerType::kNone:
        *output = media_router::RouteControllerType::kNone;
        return true;
      case media_router::mojom::RouteControllerType::kGeneric:
        *output = media_router::RouteControllerType::kGeneric;
        return true;
      case media_router::mojom::RouteControllerType::kHangouts:
        *output = media_router::RouteControllerType::kHangouts;
        return true;
    }
    return false;
  }
};

template <>
struct StructTraits<media_router::mojom::MediaRouteDataView,
                    media_router::MediaRoute> {
  static bool Read(media_router::mojom::MediaRouteDataView data,
                   media_router::MediaRoute* out);

  static std::string media_route_id(const media_router::MediaRoute& route) {
    return route.media_route_id();
  }

  static base::Optional<std::string> media_source(
      const media_router::MediaRoute& route) {
    // TODO(imcheng): If we ever convert from Mojo to C++ outside of unit tests,
    // it would be better to make the |media_source_| field on MediaRoute a
    // base::Optional<MediaSource::Id> instead so it can be returned directly
    // here.
    return route.media_source().id().empty()
               ? base::Optional<std::string>()
               : base::make_optional(route.media_source().id());
  }

  static std::string media_sink_id(const media_router::MediaRoute& route) {
    return route.media_sink_id();
  }

  static std::string description(const media_router::MediaRoute& route) {
    return route.description();
  }

  static bool is_local(const media_router::MediaRoute& route) {
    return route.is_local();
  }

  static base::Optional<std::string> custom_controller_path(
      const media_router::MediaRoute& route) {
    return route.custom_controller_path().empty()
               ? base::Optional<std::string>()
               : base::make_optional(route.custom_controller_path());
  }

  static bool supports_media_route_controller(
      const media_router::MediaRoute& route) {
    return route.supports_media_route_controller();
  }

  static media_router::RouteControllerType controller_type(
      const media_router::MediaRoute& route) {
    return route.controller_type();
  }

  static bool for_display(const media_router::MediaRoute& route) {
    return route.for_display();
  }

  static bool is_incognito(const media_router::MediaRoute& route) {
    return route.is_incognito();
  }

  static bool is_offscreen_presentation(const media_router::MediaRoute& route) {
    return route.is_offscreen_presentation();
  }
};

// PresentationConnectionState

template <>
struct EnumTraits<media_router::mojom::MediaRouter::PresentationConnectionState,
                  content::PresentationConnectionState> {
  static media_router::mojom::MediaRouter::PresentationConnectionState ToMojom(
      content::PresentationConnectionState state) {
    switch (state) {
      case content::PRESENTATION_CONNECTION_STATE_CONNECTING:
        return media_router::mojom::MediaRouter::PresentationConnectionState::
            CONNECTING;
      case content::PRESENTATION_CONNECTION_STATE_CONNECTED:
        return media_router::mojom::MediaRouter::PresentationConnectionState::
            CONNECTED;
      case content::PRESENTATION_CONNECTION_STATE_CLOSED:
        return media_router::mojom::MediaRouter::PresentationConnectionState::
            CLOSED;
      case content::PRESENTATION_CONNECTION_STATE_TERMINATED:
        return media_router::mojom::MediaRouter::PresentationConnectionState::
            TERMINATED;
    }
    NOTREACHED() << "Unknown PresentationConnectionState "
                 << static_cast<int>(state);
    return media_router::mojom::MediaRouter::PresentationConnectionState::
        TERMINATED;
  }

  static bool FromMojom(
      media_router::mojom::MediaRouter::PresentationConnectionState input,
      content::PresentationConnectionState* state) {
    switch (input) {
      case media_router::mojom::MediaRouter::PresentationConnectionState::
          CONNECTING:
        *state = content::PRESENTATION_CONNECTION_STATE_CONNECTING;
        return true;
      case media_router::mojom::MediaRouter::PresentationConnectionState::
          CONNECTED:
        *state = content::PRESENTATION_CONNECTION_STATE_CONNECTED;
        return true;
      case media_router::mojom::MediaRouter::PresentationConnectionState::
          CLOSED:
        *state = content::PRESENTATION_CONNECTION_STATE_CLOSED;
        return true;
      case media_router::mojom::MediaRouter::PresentationConnectionState::
          TERMINATED:
        *state = content::PRESENTATION_CONNECTION_STATE_TERMINATED;
        return true;
    }
    return false;
  }
};

// PresentationConnectionCloseReason

template <>
struct EnumTraits<
    media_router::mojom::MediaRouter::PresentationConnectionCloseReason,
    content::PresentationConnectionCloseReason> {
  static media_router::mojom::MediaRouter::PresentationConnectionCloseReason
  ToMojom(content::PresentationConnectionCloseReason reason) {
    switch (reason) {
      case content::PRESENTATION_CONNECTION_CLOSE_REASON_CONNECTION_ERROR:
        return media_router::mojom::MediaRouter::
            PresentationConnectionCloseReason::CONNECTION_ERROR;
      case content::PRESENTATION_CONNECTION_CLOSE_REASON_CLOSED:
        return media_router::mojom::MediaRouter::
            PresentationConnectionCloseReason::CLOSED;
      case content::PRESENTATION_CONNECTION_CLOSE_REASON_WENT_AWAY:
        return media_router::mojom::MediaRouter::
            PresentationConnectionCloseReason::WENT_AWAY;
    }
    NOTREACHED() << "Unknown PresentationConnectionCloseReason "
                 << static_cast<int>(reason);
    return media_router::mojom::MediaRouter::PresentationConnectionCloseReason::
        CONNECTION_ERROR;
  }

  static bool FromMojom(
      media_router::mojom::MediaRouter::PresentationConnectionCloseReason input,
      content::PresentationConnectionCloseReason* state) {
    switch (input) {
      case media_router::mojom::MediaRouter::PresentationConnectionCloseReason::
          CONNECTION_ERROR:
        *state = content::PRESENTATION_CONNECTION_CLOSE_REASON_CONNECTION_ERROR;
        return true;
      case media_router::mojom::MediaRouter::PresentationConnectionCloseReason::
          CLOSED:
        *state = content::PRESENTATION_CONNECTION_CLOSE_REASON_CLOSED;
        return true;
      case media_router::mojom::MediaRouter::PresentationConnectionCloseReason::
          WENT_AWAY:
        *state = content::PRESENTATION_CONNECTION_CLOSE_REASON_WENT_AWAY;
        return true;
    }
    return false;
  }
};

// RouteRequestResultCode

template <>
struct EnumTraits<media_router::mojom::RouteRequestResultCode,
                  media_router::RouteRequestResult::ResultCode> {
  static media_router::mojom::RouteRequestResultCode ToMojom(
      media_router::RouteRequestResult::ResultCode code) {
    switch (code) {
      case media_router::RouteRequestResult::UNKNOWN_ERROR:
        return media_router::mojom::RouteRequestResultCode::UNKNOWN_ERROR;
      case media_router::RouteRequestResult::OK:
        return media_router::mojom::RouteRequestResultCode::OK;
      case media_router::RouteRequestResult::TIMED_OUT:
        return media_router::mojom::RouteRequestResultCode::TIMED_OUT;
      case media_router::RouteRequestResult::ROUTE_NOT_FOUND:
        return media_router::mojom::RouteRequestResultCode::ROUTE_NOT_FOUND;
      case media_router::RouteRequestResult::SINK_NOT_FOUND:
        return media_router::mojom::RouteRequestResultCode::SINK_NOT_FOUND;
      case media_router::RouteRequestResult::INVALID_ORIGIN:
        return media_router::mojom::RouteRequestResultCode::INVALID_ORIGIN;
      case media_router::RouteRequestResult::INCOGNITO_MISMATCH:
        return media_router::mojom::RouteRequestResultCode::INCOGNITO_MISMATCH;
      case media_router::RouteRequestResult::NO_SUPPORTED_PROVIDER:
        return media_router::mojom::RouteRequestResultCode::
            NO_SUPPORTED_PROVIDER;
      case media_router::RouteRequestResult::CANCELLED:
        return media_router::mojom::RouteRequestResultCode::CANCELLED;
      default:
        NOTREACHED() << "Unknown RouteRequestResultCode "
                     << static_cast<int>(code);
        return media_router::mojom::RouteRequestResultCode::UNKNOWN_ERROR;
    }
  }

  static bool FromMojom(media_router::mojom::RouteRequestResultCode input,
                        media_router::RouteRequestResult::ResultCode* output) {
    switch (input) {
      case media_router::mojom::RouteRequestResultCode::UNKNOWN_ERROR:
        *output = media_router::RouteRequestResult::UNKNOWN_ERROR;
        return true;
      case media_router::mojom::RouteRequestResultCode::OK:
        *output = media_router::RouteRequestResult::OK;
        return true;
      case media_router::mojom::RouteRequestResultCode::TIMED_OUT:
        *output = media_router::RouteRequestResult::TIMED_OUT;
        return true;
      case media_router::mojom::RouteRequestResultCode::ROUTE_NOT_FOUND:
        *output = media_router::RouteRequestResult::ROUTE_NOT_FOUND;
        return true;
      case media_router::mojom::RouteRequestResultCode::SINK_NOT_FOUND:
        *output = media_router::RouteRequestResult::SINK_NOT_FOUND;
        return true;
      case media_router::mojom::RouteRequestResultCode::INVALID_ORIGIN:
        *output = media_router::RouteRequestResult::INVALID_ORIGIN;
        return true;
      case media_router::mojom::RouteRequestResultCode::INCOGNITO_MISMATCH:
        *output = media_router::RouteRequestResult::INCOGNITO_MISMATCH;
        return true;
      case media_router::mojom::RouteRequestResultCode::NO_SUPPORTED_PROVIDER:
        *output = media_router::RouteRequestResult::NO_SUPPORTED_PROVIDER;
        return true;
      case media_router::mojom::RouteRequestResultCode::CANCELLED:
        *output = media_router::RouteRequestResult::CANCELLED;
        return true;
    }
    return false;
  }
};

}  // namespace mojo

#endif  // CHROME_COMMON_MEDIA_ROUTER_MOJO_MEDIA_ROUTER_STRUCT_TRAITS_H_
