// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "net/cert/ev_root_ca_metadata.h"

#if defined(USE_NSS_CERTS)
#include <cert.h>
#include <pkcs11n.h>
#include <secerr.h>
#include <secoid.h>
#elif defined(OS_WIN)
#include <stdlib.h>
#endif

#include "base/lazy_instance.h"
#include "base/logging.h"
#include "net/der/input.h"
#if defined(USE_NSS_CERTS)
#include "crypto/nss_util.h"
#elif defined(PLATFORM_USES_CHROMIUM_EV_METADATA)
#include "third_party/boringssl/src/include/openssl/asn1.h"
#include "third_party/boringssl/src/include/openssl/obj.h"
#endif

namespace net {

#if defined(PLATFORM_USES_CHROMIUM_EV_METADATA)
// Raw metadata.
struct EVMetadata {
  // kMaxOIDsPerCA is the number of OIDs that we can support per root CA. At
  // least one CA has different EV policies for business vs government
  // entities and, in the case of cross-signing, we might need to list another
  // CA's policy OID under the cross-signing root.
  static const size_t kMaxOIDsPerCA = 2;
  // This is the maximum length of an OID string (including the trailing NUL).
  static const size_t kMaxOIDLength = 32;

  // The SHA-256 fingerprint of the root CA certificate, used as a unique
  // identifier for a root CA certificate.
  SHA256HashValue fingerprint;

  // The EV policy OIDs of the root CA.
  char policy_oids[kMaxOIDsPerCA][kMaxOIDLength];
};

// These certificates may be found in net/data/ssl/ev_roots.
static const EVMetadata ev_root_ca_metadata[] = {
    // AC Camerfirma S.A. Chambers of Commerce Root - 2008
    // https://www.camerfirma.com
    {
        {{0x06, 0x3e, 0x4a, 0xfa, 0xc4, 0x91, 0xdf, 0xd3, 0x32, 0xf3, 0x08,
          0x9b, 0x85, 0x42, 0xe9, 0x46, 0x17, 0xd8, 0x93, 0xd7, 0xfe, 0x94,
          0x4e, 0x10, 0xa7, 0x93, 0x7e, 0xe2, 0x9d, 0x96, 0x93, 0xc0}},
        {
            // AC Camerfirma uses the last two arcs to track how the private key
            // is managed - the effective verification policy is the same.
            "1.3.6.1.4.1.17326.10.14.2.1.2", "1.3.6.1.4.1.17326.10.14.2.2.2",
        },
    },
    // AC Camerfirma S.A. Global Chambersign Root - 2008
    // https://server2.camerfirma.com:8082
    {
        {{0x13, 0x63, 0x35, 0x43, 0x93, 0x34, 0xa7, 0x69, 0x80, 0x16, 0xa0,
          0xd3, 0x24, 0xde, 0x72, 0x28, 0x4e, 0x07, 0x9d, 0x7b, 0x52, 0x20,
          0xbb, 0x8f, 0xbd, 0x74, 0x78, 0x16, 0xee, 0xbe, 0xba, 0xca}},
        {
            // AC Camerfirma uses the last two arcs to track how the private key
            // is managed - the effective verification policy is the same.
            "1.3.6.1.4.1.17326.10.8.12.1.2", "1.3.6.1.4.1.17326.10.8.12.2.2",
        },
    },
    // AddTrust External CA Root
    // https://addtrustexternalcaroot-ev.comodoca.com
    {
        {{0x68, 0x7f, 0xa4, 0x51, 0x38, 0x22, 0x78, 0xff, 0xf0, 0xc8, 0xb1,
          0x1f, 0x8d, 0x43, 0xd5, 0x76, 0x67, 0x1c, 0x6e, 0xb2, 0xbc, 0xea,
          0xb4, 0x13, 0xfb, 0x83, 0xd9, 0x65, 0xd0, 0x6d, 0x2f, 0xf2}},
        {
            "1.3.6.1.4.1.6449.1.2.1.5.1",
            // This is the Network Solutions EV OID. However, this root
            // cross-certifies NetSol and so we need it here too.
            "1.3.6.1.4.1.782.1.2.1.8.1",
        },
    },
    // Actalis Authentication Root CA
    // https://ssltest-a.actalis.it:8443
    {
        {{0x55, 0x92, 0x60, 0x84, 0xec, 0x96, 0x3a, 0x64, 0xb9, 0x6e, 0x2a,
          0xbe, 0x01, 0xce, 0x0b, 0xa8, 0x6a, 0x64, 0xfb, 0xfe, 0xbc, 0xc7,
          0xaa, 0xb5, 0xaf, 0xc1, 0x55, 0xb3, 0x7f, 0xd7, 0x60, 0x66}},
        {"1.3.159.1.17.1", ""},
    },
    // AffirmTrust Commercial
    // https://commercial.affirmtrust.com/
    {
        {{0x03, 0x76, 0xab, 0x1d, 0x54, 0xc5, 0xf9, 0x80, 0x3c, 0xe4, 0xb2,
          0xe2, 0x01, 0xa0, 0xee, 0x7e, 0xef, 0x7b, 0x57, 0xb6, 0x36, 0xe8,
          0xa9, 0x3c, 0x9b, 0x8d, 0x48, 0x60, 0xc9, 0x6f, 0x5f, 0xa7}},
        {"1.3.6.1.4.1.34697.2.1", ""},
    },
    // AffirmTrust Networking
    // https://networking.affirmtrust.com:4431
    {
        {{0x0a, 0x81, 0xec, 0x5a, 0x92, 0x97, 0x77, 0xf1, 0x45, 0x90, 0x4a,
          0xf3, 0x8d, 0x5d, 0x50, 0x9f, 0x66, 0xb5, 0xe2, 0xc5, 0x8f, 0xcd,
          0xb5, 0x31, 0x05, 0x8b, 0x0e, 0x17, 0xf3, 0xf0, 0xb4, 0x1b}},
        {"1.3.6.1.4.1.34697.2.2", ""},
    },
    // AffirmTrust Premium
    // https://premium.affirmtrust.com:4432/
    {
        {{0x70, 0xa7, 0x3f, 0x7f, 0x37, 0x6b, 0x60, 0x07, 0x42, 0x48, 0x90,
          0x45, 0x34, 0xb1, 0x14, 0x82, 0xd5, 0xbf, 0x0e, 0x69, 0x8e, 0xcc,
          0x49, 0x8d, 0xf5, 0x25, 0x77, 0xeb, 0xf2, 0xe9, 0x3b, 0x9a}},
        {"1.3.6.1.4.1.34697.2.3", ""},
    },
    // AffirmTrust Premium ECC
    // https://premiumecc.affirmtrust.com:4433/
    {
        {{0xbd, 0x71, 0xfd, 0xf6, 0xda, 0x97, 0xe4, 0xcf, 0x62, 0xd1, 0x64,
          0x7a, 0xdd, 0x25, 0x81, 0xb0, 0x7d, 0x79, 0xad, 0xf8, 0x39, 0x7e,
          0xb4, 0xec, 0xba, 0x9c, 0x5e, 0x84, 0x88, 0x82, 0x14, 0x23}},
        {"1.3.6.1.4.1.34697.2.4", ""},
    },
    // Amazon Root CA 1
    // https://good.sca1a.amazontrust.com/
    {
        {{0x8e, 0xcd, 0xe6, 0x88, 0x4f, 0x3d, 0x87, 0xb1, 0x12, 0x5b, 0xa3,
          0x1a, 0xc3, 0xfc, 0xb1, 0x3d, 0x70, 0x16, 0xde, 0x7f, 0x57, 0xcc,
          0x90, 0x4f, 0xe1, 0xcb, 0x97, 0xc6, 0xae, 0x98, 0x19, 0x6e}},
        {"2.23.140.1.1", ""},
    },
    // Amazon Root CA 2
    // https://good.sca2a.amazontrust.com/
    {
        {{0x1b, 0xa5, 0xb2, 0xaa, 0x8c, 0x65, 0x40, 0x1a, 0x82, 0x96, 0x01,
          0x18, 0xf8, 0x0b, 0xec, 0x4f, 0x62, 0x30, 0x4d, 0x83, 0xce, 0xc4,
          0x71, 0x3a, 0x19, 0xc3, 0x9c, 0x01, 0x1e, 0xa4, 0x6d, 0xb4}},
        {"2.23.140.1.1", ""},
    },
    // Amazon Root CA 3
    // https://good.sca3a.amazontrust.com/
    {
        {{0x18, 0xce, 0x6c, 0xfe, 0x7b, 0xf1, 0x4e, 0x60, 0xb2, 0xe3, 0x47,
          0xb8, 0xdf, 0xe8, 0x68, 0xcb, 0x31, 0xd0, 0x2e, 0xbb, 0x3a, 0xda,
          0x27, 0x15, 0x69, 0xf5, 0x03, 0x43, 0xb4, 0x6d, 0xb3, 0xa4}},
        {"2.23.140.1.1", ""},
    },
    // Amazon Root CA 4
    // https://good.sca4a.amazontrust.com/
    {
        {{0xe3, 0x5d, 0x28, 0x41, 0x9e, 0xd0, 0x20, 0x25, 0xcf, 0xa6, 0x90,
          0x38, 0xcd, 0x62, 0x39, 0x62, 0x45, 0x8d, 0xa5, 0xc6, 0x95, 0xfb,
          0xde, 0xa3, 0xc2, 0x2b, 0x0b, 0xfb, 0x25, 0x89, 0x70, 0x92}},
        {"2.23.140.1.1", ""},
    },
    // Autoridad de Certificacion Firmaprofesional CIF A62634068
    // https://publifirma.firmaprofesional.com/
    {
        {{0x04, 0x04, 0x80, 0x28, 0xbf, 0x1f, 0x28, 0x64, 0xd4, 0x8f, 0x9a,
          0xd4, 0xd8, 0x32, 0x94, 0x36, 0x6a, 0x82, 0x88, 0x56, 0x55, 0x3f,
          0x3b, 0x14, 0x30, 0x3f, 0x90, 0x14, 0x7f, 0x5d, 0x40, 0xef}},
        {"1.3.6.1.4.1.13177.10.1.3.10", ""},
    },
    // Baltimore CyberTrust Root
    // https://secure.omniroot.com/repository/
    {
        {{0x16, 0xaf, 0x57, 0xa9, 0xf6, 0x76, 0xb0, 0xab, 0x12, 0x60, 0x95,
          0xaa, 0x5e, 0xba, 0xde, 0xf2, 0x2a, 0xb3, 0x11, 0x19, 0xd6, 0x44,
          0xac, 0x95, 0xcd, 0x4b, 0x93, 0xdb, 0xf3, 0xf2, 0x6a, 0xeb}},
        {"1.3.6.1.4.1.6334.1.100.1", ""},
    },
    // Buypass Class 3 CA 1
    // https://valid.evident.ca13.ssl.buypass.no/
    {
        {{0xb7, 0xb1, 0x2b, 0x17, 0x1f, 0x82, 0x1d, 0xaa, 0x99, 0x0c, 0xd0,
          0xfe, 0x50, 0x87, 0xb1, 0x28, 0x44, 0x8b, 0xa8, 0xe5, 0x18, 0x4f,
          0x84, 0xc5, 0x1e, 0x02, 0xb5, 0xc8, 0xfb, 0x96, 0x2b, 0x24}},
        {"2.16.578.1.26.1.3.3", ""},
    },
    // Buypass Class 3 Root CA
    // https://valid.evident.ca23.ssl.buypass.no/
    {
        {{0xed, 0xf7, 0xeb, 0xbc, 0xa2, 0x7a, 0x2a, 0x38, 0x4d, 0x38, 0x7b,
          0x7d, 0x40, 0x10, 0xc6, 0x66, 0xe2, 0xed, 0xb4, 0x84, 0x3e, 0x4c,
          0x29, 0xb4, 0xae, 0x1d, 0x5b, 0x93, 0x32, 0xe6, 0xb2, 0x4d}},
        {"2.16.578.1.26.1.3.3", ""},
    },
    // CertPlus Class 2 Primary CA (KEYNECTIS)
    // https://www.keynectis.com/
    {
        {{0x0f, 0x99, 0x3c, 0x8a, 0xef, 0x97, 0xba, 0xaf, 0x56, 0x87, 0x14,
          0x0e, 0xd5, 0x9a, 0xd1, 0x82, 0x1b, 0xb4, 0xaf, 0xac, 0xf0, 0xaa,
          0x9a, 0x58, 0xb5, 0xd5, 0x7a, 0x33, 0x8a, 0x3a, 0xfb, 0xcb}},
        {"1.3.6.1.4.1.22234.2.5.2.3.1", ""},
    },
    // Certum Trusted Network CA
    // https://juice.certum.pl/
    {
        {{0x5c, 0x58, 0x46, 0x8d, 0x55, 0xf5, 0x8e, 0x49, 0x7e, 0x74, 0x39,
          0x82, 0xd2, 0xb5, 0x00, 0x10, 0xb6, 0xd1, 0x65, 0x37, 0x4a, 0xcf,
          0x83, 0xa7, 0xd4, 0xa3, 0x2d, 0xb7, 0x68, 0xc4, 0x40, 0x8e}},
        {"1.2.616.1.113527.2.5.1.1", ""},
    },
    // CFCA EV ROOT
    // https://www.erenepu.com/
    {
        {{0x5c, 0xc3, 0xd7, 0x8e, 0x4e, 0x1d, 0x5e, 0x45, 0x54, 0x7a, 0x04,
          0xe6, 0x87, 0x3e, 0x64, 0xf9, 0x0c, 0xf9, 0x53, 0x6d, 0x1c, 0xcc,
          0x2e, 0xf8, 0x00, 0xf3, 0x55, 0xc4, 0xc5, 0xfd, 0x70, 0xfd}},
        {"2.16.156.112554.3", ""},
    },
    // China Internet Network Information Center EV Certificates Root
    // https://evdemo.cnnic.cn/
    {
        // Root
        {{0x1c, 0x01, 0xc6, 0xf4, 0xdb, 0xb2, 0xfe, 0xfc, 0x22, 0x55, 0x8b,
          0x2b, 0xca, 0x32, 0x56, 0x3f, 0x49, 0x84, 0x4a, 0xcf, 0xc3, 0x2b,
          0x7b, 0xe4, 0xb0, 0xff, 0x59, 0x9f, 0x9e, 0x8c, 0x7a, 0xf7}},
        {"1.3.6.1.4.1.29836.1.10", ""},
    },
    // COMODO Certification Authority
    // https://secure.comodo.com/
    {
        {{0x0c, 0x2c, 0xd6, 0x3d, 0xf7, 0x80, 0x6f, 0xa3, 0x99, 0xed, 0xe8,
          0x09, 0x11, 0x6b, 0x57, 0x5b, 0xf8, 0x79, 0x89, 0xf0, 0x65, 0x18,
          0xf9, 0x80, 0x8c, 0x86, 0x05, 0x03, 0x17, 0x8b, 0xaf, 0x66}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // COMODO Certification Authority (reissued certificate with NotBefore of
    // Jan 1 00:00:00 2011 GMT)
    // https://secure.comodo.com/
    {
        {{0x1a, 0x0d, 0x20, 0x44, 0x5d, 0xe5, 0xba, 0x18, 0x62, 0xd1, 0x9e,
          0xf8, 0x80, 0x85, 0x8c, 0xbc, 0xe5, 0x01, 0x02, 0xb3, 0x6e, 0x8f,
          0x0a, 0x04, 0x0c, 0x3c, 0x69, 0xe7, 0x45, 0x22, 0xfe, 0x6e}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // COMODO ECC Certification Authority
    // https://comodoecccertificationauthority-ev.comodoca.com/
    {
        {{0x17, 0x93, 0x92, 0x7a, 0x06, 0x14, 0x54, 0x97, 0x89, 0xad, 0xce,
          0x2f, 0x8f, 0x34, 0xf7, 0xf0, 0xb6, 0x6d, 0x0f, 0x3a, 0xe3, 0xa3,
          0xb8, 0x4d, 0x21, 0xec, 0x15, 0xdb, 0xba, 0x4f, 0xad, 0xc7}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // COMODO RSA Certification Authority
    // https://comodorsacertificationauthority-ev.comodoca.com/
    {
        {{0x52, 0xf0, 0xe1, 0xc4, 0xe5, 0x8e, 0xc6, 0x29, 0x29, 0x1b, 0x60,
          0x31, 0x7f, 0x07, 0x46, 0x71, 0xb8, 0x5d, 0x7e, 0xa8, 0x0d, 0x5b,
          0x07, 0x27, 0x34, 0x63, 0x53, 0x4b, 0x32, 0xb4, 0x02, 0x34}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // Cybertrust Global Root
    // https://evup.cybertrust.ne.jp/ctj-ev-upgrader/evseal.gif
    {
        {{0x96, 0x0a, 0xdf, 0x00, 0x63, 0xe9, 0x63, 0x56, 0x75, 0x0c, 0x29,
          0x65, 0xdd, 0x0a, 0x08, 0x67, 0xda, 0x0b, 0x9c, 0xbd, 0x6e, 0x77,
          0x71, 0x4a, 0xea, 0xfb, 0x23, 0x49, 0xab, 0x39, 0x3d, 0xa3}},
        {"1.3.6.1.4.1.6334.1.100.1", ""},
    },
    // DigiCert High Assurance EV Root CA
    // https://www.digicert.com
    {
        {{0x74, 0x31, 0xe5, 0xf4, 0xc3, 0xc1, 0xce, 0x46, 0x90, 0x77, 0x4f,
          0x0b, 0x61, 0xe0, 0x54, 0x40, 0x88, 0x3b, 0xa9, 0xa0, 0x1e, 0xd0,
          0x0b, 0xa6, 0xab, 0xd7, 0x80, 0x6e, 0xd3, 0xb1, 0x18, 0xcf}},
        {"2.16.840.1.114412.2.1", ""},
    },
    // D-TRUST Root Class 3 CA 2 EV 2009
    // https://certdemo-ev-valid.ssl.d-trust.net/
    {
        {{0xee, 0xc5, 0x49, 0x6b, 0x98, 0x8c, 0xe9, 0x86, 0x25, 0xb9, 0x34,
          0x09, 0x2e, 0xec, 0x29, 0x08, 0xbe, 0xd0, 0xb0, 0xf3, 0x16, 0xc2,
          0xd4, 0x73, 0x0c, 0x84, 0xea, 0xf1, 0xf3, 0xd3, 0x48, 0x81}},
        {"1.3.6.1.4.1.4788.2.202.1", ""},
    },
    // Entrust.net Secure Server Certification Authority
    // https://www.entrust.net/
    {
        {{0x62, 0xf2, 0x40, 0x27, 0x8c, 0x56, 0x4c, 0x4d, 0xd8, 0xbf, 0x7d,
          0x9d, 0x4f, 0x6f, 0x36, 0x6e, 0xa8, 0x94, 0xd2, 0x2f, 0x5f, 0x34,
          0xd9, 0x89, 0xa9, 0x83, 0xac, 0xec, 0x2f, 0xff, 0xed, 0x50}},
        {"2.16.840.1.114028.10.1.2", ""},
    },
    // Entrust Root Certification Authority
    // https://www.entrust.net/
    {
        {{0x73, 0xc1, 0x76, 0x43, 0x4f, 0x1b, 0xc6, 0xd5, 0xad, 0xf4, 0x5b,
          0x0e, 0x76, 0xe7, 0x27, 0x28, 0x7c, 0x8d, 0xe5, 0x76, 0x16, 0xc1,
          0xe6, 0xe6, 0x14, 0x1a, 0x2b, 0x2c, 0xbc, 0x7d, 0x8e, 0x4c}},
        {"2.16.840.1.114028.10.1.2", ""},
    },
    // Entrust Root Certification Authority – G2
    // https://validg2.entrust.net
    {
        {{0x43, 0xdf, 0x57, 0x74, 0xb0, 0x3e, 0x7f, 0xef, 0x5f, 0xe4, 0x0d,
          0x93, 0x1a, 0x7b, 0xed, 0xf1, 0xbb, 0x2e, 0x6b, 0x42, 0x73, 0x8c,
          0x4e, 0x6d, 0x38, 0x41, 0x10, 0x3d, 0x3a, 0xa7, 0xf3, 0x39}},
        {"2.16.840.1.114028.10.1.2", ""},
    },
    // Entrust Root Certification Authority – EC1
    // https://validec.entrust.net
    {
        {{0x02, 0xed, 0x0e, 0xb2, 0x8c, 0x14, 0xda, 0x45, 0x16, 0x5c, 0x56,
          0x67, 0x91, 0x70, 0x0d, 0x64, 0x51, 0xd7, 0xfb, 0x56, 0xf0, 0xb2,
          0xab, 0x1d, 0x3b, 0x8e, 0xb0, 0x70, 0xe5, 0x6e, 0xdf, 0xf5}},
        {"2.16.840.1.114028.10.1.2", ""},
    },
    // Equifax Secure Certificate Authority (GeoTrust)
    // https://www.geotrust.com/
    {
        {{0x08, 0x29, 0x7a, 0x40, 0x47, 0xdb, 0xa2, 0x36, 0x80, 0xc7, 0x31,
          0xdb, 0x6e, 0x31, 0x76, 0x53, 0xca, 0x78, 0x48, 0xe1, 0xbe, 0xbd,
          0x3a, 0x0b, 0x01, 0x79, 0xa7, 0x07, 0xf9, 0x2c, 0xf1, 0x78}},
        {"1.3.6.1.4.1.14370.1.6", ""},
    },
    // E-Tugra Certification Authority
    // https://sslev.e-tugra.com.tr
    {
        {{0xb0, 0xbf, 0xd5, 0x2b, 0xb0, 0xd7, 0xd9, 0xbd, 0x92, 0xbf, 0x5d,
          0x4d, 0xc1, 0x3d, 0xa2, 0x55, 0xc0, 0x2c, 0x54, 0x2f, 0x37, 0x83,
          0x65, 0xea, 0x89, 0x39, 0x11, 0xf5, 0x5e, 0x55, 0xf2, 0x3c}},
        {"2.16.792.3.0.4.1.1.4", ""},
    },
    // GeoTrust Primary Certification Authority
    // https://www.geotrust.com/
    {
        {{0x37, 0xd5, 0x10, 0x06, 0xc5, 0x12, 0xea, 0xab, 0x62, 0x64, 0x21,
          0xf1, 0xec, 0x8c, 0x92, 0x01, 0x3f, 0xc5, 0xf8, 0x2a, 0xe9, 0x8e,
          0xe5, 0x33, 0xeb, 0x46, 0x19, 0xb8, 0xde, 0xb4, 0xd0, 0x6c}},
        {"1.3.6.1.4.1.14370.1.6", ""},
    },
    // GeoTrust Primary Certification Authority - G2
    {
        {{0x5e, 0xdb, 0x7a, 0xc4, 0x3b, 0x82, 0xa0, 0x6a, 0x87, 0x61, 0xe8,
          0xd7, 0xbe, 0x49, 0x79, 0xeb, 0xf2, 0x61, 0x1f, 0x7d, 0xd7, 0x9b,
          0xf9, 0x1c, 0x1c, 0x6b, 0x56, 0x6a, 0x21, 0x9e, 0xd7, 0x66}},
        {"1.3.6.1.4.1.14370.1.6", ""},
    },
    // GeoTrust Primary Certification Authority - G3
    {
        {{0xb4, 0x78, 0xb8, 0x12, 0x25, 0x0d, 0xf8, 0x78, 0x63, 0x5c, 0x2a,
          0xa7, 0xec, 0x7d, 0x15, 0x5e, 0xaa, 0x62, 0x5e, 0xe8, 0x29, 0x16,
          0xe2, 0xcd, 0x29, 0x43, 0x61, 0x88, 0x6c, 0xd1, 0xfb, 0xd4}},
        {"1.3.6.1.4.1.14370.1.6", ""},
    },
    // GlobalSign Root CA - R2
    // https://www.globalsign.com/
    {
        {{0xca, 0x42, 0xdd, 0x41, 0x74, 0x5f, 0xd0, 0xb8, 0x1e, 0xb9, 0x02,
          0x36, 0x2c, 0xf9, 0xd8, 0xbf, 0x71, 0x9d, 0xa1, 0xbd, 0x1b, 0x1e,
          0xfc, 0x94, 0x6f, 0x5b, 0x4c, 0x99, 0xf4, 0x2c, 0x1b, 0x9e}},
        {"1.3.6.1.4.1.4146.1.1", ""},
    },
    // GlobalSign Root CA
    {
        {{0xeb, 0xd4, 0x10, 0x40, 0xe4, 0xbb, 0x3e, 0xc7, 0x42, 0xc9, 0xe3,
          0x81, 0xd3, 0x1e, 0xf2, 0xa4, 0x1a, 0x48, 0xb6, 0x68, 0x5c, 0x96,
          0xe7, 0xce, 0xf3, 0xc1, 0xdf, 0x6c, 0xd4, 0x33, 0x1c, 0x99}},
        {"1.3.6.1.4.1.4146.1.1", ""},
    },
    // GlobalSign Root CA - R3
    // https://2029.globalsign.com/
    {
        {{0xcb, 0xb5, 0x22, 0xd7, 0xb7, 0xf1, 0x27, 0xad, 0x6a, 0x01, 0x13,
          0x86, 0x5b, 0xdf, 0x1c, 0xd4, 0x10, 0x2e, 0x7d, 0x07, 0x59, 0xaf,
          0x63, 0x5a, 0x7c, 0xf4, 0x72, 0x0d, 0xc9, 0x63, 0xc5, 0x3b}},
        {"1.3.6.1.4.1.4146.1.1", ""},
    },
    // GlobalSign ECC Root CA - R4
    // https://2038r4.globalsign.com
    {
        {{0xbe, 0xc9, 0x49, 0x11, 0xc2, 0x95, 0x56, 0x76, 0xdb, 0x6c, 0x0a,
          0x55, 0x09, 0x86, 0xd7, 0x6e, 0x3b, 0xa0, 0x05, 0x66, 0x7c, 0x44,
          0x2c, 0x97, 0x62, 0xb4, 0xfb, 0xb7, 0x73, 0xde, 0x22, 0x8c}},
        {"1.3.6.1.4.1.4146.1.1", ""},
    },
    // GlobalSign ECC Root CA - R5
    // https://2038r5.globalsign.com/
    {
        {{0x17, 0x9f, 0xbc, 0x14, 0x8a, 0x3d, 0xd0, 0x0f, 0xd2, 0x4e, 0xa1,
          0x34, 0x58, 0xcc, 0x43, 0xbf, 0xa7, 0xf5, 0x9c, 0x81, 0x82, 0xd7,
          0x83, 0xa5, 0x13, 0xf6, 0xeb, 0xec, 0x10, 0x0c, 0x89, 0x24}},
        {"1.3.6.1.4.1.4146.1.1", ""},
    },
    // Go Daddy Class 2 Certification Authority
    // https://www.godaddy.com/
    {
        {{0xc3, 0x84, 0x6b, 0xf2, 0x4b, 0x9e, 0x93, 0xca, 0x64, 0x27, 0x4c,
          0x0e, 0xc6, 0x7c, 0x1e, 0xcc, 0x5e, 0x02, 0x4f, 0xfc, 0xac, 0xd2,
          0xd7, 0x40, 0x19, 0x35, 0x0e, 0x81, 0xfe, 0x54, 0x6a, 0xe4}},
        {"2.16.840.1.114413.1.7.23.3", ""},
    },
    // Go Daddy Root Certificate Authority - G2
    // https://valid.gdig2.catest.godaddy.com/
    {
        {{0x45, 0x14, 0x0b, 0x32, 0x47, 0xeb, 0x9c, 0xc8, 0xc5, 0xb4, 0xf0,
          0xd7, 0xb5, 0x30, 0x91, 0xf7, 0x32, 0x92, 0x08, 0x9e, 0x6e, 0x5a,
          0x63, 0xe2, 0x74, 0x9d, 0xd3, 0xac, 0xa9, 0x19, 0x8e, 0xda}},
        {"2.16.840.1.114413.1.7.23.3", ""},
    },
    // GTE CyberTrust Global Root
    // https://www.cybertrust.ne.jp/
    {
        {{0xa5, 0x31, 0x25, 0x18, 0x8d, 0x21, 0x10, 0xaa, 0x96, 0x4b, 0x02,
          0xc7, 0xb7, 0xc6, 0xda, 0x32, 0x03, 0x17, 0x08, 0x94, 0xe5, 0xfb,
          0x71, 0xff, 0xfb, 0x66, 0x67, 0xd5, 0xe6, 0x81, 0x0a, 0x36}},
        {"1.3.6.1.4.1.6334.1.100.1", ""},
    },
    // Izenpe.com - SHA256 root
    // The first OID is for businesses and the second for government entities.
    // These are the test sites, respectively:
    // https://servicios.izenpe.com
    // https://servicios1.izenpe.com
    {
        {{0x25, 0x30, 0xcc, 0x8e, 0x98, 0x32, 0x15, 0x02, 0xba, 0xd9, 0x6f,
          0x9b, 0x1f, 0xba, 0x1b, 0x09, 0x9e, 0x2d, 0x29, 0x9e, 0x0f, 0x45,
          0x48, 0xbb, 0x91, 0x4f, 0x36, 0x3b, 0xc0, 0xd4, 0x53, 0x1f}},
        {"1.3.6.1.4.1.14777.6.1.1", "1.3.6.1.4.1.14777.6.1.2"},
    },
    // Izenpe.com - SHA1 root
    // Windows XP finds this, SHA1, root instead. The policy OIDs are the same
    // as for the SHA256 root, above.
    {
        {{0x23, 0x80, 0x42, 0x03, 0xca, 0x45, 0xd8, 0xcd, 0xe7, 0x16, 0xb8,
          0xc1, 0x3b, 0xf3, 0xb4, 0x48, 0x45, 0x7f, 0xa0, 0x6c, 0xc1, 0x02,
          0x50, 0x99, 0x7f, 0xa0, 0x14, 0x58, 0x31, 0x7c, 0x41, 0xe5}},
        {"1.3.6.1.4.1.14777.6.1.1", "1.3.6.1.4.1.14777.6.1.2"},
    },
    // LuxTrust Global Root 2
    // https://ltsslca5.trustme.lu/
    {
        {{0x54, 0x45, 0x5f, 0x71, 0x29, 0xc2, 0x0b, 0x14, 0x47, 0xc4, 0x18,
          0xf9, 0x97, 0x16, 0x8f, 0x24, 0xc5, 0x8f, 0xc5, 0x02, 0x3b, 0xf5,
          0xda, 0x5b, 0xe2, 0xeb, 0x6e, 0x1d, 0xd8, 0x90, 0x2e, 0xd5}},
        {"1.3.171.1.1.10.5.2", ""},
    },
    // Network Solutions Certificate Authority
    // https://www.networksolutions.com/website-packages/index.jsp
    {
        {{0x15, 0xf0, 0xba, 0x00, 0xa3, 0xac, 0x7a, 0xf3, 0xac, 0x88, 0x4c,
          0x07, 0x2b, 0x10, 0x11, 0xa0, 0x77, 0xbd, 0x77, 0xc0, 0x97, 0xf4,
          0x01, 0x64, 0xb2, 0xf8, 0x59, 0x8a, 0xbd, 0x83, 0x86, 0x0c}},
        {"1.3.6.1.4.1.782.1.2.1.8.1", ""},
    },
    // Network Solutions Certificate Authority (reissued certificate with
    // NotBefore of Jan  1 00:00:00 2011 GMT).
    // https://www.networksolutions.com/website-packages/index.jsp
    {
        {{0x00, 0x16, 0x86, 0xcd, 0x18, 0x1f, 0x83, 0xa1, 0xb1, 0x21, 0x7d,
          0x30, 0x5b, 0x36, 0x5c, 0x41, 0xe3, 0x47, 0x0a, 0x78, 0xa1, 0xd3,
          0x7b, 0x13, 0x4a, 0x98, 0xcd, 0x54, 0x7b, 0x92, 0xda, 0xb3}},
        {"1.3.6.1.4.1.782.1.2.1.8.1", ""},
    },
    // OISTE WISeKey Global Root GB CA
    // https://goodevssl.wisekey.com
    {
        {{0x6b, 0x9c, 0x08, 0xe8, 0x6e, 0xb0, 0xf7, 0x67, 0xcf, 0xad, 0x65,
          0xcd, 0x98, 0xb6, 0x21, 0x49, 0xe5, 0x49, 0x4a, 0x67, 0xf5, 0x84,
          0x5e, 0x7b, 0xd1, 0xed, 0x01, 0x9f, 0x27, 0xb8, 0x6b, 0xd6}},
        {"2.16.756.5.14.7.4.8", ""},
    },
    // QuoVadis Root CA 2
    // https://www.quovadis.bm/
    {
        {{0x85, 0xa0, 0xdd, 0x7d, 0xd7, 0x20, 0xad, 0xb7, 0xff, 0x05, 0xf8,
          0x3d, 0x54, 0x2b, 0x20, 0x9d, 0xc7, 0xff, 0x45, 0x28, 0xf7, 0xd6,
          0x77, 0xb1, 0x83, 0x89, 0xfe, 0xa5, 0xe5, 0xc4, 0x9e, 0x86}},
        {"1.3.6.1.4.1.8024.0.2.100.1.2", ""},
    },
    // QuoVadis Root CA 2 G3
    // https://evsslicag3-v.quovadisglobal.com/
    {
        {{0x8f, 0xe4, 0xfb, 0x0a, 0xf9, 0x3a, 0x4d, 0x0d, 0x67, 0xdb, 0x0b,
          0xeb, 0xb2, 0x3e, 0x37, 0xc7, 0x1b, 0xf3, 0x25, 0xdc, 0xbc, 0xdd,
          0x24, 0x0e, 0xa0, 0x4d, 0xaf, 0x58, 0xb4, 0x7e, 0x18, 0x40}},
        {"1.3.6.1.4.1.8024.0.2.100.1.2", ""},
    },
    // SecureTrust CA, SecureTrust Corporation
    // https://www.securetrust.com
    // https://www.trustwave.com/
    {
        {{0xf1, 0xc1, 0xb5, 0x0a, 0xe5, 0xa2, 0x0d, 0xd8, 0x03, 0x0e, 0xc9,
          0xf6, 0xbc, 0x24, 0x82, 0x3d, 0xd3, 0x67, 0xb5, 0x25, 0x57, 0x59,
          0xb4, 0xe7, 0x1b, 0x61, 0xfc, 0xe9, 0xf7, 0x37, 0x5d, 0x73}},
        {"2.16.840.1.114404.1.1.2.4.1", ""},
    },
    // Secure Global CA, SecureTrust Corporation
    {
        {{0x42, 0x00, 0xf5, 0x04, 0x3a, 0xc8, 0x59, 0x0e, 0xbb, 0x52, 0x7d,
          0x20, 0x9e, 0xd1, 0x50, 0x30, 0x29, 0xfb, 0xcb, 0xd4, 0x1c, 0xa1,
          0xb5, 0x06, 0xec, 0x27, 0xf1, 0x5a, 0xde, 0x7d, 0xac, 0x69}},
        {"2.16.840.1.114404.1.1.2.4.1", ""},
    },
    // Security Communication RootCA1
    // https://www.secomtrust.net/contact/form.html
    {
        {{0xe7, 0x5e, 0x72, 0xed, 0x9f, 0x56, 0x0e, 0xec, 0x6e, 0xb4, 0x80,
          0x00, 0x73, 0xa4, 0x3f, 0xc3, 0xad, 0x19, 0x19, 0x5a, 0x39, 0x22,
          0x82, 0x01, 0x78, 0x95, 0x97, 0x4a, 0x99, 0x02, 0x6b, 0x6c}},
        {"1.2.392.200091.100.721.1", ""},
    },
    // Security Communication EV RootCA1
    // https://www.secomtrust.net/contact/form.html
    {
        {{0xa2, 0x2d, 0xba, 0x68, 0x1e, 0x97, 0x37, 0x6e, 0x2d, 0x39, 0x7d,
          0x72, 0x8a, 0xae, 0x3a, 0x9b, 0x62, 0x96, 0xb9, 0xfd, 0xba, 0x60,
          0xbc, 0x2e, 0x11, 0xf6, 0x47, 0xf2, 0xc6, 0x75, 0xfb, 0x37}},
        {"1.2.392.200091.100.721.1", ""},
    },
    // Security Communication EV RootCA2
    // https://www.secomtrust.net/contact/form.html
    {
        {{0x51, 0x3b, 0x2c, 0xec, 0xb8, 0x10, 0xd4, 0xcd, 0xe5, 0xdd, 0x85,
          0x39, 0x1a, 0xdf, 0xc6, 0xc2, 0xdd, 0x60, 0xd8, 0x7b, 0xb7, 0x36,
          0xd2, 0xb5, 0x21, 0x48, 0x4a, 0xa4, 0x7a, 0x0e, 0xbe, 0xf6}},
        {"1.2.392.200091.100.721.1", ""},
    },
    // Staat der Nederlanden EV Root CA
    // https://pkioevssl-v.quovadisglobal.com/
    {
        {{0x4d, 0x24, 0x91, 0x41, 0x4c, 0xfe, 0x95, 0x67, 0x46, 0xec, 0x4c,
          0xef, 0xa6, 0xcf, 0x6f, 0x72, 0xe2, 0x8a, 0x13, 0x29, 0x43, 0x2f,
          0x9d, 0x8a, 0x90, 0x7a, 0xc4, 0xcb, 0x5d, 0xad, 0xc1, 0x5a}},
        {"2.16.528.1.1003.1.2.7", ""},
    },
    // StartCom Certification Authority
    // https://www.startssl.com/
    {
        {{0xc7, 0x66, 0xa9, 0xbe, 0xf2, 0xd4, 0x07, 0x1c, 0x86, 0x3a, 0x31,
          0xaa, 0x49, 0x20, 0xe8, 0x13, 0xb2, 0xd1, 0x98, 0x60, 0x8c, 0xb7,
          0xb7, 0xcf, 0xe2, 0x11, 0x43, 0xb8, 0x36, 0xdf, 0x09, 0xea}},
        {"1.3.6.1.4.1.23223.1.1.1", ""},
    },
    // Starfield Class 2 Certification Authority
    // https://www.starfieldtech.com/
    {
        {{0x14, 0x65, 0xfa, 0x20, 0x53, 0x97, 0xb8, 0x76, 0xfa, 0xa6, 0xf0,
          0xa9, 0x95, 0x8e, 0x55, 0x90, 0xe4, 0x0f, 0xcc, 0x7f, 0xaa, 0x4f,
          0xb7, 0xc2, 0xc8, 0x67, 0x75, 0x21, 0xfb, 0x5f, 0xb6, 0x58}},
        {"2.16.840.1.114414.1.7.23.3", ""},
    },
    // Starfield Root Certificate Authority - G2
    // https://valid.sfig2.catest.starfieldtech.com/
    {
        {{0x2c, 0xe1, 0xcb, 0x0b, 0xf9, 0xd2, 0xf9, 0xe1, 0x02, 0x99, 0x3f,
          0xbe, 0x21, 0x51, 0x52, 0xc3, 0xb2, 0xdd, 0x0c, 0xab, 0xde, 0x1c,
          0x68, 0xe5, 0x31, 0x9b, 0x83, 0x91, 0x54, 0xdb, 0xb7, 0xf5}},
        {"2.16.840.1.114414.1.7.23.3", ""},
    },
    // Starfield Services Root Certificate Authority - G2
    // https://valid.sfsg2.catest.starfieldtech.com/
    {
        {{0x56, 0x8d, 0x69, 0x05, 0xa2, 0xc8, 0x87, 0x08, 0xa4, 0xb3, 0x02,
          0x51, 0x90, 0xed, 0xcf, 0xed, 0xb1, 0x97, 0x4a, 0x60, 0x6a, 0x13,
          0xc6, 0xe5, 0x29, 0x0f, 0xcb, 0x2a, 0xe6, 0x3e, 0xda, 0xb5}},
        {"2.16.840.1.114414.1.7.24.3", "2.23.140.1.1"},
    },
    // SwissSign Gold CA - G2
    // https://testevg2.swisssign.net/
    {
        {{0x62, 0xdd, 0x0b, 0xe9, 0xb9, 0xf5, 0x0a, 0x16, 0x3e, 0xa0, 0xf8,
          0xe7, 0x5c, 0x05, 0x3b, 0x1e, 0xca, 0x57, 0xea, 0x55, 0xc8, 0x68,
          0x8f, 0x64, 0x7c, 0x68, 0x81, 0xf2, 0xc8, 0x35, 0x7b, 0x95}},
        {"2.16.756.1.89.1.2.1.1", ""},
    },
    // Swisscom Root EV CA 2
    // https://test-quarz-ev-ca-2.pre.swissdigicert.ch
    {
        {{0xd9, 0x5f, 0xea, 0x3c, 0xa4, 0xee, 0xdc, 0xe7, 0x4c, 0xd7, 0x6e,
          0x75, 0xfc, 0x6d, 0x1f, 0xf6, 0x2c, 0x44, 0x1f, 0x0f, 0xa8, 0xbc,
          0x77, 0xf0, 0x34, 0xb1, 0x9e, 0x5d, 0xb2, 0x58, 0x01, 0x5d}},
        {"2.16.756.1.83.21.0", ""},
    },
    // Thawte Premium Server CA
    // https://www.thawte.com/
    {
        {{0xab, 0x70, 0x36, 0x36, 0x5c, 0x71, 0x54, 0xaa, 0x29, 0xc2, 0xc2,
          0x9f, 0x5d, 0x41, 0x91, 0x16, 0x3b, 0x16, 0x2a, 0x22, 0x25, 0x01,
          0x13, 0x57, 0xd5, 0x6d, 0x07, 0xff, 0xa7, 0xbc, 0x1f, 0x72}},
        {"2.16.840.1.113733.1.7.48.1", ""},
    },
    // thawte Primary Root CA
    // https://www.thawte.com/
    {
        {{0x8d, 0x72, 0x2f, 0x81, 0xa9, 0xc1, 0x13, 0xc0, 0x79, 0x1d, 0xf1,
          0x36, 0xa2, 0x96, 0x6d, 0xb2, 0x6c, 0x95, 0x0a, 0x97, 0x1d, 0xb4,
          0x6b, 0x41, 0x99, 0xf4, 0xea, 0x54, 0xb7, 0x8b, 0xfb, 0x9f}},
        {"2.16.840.1.113733.1.7.48.1", ""},
    },
    // thawte Primary Root CA - G2
    {
        {{0xa4, 0x31, 0x0d, 0x50, 0xaf, 0x18, 0xa6, 0x44, 0x71, 0x90, 0x37,
          0x2a, 0x86, 0xaf, 0xaf, 0x8b, 0x95, 0x1f, 0xfb, 0x43, 0x1d, 0x83,
          0x7f, 0x1e, 0x56, 0x88, 0xb4, 0x59, 0x71, 0xed, 0x15, 0x57}},
        {"2.16.840.1.113733.1.7.48.1", ""},
    },
    // thawte Primary Root CA - G3
    {
        {{0x4b, 0x03, 0xf4, 0x58, 0x07, 0xad, 0x70, 0xf2, 0x1b, 0xfc, 0x2c,
          0xae, 0x71, 0xc9, 0xfd, 0xe4, 0x60, 0x4c, 0x06, 0x4c, 0xf5, 0xff,
          0xb6, 0x86, 0xba, 0xe5, 0xdb, 0xaa, 0xd7, 0xfd, 0xd3, 0x4c}},
        {"2.16.840.1.113733.1.7.48.1", ""},
    },
    // TWCA Global Root CA
    // https://evssldemo3.twca.com.tw/index.html
    {
        {{0x59, 0x76, 0x90, 0x07, 0xf7, 0x68, 0x5d, 0x0f, 0xcd, 0x50, 0x87,
          0x2f, 0x9f, 0x95, 0xd5, 0x75, 0x5a, 0x5b, 0x2b, 0x45, 0x7d, 0x81,
          0xf3, 0x69, 0x2b, 0x61, 0x0a, 0x98, 0x67, 0x2f, 0x0e, 0x1b}},
        {"1.3.6.1.4.1.40869.1.1.22.3", ""},
    },
    // TWCA Root Certification Authority
    // https://evssldemo.twca.com.tw/index.html
    {
        {{0xbf, 0xd8, 0x8f, 0xe1, 0x10, 0x1c, 0x41, 0xae, 0x3e, 0x80, 0x1b,
          0xf8, 0xbe, 0x56, 0x35, 0x0e, 0xe9, 0xba, 0xd1, 0xa6, 0xb9, 0xbd,
          0x51, 0x5e, 0xdc, 0x5c, 0x6d, 0x5b, 0x87, 0x11, 0xac, 0x44}},
        {"1.3.6.1.4.1.40869.1.1.22.3", ""},
    },
    // T-TeleSec GlobalRoot Class 3
    // http://www.telesec.de/ / https://root-class3.test.telesec.de/
    {
        {{0xfd, 0x73, 0xda, 0xd3, 0x1c, 0x64, 0x4f, 0xf1, 0xb4, 0x3b, 0xef,
          0x0c, 0xcd, 0xda, 0x96, 0x71, 0x0b, 0x9c, 0xd9, 0x87, 0x5e, 0xca,
          0x7e, 0x31, 0x70, 0x7a, 0xf3, 0xe9, 0x6d, 0x52, 0x2b, 0xbd}},
        {"1.3.6.1.4.1.7879.13.24.1", ""},
    },
    // USERTrust ECC Certification Authority
    // https://usertrustecccertificationauthority-ev.comodoca.com/
    {
        {{0x4f, 0xf4, 0x60, 0xd5, 0x4b, 0x9c, 0x86, 0xda, 0xbf, 0xbc, 0xfc,
          0x57, 0x12, 0xe0, 0x40, 0x0d, 0x2b, 0xed, 0x3f, 0xbc, 0x4d, 0x4f,
          0xbd, 0xaa, 0x86, 0xe0, 0x6a, 0xdc, 0xd2, 0xa9, 0xad, 0x7a}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // USERTrust RSA Certification Authority
    // https://usertrustrsacertificationauthority-ev.comodoca.com/
    {
        {{0xe7, 0x93, 0xc9, 0xb0, 0x2f, 0xd8, 0xaa, 0x13, 0xe2, 0x1c, 0x31,
          0x22, 0x8a, 0xcc, 0xb0, 0x81, 0x19, 0x64, 0x3b, 0x74, 0x9c, 0x89,
          0x89, 0x64, 0xb1, 0x74, 0x6d, 0x46, 0xc3, 0xd4, 0xcb, 0xd2}},
        {"1.3.6.1.4.1.6449.1.2.1.5.1", ""},
    },
    // UTN-USERFirst-Hardware
    {
        {{0x6e, 0xa5, 0x47, 0x41, 0xd0, 0x04, 0x66, 0x7e, 0xed, 0x1b, 0x48,
          0x16, 0x63, 0x4a, 0xa3, 0xa7, 0x9e, 0x6e, 0x4b, 0x96, 0x95, 0x0f,
          0x82, 0x79, 0xda, 0xfc, 0x8d, 0x9b, 0xd8, 0x81, 0x21, 0x37}},
        {
            "1.3.6.1.4.1.6449.1.2.1.5.1",
            // This is the Network Solutions EV OID. However, this root
            // cross-certifies NetSol and so we need it here too.
            "1.3.6.1.4.1.782.1.2.1.8.1",
        },
    },
    // ValiCert Class 2 Policy Validation Authority
    {
        {{0x58, 0xd0, 0x17, 0x27, 0x9c, 0xd4, 0xdc, 0x63, 0xab, 0xdd, 0xb1,
          0x96, 0xa6, 0xc9, 0x90, 0x6c, 0x30, 0xc4, 0xe0, 0x87, 0x83, 0xea,
          0xe8, 0xc1, 0x60, 0x99, 0x54, 0xd6, 0x93, 0x55, 0x59, 0x6b}},
        {"2.16.840.1.114413.1.7.23.3", "2.16.840.1.114414.1.7.23.3"},
    },
    // VeriSign Class 3 Public Primary Certification Authority
    // https://www.verisign.com/
    {
        {{0xe7, 0x68, 0x56, 0x34, 0xef, 0xac, 0xf6, 0x9a, 0xce, 0x93, 0x9a,
          0x6b, 0x25, 0x5b, 0x7b, 0x4f, 0xab, 0xef, 0x42, 0x93, 0x5b, 0x50,
          0xa2, 0x65, 0xac, 0xb5, 0xcb, 0x60, 0x27, 0xe4, 0x4e, 0x70}},
        {"2.16.840.1.113733.1.7.23.6", ""},
    },
    // VeriSign Class 3 Public Primary Certification Authority - G4
    {
        {{0x69, 0xdd, 0xd7, 0xea, 0x90, 0xbb, 0x57, 0xc9, 0x3e, 0x13, 0x5d,
          0xc8, 0x5e, 0xa6, 0xfc, 0xd5, 0x48, 0x0b, 0x60, 0x32, 0x39, 0xbd,
          0xc4, 0x54, 0xfc, 0x75, 0x8b, 0x2a, 0x26, 0xcf, 0x7f, 0x79}},
        {"2.16.840.1.113733.1.7.23.6", ""},
    },
    // VeriSign Class 3 Public Primary Certification Authority - G5
    // https://www.verisign.com/
    {
        {{0x9a, 0xcf, 0xab, 0x7e, 0x43, 0xc8, 0xd8, 0x80, 0xd0, 0x6b, 0x26,
          0x2a, 0x94, 0xde, 0xee, 0xe4, 0xb4, 0x65, 0x99, 0x89, 0xc3, 0xd0,
          0xca, 0xf1, 0x9b, 0xaf, 0x64, 0x05, 0xe4, 0x1a, 0xb7, 0xdf}},
        {"2.16.840.1.113733.1.7.23.6", ""},
    },
    // VeriSign Universal Root Certification Authority
    {
        {{0x23, 0x99, 0x56, 0x11, 0x27, 0xa5, 0x71, 0x25, 0xde, 0x8c, 0xef,
          0xea, 0x61, 0x0d, 0xdf, 0x2f, 0xa0, 0x78, 0xb5, 0xc8, 0x06, 0x7f,
          0x4e, 0x82, 0x82, 0x90, 0xbf, 0xb8, 0x60, 0xe8, 0x4b, 0x3c}},
        {"2.16.840.1.113733.1.7.23.6", ""},
    },
    // Wells Fargo WellsSecure Public Root Certificate Authority
    // https://nerys.wellsfargo.com/test.html
    {
        {{0xa7, 0x12, 0x72, 0xae, 0xaa, 0xa3, 0xcf, 0xe8, 0x72, 0x7f, 0x7f,
          0xb3, 0x9f, 0x0f, 0xb3, 0xd1, 0xe5, 0x42, 0x6e, 0x90, 0x60, 0xb0,
          0x6e, 0xe6, 0xf1, 0x3e, 0x9a, 0x3c, 0x58, 0x33, 0xcd, 0x43}},
        {"2.16.840.1.114171.500.9", ""},
    },
    // XRamp Global Certification Authority
    {
        {{0xce, 0xcd, 0xdc, 0x90, 0x50, 0x99, 0xd8, 0xda, 0xdf, 0xc5, 0xb1,
          0xd2, 0x09, 0xb7, 0x37, 0xcb, 0xe2, 0xc1, 0x8c, 0xfb, 0x2c, 0x10,
          0xc0, 0xff, 0x0b, 0xcf, 0x0d, 0x32, 0x86, 0xfc, 0x1a, 0xa2}},
        {"2.16.840.1.114404.1.1.2.4.1", ""},
    }};

#endif  // defined(PLATFORM_USES_CHROMIUM_EV_METADATA)

static base::LazyInstance<EVRootCAMetadata>::Leaky
    g_ev_root_ca_metadata = LAZY_INSTANCE_INITIALIZER;

// static
EVRootCAMetadata* EVRootCAMetadata::GetInstance() {
  return g_ev_root_ca_metadata.Pointer();
}

#if defined(USE_NSS_CERTS)

namespace {
// Converts a DER-encoded OID (without leading tag and length) to a SECOidTag.
//
// Returns true if it was able to find an *existing* SECOidTag (it will not
// register one if missing).
//
// Since all the EV OIDs are registered during EVRootCAMetadata's constructor,
// doing a lookup only needs to consider existing OID tags.
bool ConvertBytesToSecOidTag(const der::Input& oid, SECOidTag* out) {
  SECItem item;
  item.data = const_cast<uint8_t*>(oid.UnsafeData());
  item.len = oid.Length();
  *out = SECOID_FindOIDTag(&item);
  return *out != SEC_OID_UNKNOWN;
}

}  // namespace

bool EVRootCAMetadata::IsEVPolicyOID(PolicyOID policy_oid) const {
  return policy_oids_.find(policy_oid) != policy_oids_.end();
}

bool EVRootCAMetadata::IsEVPolicyOIDGivenBytes(
    const der::Input& policy_oid) const {
  SECOidTag oid_tag;
  return ConvertBytesToSecOidTag(policy_oid, &oid_tag) &&
         IsEVPolicyOID(oid_tag);
}

bool EVRootCAMetadata::HasEVPolicyOID(const SHA256HashValue& fingerprint,
                                      PolicyOID policy_oid) const {
  PolicyOIDMap::const_iterator iter = ev_policy_.find(fingerprint);
  if (iter == ev_policy_.end())
    return false;
  for (std::vector<PolicyOID>::const_iterator
       j = iter->second.begin(); j != iter->second.end(); ++j) {
    if (*j == policy_oid)
      return true;
  }
  return false;
}

bool EVRootCAMetadata::HasEVPolicyOIDGivenBytes(
    const SHA256HashValue& fingerprint,
    const der::Input& policy_oid) const {
  SECOidTag oid_tag;
  return ConvertBytesToSecOidTag(policy_oid, &oid_tag) &&
         HasEVPolicyOID(fingerprint, oid_tag);
}

// static
bool EVRootCAMetadata::IsCaBrowserForumEvOid(PolicyOID policy_oid) {
  // OID: 2.23.140.1.1
  const uint8_t kCabEvOid[] = {0x67, 0x81, 0x0c, 0x01, 0x01};
  SECItem item;
  item.data = const_cast<uint8_t*>(&kCabEvOid[0]);
  item.len = sizeof(kCabEvOid);
  return policy_oid == SECOID_FindOIDTag(&item);
}

bool EVRootCAMetadata::AddEVCA(const SHA256HashValue& fingerprint,
                               const char* policy) {
  if (ev_policy_.find(fingerprint) != ev_policy_.end())
    return false;

  PolicyOID oid;
  if (!RegisterOID(policy, &oid))
    return false;

  ev_policy_[fingerprint].push_back(oid);
  policy_oids_.insert(oid);

  return true;
}

bool EVRootCAMetadata::RemoveEVCA(const SHA256HashValue& fingerprint) {
  PolicyOIDMap::iterator it = ev_policy_.find(fingerprint);
  if (it == ev_policy_.end())
    return false;
  PolicyOID oid = it->second[0];
  ev_policy_.erase(it);
  policy_oids_.erase(oid);
  return true;
}

// static
bool EVRootCAMetadata::RegisterOID(const char* policy,
                                   PolicyOID* out) {
  PRUint8 buf[64];
  SECItem oid_item;
  oid_item.data = buf;
  oid_item.len = sizeof(buf);
  SECStatus status = SEC_StringToOID(NULL, &oid_item, policy, 0);
  if (status != SECSuccess)
    return false;

  // Register the OID.
  SECOidData od;
  od.oid.len = oid_item.len;
  od.oid.data = oid_item.data;
  od.offset = SEC_OID_UNKNOWN;
  od.desc = policy;
  od.mechanism = CKM_INVALID_MECHANISM;
  od.supportedExtension = INVALID_CERT_EXTENSION;
  *out = SECOID_AddEntry(&od);
  return *out != SEC_OID_UNKNOWN;
}

#elif defined(OS_WIN)

namespace {

bool ConvertBytesToDottedString(const der::Input& policy_oid,
                                std::string* dotted) {
  ASN1_OBJECT obj;
  memset(&obj, 0, sizeof(obj));
  obj.data = policy_oid.UnsafeData();
  obj.length = policy_oid.Length();

  // Determine the length of the dotted string.
  int len = OBJ_obj2txt(nullptr, 0, &obj, 1 /* dont_search_names */);
  if (len == -1)
    return false;

  // Write the dotted string into |*dotted|.
  dotted->resize(len + 1);
  return len == OBJ_obj2txt(&(*dotted)[0], static_cast<int>(dotted->size()),
                            &obj, 1 /* dont_search_names */);
}

}  // namespace

bool EVRootCAMetadata::IsEVPolicyOID(PolicyOID policy_oid) const {
  for (size_t i = 0; i < arraysize(ev_root_ca_metadata); i++) {
    for (size_t j = 0; j < arraysize(ev_root_ca_metadata[i].policy_oids); j++) {
      if (ev_root_ca_metadata[i].policy_oids[j][0] == '\0')
        break;
      if (strcmp(policy_oid, ev_root_ca_metadata[i].policy_oids[j]) == 0)
        return true;
    }
  }

  for (ExtraEVCAMap::const_iterator i = extra_cas_.begin();
       i != extra_cas_.end(); i++) {
    if (i->second == policy_oid)
      return true;
  }

  return false;
}

bool EVRootCAMetadata::IsEVPolicyOIDGivenBytes(
    const der::Input& policy_oid) const {
  std::string dotted;
  return ConvertBytesToDottedString(policy_oid, &dotted) &&
         IsEVPolicyOID(dotted.c_str());
}

bool EVRootCAMetadata::HasEVPolicyOID(const SHA256HashValue& fingerprint,
                                      PolicyOID policy_oid) const {
  for (size_t i = 0; i < arraysize(ev_root_ca_metadata); i++) {
    if (fingerprint != ev_root_ca_metadata[i].fingerprint)
      continue;
    for (size_t j = 0; j < arraysize(ev_root_ca_metadata[i].policy_oids); j++) {
      if (ev_root_ca_metadata[i].policy_oids[j][0] == '\0')
        break;
      if (strcmp(policy_oid, ev_root_ca_metadata[i].policy_oids[j]) == 0)
        return true;
    }
    return false;
  }

  ExtraEVCAMap::const_iterator it = extra_cas_.find(fingerprint);
  return it != extra_cas_.end() && it->second == policy_oid;
}

bool EVRootCAMetadata::HasEVPolicyOIDGivenBytes(
    const SHA256HashValue& fingerprint,
    const der::Input& policy_oid) const {
  std::string dotted;
  return ConvertBytesToDottedString(policy_oid, &dotted) &&
         HasEVPolicyOID(fingerprint, dotted.c_str());
}

// static
bool EVRootCAMetadata::IsCaBrowserForumEvOid(PolicyOID policy_oid) {
  return strcmp(policy_oid, "2.23.140.1.1") == 0;
}

bool EVRootCAMetadata::AddEVCA(const SHA256HashValue& fingerprint,
                               const char* policy) {
  for (size_t i = 0; i < arraysize(ev_root_ca_metadata); i++) {
    if (fingerprint == ev_root_ca_metadata[i].fingerprint)
      return false;
  }

  if (extra_cas_.find(fingerprint) != extra_cas_.end())
    return false;

  extra_cas_[fingerprint] = policy;
  return true;
}

bool EVRootCAMetadata::RemoveEVCA(const SHA256HashValue& fingerprint) {
  ExtraEVCAMap::iterator it = extra_cas_.find(fingerprint);
  if (it == extra_cas_.end())
    return false;
  extra_cas_.erase(it);
  return true;
}

#elif defined(PLATFORM_USES_CHROMIUM_EV_METADATA)

namespace {

std::string OIDStringToDER(const char* policy) {
  bssl::UniquePtr<ASN1_OBJECT> obj(
      OBJ_txt2obj(policy, 1 /* dont_search_names */));
  if (!obj)
    return std::string();

  return std::string(reinterpret_cast<const char*>(obj->data), obj->length);
}

}  // namespace

bool EVRootCAMetadata::IsEVPolicyOID(PolicyOID policy_oid) const {
  return policy_oids_.find(policy_oid.AsString()) != policy_oids_.end();
}

bool EVRootCAMetadata::IsEVPolicyOIDGivenBytes(
    const der::Input& policy_oid) const {
  // This implementation uses DER bytes already, so the two functions are the
  // same.
  return IsEVPolicyOID(policy_oid);
}

bool EVRootCAMetadata::HasEVPolicyOID(const SHA256HashValue& fingerprint,
                                      PolicyOID policy_oid) const {
  PolicyOIDMap::const_iterator iter = ev_policy_.find(fingerprint);
  if (iter == ev_policy_.end())
    return false;
  for (const std::string& ev_oid : iter->second) {
    if (der::Input(&ev_oid) == policy_oid)
      return true;
  }
  return false;
}

bool EVRootCAMetadata::HasEVPolicyOIDGivenBytes(
    const SHA256HashValue& fingerprint,
    const der::Input& policy_oid) const {
  // The implementation uses DER bytes already, so the two functions are the
  // same.
  return HasEVPolicyOID(fingerprint, policy_oid);
}

// static
bool EVRootCAMetadata::IsCaBrowserForumEvOid(PolicyOID policy_oid) {
  const uint8_t kCabEvOid[] = {0x67, 0x81, 0x0c, 0x01, 0x01};
  return der::Input(kCabEvOid) == policy_oid;
}

bool EVRootCAMetadata::AddEVCA(const SHA256HashValue& fingerprint,
                               const char* policy) {
  if (ev_policy_.find(fingerprint) != ev_policy_.end())
    return false;

  std::string der_policy = OIDStringToDER(policy);
  if (der_policy.empty())
    return false;

  ev_policy_[fingerprint].push_back(der_policy);
  policy_oids_.insert(der_policy);
  return true;
}

bool EVRootCAMetadata::RemoveEVCA(const SHA256HashValue& fingerprint) {
  PolicyOIDMap::iterator it = ev_policy_.find(fingerprint);
  if (it == ev_policy_.end())
    return false;
  std::string oid = it->second[0];
  ev_policy_.erase(it);
  policy_oids_.erase(oid);
  return true;
}

#else

// These are just stub functions for platforms where we don't use this EV
// metadata.
//

bool EVRootCAMetadata::IsEVPolicyOID(PolicyOID policy_oid) const {
  LOG(WARNING) << "Not implemented";
  return false;
}

bool EVRootCAMetadata::IsEVPolicyOIDGivenBytes(
    const der::Input& policy_oid) const {
  LOG(WARNING) << "Not implemented";
  return false;
}

bool EVRootCAMetadata::HasEVPolicyOID(const SHA256HashValue& fingerprint,
                                      PolicyOID policy_oid) const {
  LOG(WARNING) << "Not implemented";
  return false;
}

bool EVRootCAMetadata::HasEVPolicyOIDGivenBytes(
    const SHA256HashValue& fingerprint,
    const der::Input& policy_oid) const {
  LOG(WARNING) << "Not implemented";
  return false;
}

bool EVRootCAMetadata::AddEVCA(const SHA256HashValue& fingerprint,
                               const char* policy) {
  LOG(WARNING) << "Not implemented";
  return true;
}

bool EVRootCAMetadata::RemoveEVCA(const SHA256HashValue& fingerprint) {
  LOG(WARNING) << "Not implemented";
  return true;
}

#endif

EVRootCAMetadata::EVRootCAMetadata() {
  // Constructs the object from the raw metadata in ev_root_ca_metadata.
#if defined(USE_NSS_CERTS)
  crypto::EnsureNSSInit();

  for (size_t i = 0; i < arraysize(ev_root_ca_metadata); i++) {
    const EVMetadata& metadata = ev_root_ca_metadata[i];
    for (size_t j = 0; j < arraysize(metadata.policy_oids); j++) {
      if (metadata.policy_oids[j][0] == '\0')
        break;
      const char* policy_oid = metadata.policy_oids[j];

      PolicyOID policy;
      if (!RegisterOID(policy_oid, &policy)) {
        LOG(ERROR) << "Failed to register OID: " << policy_oid;
        continue;
      }

      ev_policy_[metadata.fingerprint].push_back(policy);
      policy_oids_.insert(policy);
    }
  }
#elif defined(PLATFORM_USES_CHROMIUM_EV_METADATA) && !defined(OS_WIN)
  for (size_t i = 0; i < arraysize(ev_root_ca_metadata); i++) {
    const EVMetadata& metadata = ev_root_ca_metadata[i];
    for (size_t j = 0; j < arraysize(metadata.policy_oids); j++) {
      if (metadata.policy_oids[j][0] == '\0')
        break;
      const char* policy_oid = metadata.policy_oids[j];

      PolicyOID policy;
      std::string policy_der = OIDStringToDER(policy_oid);
      if (policy_der.empty()) {
        LOG(ERROR) << "Failed to register OID: " << policy_oid;
        continue;
      }

      ev_policy_[metadata.fingerprint].push_back(policy_der);
      policy_oids_.insert(policy_der);
    }
  }
#endif
}

EVRootCAMetadata::~EVRootCAMetadata() = default;

}  // namespace net
