// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

GEN_INCLUDE(['../chromevox/testing/callback_helper.js']);

/**
 * Base class for browser tests for select-to-speak.
 * @constructor
 */
function SelectToSpeakE2ETest() {
  this.callbackHelper_ = new CallbackHelper(this);
}

SelectToSpeakE2ETest.prototype = {
  __proto__: testing.Test.prototype,

  /**
   * @override
   * No UI in the background context.
   */
  runAccessibilityChecks: false,

  /** @override */
  isAsync: true,

  /** @override */
  browsePreload: null,

  /** @override */
  testGenCppIncludes: function() {
    GEN_BLOCK(function() { /*!
#include "ash/accessibility/accessibility_delegate.h"
#include "ash/shell.h"
#include "base/bind.h"
#include "base/callback.h"
#include "chrome/browser/chromeos/accessibility/accessibility_manager.h"
#include "chrome/common/extensions/extension_constants.h"
#include "ui/keyboard/keyboard_util.h"
    */ });
  },

  /** @override */
  testGenPreamble: function() {
    GEN_BLOCK(function() { /*!
  //keyboard::SetRequestedKeyboardState(keyboard::KEYBOARD_STATE_ENABLED);
  //ash::Shell::Get()->CreateKeyboard();
  base::Closure load_cb =
      base::Bind(&chromeos::AccessibilityManager::SetSelectToSpeakEnabled,
          base::Unretained(chromeos::AccessibilityManager::Get()),
          true);
  chromeos::AccessibilityManager::Get()->SetSelectToSpeakEnabled(true);
  WaitForExtension(extension_misc::kSelectToSpeakExtensionId, load_cb);
    */ });
  },

  /**
   * Creates a callback that optionally calls {@code opt_callback} when
   * called.  If this method is called one or more times, then
   * {@code testDone()} will be called when all callbacks have been called.
   * @param {Function=} opt_callback Wrapped callback that will have its this
   *        reference bound to the test fixture.
   * @return {Function}
   */
  newCallback: function(opt_callback) {
    return this.callbackHelper_.wrap(opt_callback);
  },

  /**
   * Asserts that two strings are equal, collapsing repeated spaces and
   * removing leading / trailing whitespace.
   * @param {string} first The first string to compare.
   * @param {string} second The second string to compare.
   */
  assertEqualsCollapseWhitespace: function(first, second) {
    assertEquals(
        first.replace(/\s+/g, ' ').replace(/^\s/, '').replace(/\s$/, ''),
        second.replace(/\s+/g, ' ').replace(/^\s/, '').replace(/\s$/, ''));
  },

  /**
   * From chromevox_next_e2e_test_base.js
   * Gets the desktop from the automation API and Launches a new tab with
   * the given document, and runs |callback| when a load complete fires.
   * Arranges to call |testDone()| after |callback| returns.
   * NOTE: Callbacks created inside |callback| must be wrapped with
   * |this.newCallback| if passed to asynchonous calls.  Otherwise, the test
   * will be finished prematurely.
   * @param {string} url Url to load and wait for.
   * @param {function(chrome.automation.AutomationNode)} callback
   *     Called once the document is ready.
   */
  runWithLoadedTree: function(url, callback) {
    callback = this.newCallback(callback);
    chrome.automation.getDesktop(function(r) {
      var listener = function(evt) {
        if (evt.target.root.url != url)
          return;

        r.removeEventListener('focus', listener, true);
        r.removeEventListener('loadComplete', listener, true);
        callback && callback(evt.target);
        callback = null;
      };
      r.addEventListener('focus', listener, true);
      r.addEventListener('loadComplete', listener, true);
      var createParams = {active: true, url: url};
      chrome.tabs.create(createParams);
    }.bind(this));
  },
};
