// Copyright 2015 the V8 project authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef V8_COMPILER_COMPILATION_DEPENDENCIES_H_
#define V8_COMPILER_COMPILATION_DEPENDENCIES_H_

#include "src/compiler/js-heap-broker.h"
#include "src/objects.h"
#include "src/zone/zone-containers.h"

namespace v8 {
namespace internal {
namespace compiler {

// Collects and installs dependencies of the code that is being generated.
class V8_EXPORT_PRIVATE CompilationDependencies : public ZoneObject {
 public:
  CompilationDependencies(Isolate* isolate, Zone* zone);

  V8_WARN_UNUSED_RESULT bool Commit(Handle<Code> code);

  // Return the initial map of {function} and record the assumption that it
  // stays the intial map.
  MapRef DependOnInitialMap(const JSFunctionRef& function);

  // Record the assumption that {map} stays stable.
  void DependOnStableMap(const MapRef& map);

  // Record the assumption that {target_map} can be transitioned to, i.e., that
  // it does not become deprecated.
  void DependOnTransition(const MapRef& target_map);

  // Return the pretenure mode of {site} and record the assumption that it does
  // not change.
  PretenureFlag DependOnPretenureMode(const AllocationSiteRef& site);

  // Record the assumption that the field type of a field does not change. The
  // field is identified by the arguments.
  void DependOnFieldType(const MapRef& map, int descriptor);

  // Record the assumption that neither {cell}'s {CellType} changes, nor the
  // {IsReadOnly()} flag of {cell}'s {PropertyDetails}.
  void DependOnGlobalProperty(const PropertyCellRef& cell);

  // Record the assumption that the protector remains valid.
  void DependOnProtector(const PropertyCellRef& cell);

  // Record the assumption that {site}'s {ElementsKind} doesn't change.
  void DependOnElementsKind(const AllocationSiteRef& site);

  // Depend on the stability of (the maps of) all prototypes of every class in
  // {receiver_type} up to (and including) the {holder}.
  // TODO(neis): Fully brokerize!
  void DependOnStablePrototypeChains(
      JSHeapBroker* broker, Handle<Context> native_context,
      std::vector<Handle<Map>> const& receiver_maps, Handle<JSObject> holder);

  // Like DependOnElementsKind but also applies to all nested allocation sites.
  void DependOnElementsKinds(const AllocationSiteRef& site);

  // Exposed only for testing purposes.
  bool AreValid() const;

  // Exposed only because C++.
  class Dependency;

 private:
  Zone* zone_;
  ZoneForwardList<Dependency*> dependencies_;
};

}  // namespace compiler
}  // namespace internal
}  // namespace v8

#endif  // V8_COMPILER_COMPILATION_DEPENDENCIES_H_
