/*
 *  Copyright (c) 2018 The WebM project authors. All Rights Reserved.
 *
 *  Use of this source code is governed by a BSD-style license
 *  that can be found in the LICENSE file in the root of the source
 *  tree. An additional intellectual property rights grant can be found
 *  in the file PATENTS.  All contributing project authors may
 *  be found in the AUTHORS file in the root of the source tree.
 */

#ifndef VP9_ENCODER_VP9_PARTITION_MODELS_H_
#define VP9_ENCODER_VP9_PARTITION_MODELS_H_

#ifdef __cplusplus
extern "C" {
#endif

#define NN_MAX_HIDDEN_LAYERS 10
#define NN_MAX_NODES_PER_LAYER 128

// Neural net model config.
typedef struct {
  int num_inputs;         // Number of input nodes, i.e. features.
  int num_outputs;        // Number of output nodes.
  int num_hidden_layers;  // Number of hidden layers, maximum 10.
  // Number of nodes for each hidden layer.
  int num_hidden_nodes[NN_MAX_HIDDEN_LAYERS];
  // Weight parameters, indexed by layer.
  const float *weights[NN_MAX_HIDDEN_LAYERS + 1];
  // Bias parameters, indexed by layer.
  const float *bias[NN_MAX_HIDDEN_LAYERS + 1];
} NN_CONFIG;

// Partition search breakout model.
#define FEATURES 4
#define Q_CTX 3
#define RESOLUTION_CTX 2
static const float
    vp9_partition_breakout_weights_64[RESOLUTION_CTX][Q_CTX][FEATURES + 1] = {
      {
          {
              -0.016673f,
              -0.001025f,
              -0.000032f,
              0.000833f,
              1.94261885f - 2.1f,
          },
          {
              -0.160867f,
              -0.002101f,
              0.000011f,
              0.002448f,
              1.65738142f - 2.5f,
          },
          {
              -0.628934f,
              -0.011459f,
              -0.000009f,
              0.013833f,
              1.47982645f - 1.6f,
          },
      },
      {
          {
              -0.064309f,
              -0.006121f,
              0.000232f,
              0.005778f,
              0.7989465f - 5.0f,
          },
          {
              -0.314957f,
              -0.009346f,
              -0.000225f,
              0.010072f,
              2.80695581f - 5.5f,
          },
          {
              -0.635535f,
              -0.015135f,
              0.000091f,
              0.015247f,
              2.90381241f - 5.0f,
          },
      },
    };

static const float
    vp9_partition_breakout_weights_32[RESOLUTION_CTX][Q_CTX][FEATURES + 1] = {
      {
          {
              -0.010554f,
              -0.003081f,
              -0.000134f,
              0.004491f,
              1.68445992f - 3.5f,
          },
          {
              -0.051489f,
              -0.007609f,
              0.000016f,
              0.009792f,
              1.28089404f - 2.5f,
          },
          {
              -0.163097f,
              -0.013081f,
              0.000022f,
              0.019006f,
              1.36129403f - 3.2f,
          },
      },
      {
          {
              -0.024629f,
              -0.006492f,
              -0.000254f,
              0.004895f,
              1.27919173f - 4.5f,
          },
          {
              -0.083936f,
              -0.009827f,
              -0.000200f,
              0.010399f,
              2.73731065f - 4.5f,
          },
          {
              -0.279052f,
              -0.013334f,
              0.000289f,
              0.023203f,
              2.43595719f - 3.5f,
          },
      },
    };

static const float
    vp9_partition_breakout_weights_16[RESOLUTION_CTX][Q_CTX][FEATURES + 1] = {
      {
          {
              -0.013154f,
              -0.002404f,
              -0.000977f,
              0.008450f,
              2.57404566f - 5.5f,
          },
          {
              -0.019146f,
              -0.004018f,
              0.000064f,
              0.008187f,
              2.15043926f - 2.5f,
          },
          {
              -0.075755f,
              -0.010858f,
              0.000030f,
              0.024505f,
              2.06848121f - 2.5f,
          },
      },
      {
          {
              -0.007636f,
              -0.002751f,
              -0.000682f,
              0.005968f,
              0.19225763f - 4.5f,
          },
          {
              -0.047306f,
              -0.009113f,
              -0.000518f,
              0.016007f,
              2.61068869f - 4.0f,
          },
          {
              -0.069336f,
              -0.010448f,
              -0.001120f,
              0.023083f,
              1.47591054f - 5.5f,
          },
      },
    };

static const float vp9_partition_breakout_weights_8[RESOLUTION_CTX][Q_CTX]
                                                   [FEATURES + 1] = {
                                                     {
                                                         {
                                                             -0.011807f,
                                                             -0.009873f,
                                                             -0.000931f,
                                                             0.034768f,
                                                             1.32254851f - 2.0f,
                                                         },
                                                         {
                                                             -0.003861f,
                                                             -0.002701f,
                                                             0.000100f,
                                                             0.013876f,
                                                             1.96755111f - 1.5f,
                                                         },
                                                         {
                                                             -0.013522f,
                                                             -0.008677f,
                                                             -0.000562f,
                                                             0.034468f,
                                                             1.53440356f - 1.5f,
                                                         },
                                                     },
                                                     {
                                                         {
                                                             -0.003221f,
                                                             -0.002125f,
                                                             0.000993f,
                                                             0.012768f,
                                                             0.03541421f - 2.0f,
                                                         },
                                                         {
                                                             -0.006069f,
                                                             -0.007335f,
                                                             0.000229f,
                                                             0.026104f,
                                                             0.17135315f - 1.5f,
                                                         },
                                                         {
                                                             -0.039894f,
                                                             -0.011419f,
                                                             0.000070f,
                                                             0.061817f,
                                                             0.6739977f - 1.5f,
                                                         },
                                                     },
                                                   };
#undef FEATURES
#undef Q_CTX
#undef RESOLUTION_CTX

// Rectangular partition search pruning model.
#define FEATURES 17
#define LABELS 4
static const float vp9_rect_part_nn_weights_16_layer0[FEATURES * 32] = {
  1.262885f,  -0.533345f, -0.161280f, 0.106098f,  0.194799f,  0.003600f,
  0.394783f,  -0.053954f, 0.264474f,  -0.016651f, 0.376765f,  0.221471f,
  0.489799f,  0.054924f,  0.018292f,  0.037633f,  -0.053430f, 1.092426f,
  0.205791f,  -0.055661f, -0.227335f, 0.301274f,  -0.169917f, 0.100426f,
  0.254388f,  0.103465f,  0.189560f,  0.116479f,  1.647195f,  -0.667044f,
  0.067795f,  -0.044580f, 0.019428f,  0.072938f,  -0.797569f, -0.077539f,
  -0.225636f, 0.262883f,  -1.048009f, 0.210118f,  -0.416156f, -0.143741f,
  -0.296985f, 0.205918f,  -0.517383f, -0.118527f, -0.396606f, -0.113128f,
  -0.279468f, 0.096141f,  -0.342051f, -0.337036f, 0.143222f,  -0.860280f,
  0.137169f,  0.339767f,  -0.336076f, 0.071988f,  0.251557f,  -0.004068f,
  0.170734f,  0.237283f,  -0.332443f, 0.073643f,  0.375357f,  0.220407f,
  0.150708f,  -0.176979f, 0.265786f,  -0.105878f, -0.337465f, -0.000491f,
  0.234308f,  -0.098973f, 0.129038f,  -0.205936f, -0.034793f, -0.106981f,
  0.009974f,  0.037861f,  -0.282874f, -0.354414f, 0.023021f,  -0.266749f,
  -0.041762f, -0.721725f, 0.182262f,  -0.273945f, 0.123722f,  -0.036749f,
  -0.788645f, -0.081560f, -0.472226f, 0.004654f,  -0.756766f, -0.132186f,
  1.085412f,  -0.221324f, -0.072577f, -0.172834f, -0.104831f, -1.391641f,
  -0.345893f, 0.194442f,  -0.306583f, -0.041813f, -0.267635f, -0.218568f,
  -0.178452f, 0.044421f,  -0.128042f, -0.094797f, -0.253724f, 0.273931f,
  0.144843f,  -0.401416f, -0.014354f, -0.348929f, 0.123550f,  0.494504f,
  -0.007050f, -0.143830f, 0.111292f,  0.211057f,  -1.579988f, 0.117744f,
  -1.732487f, 0.009320f,  -1.162696f, 0.176687f,  -0.705609f, 0.524827f,
  0.089822f,  0.082976f,  -0.023681f, 0.006120f,  -0.907175f, -0.026273f,
  0.019027f,  0.027170f,  -0.462563f, -0.535335f, 0.202231f,  0.709803f,
  -0.112251f, -1.213869f, 0.225714f,  0.323785f,  -0.518254f, -0.014235f,
  -0.070790f, -0.369589f, 0.373399f,  0.002738f,  0.175113f,  0.084529f,
  -0.101586f, -0.018978f, 0.773392f,  -0.673230f, -0.549279f, 0.790196f,
  0.658609f,  -0.826831f, -0.514211f, 0.575341f,  -0.711311f, 0.276289f,
  -0.435715f, 0.392986f,  -0.079298f, -0.318719f, 0.188429f,  -0.114366f,
  0.172527f,  -0.261721f, -0.216761f, 0.163822f,  -0.189374f, -0.391901f,
  0.142013f,  -0.135046f, 0.144419f,  0.053887f,  0.074673f,  -0.290791f,
  -0.039560f, -0.103830f, -0.330263f, -0.042091f, 0.050646f,  -0.057466f,
  -0.069064f, -0.412864f, 0.071097f,  0.126693f,  0.175397f,  -0.168485f,
  0.018129f,  -0.419188f, -0.272024f, -0.436859f, -0.425711f, -0.024382f,
  0.248042f,  -0.169090f, -0.346878f, -0.070926f, 0.292278f,  -0.197610f,
  -0.218286f, 0.290846f,  0.297843f,  0.247394f,  -0.160736f, 0.110314f,
  0.276000f,  -0.301676f, -0.232816f, -0.127576f, -0.174457f, -0.124503f,
  0.264880f,  -0.332379f, 0.012659f,  -0.197333f, 0.604700f,  0.801582f,
  0.758702f,  0.691880f,  0.440917f,  0.773548f,  0.064242f,  1.147508f,
  -0.127543f, -0.189628f, -0.122994f, -0.226776f, -0.053531f, -0.187548f,
  0.226554f,  -0.273451f, 0.011751f,  0.009133f,  0.185091f,  0.003031f,
  0.000525f,  0.221829f,  0.331550f,  -0.202558f, -0.286550f, 0.100683f,
  0.268818f,  0.179971f,  -0.050016f, 0.579665f,  0.015911f,  0.033068f,
  0.077768f,  -0.017757f, -1.411251f, 0.051519f,  -1.745767f, 0.011258f,
  -1.947372f, 0.111396f,  -1.112755f, -0.008989f, -0.006211f, -0.002098f,
  -0.015236f, -0.095697f, -0.095820f, 0.044622f,  -0.112096f, 0.060000f,
  0.138957f,  -0.462708f, 0.590790f,  -0.021405f, -0.283744f, -1.141749f,
  0.213121f,  -0.332311f, -0.314090f, -0.789311f, 0.157605f,  -0.438019f,
  0.642189f,  -0.340764f, -0.996025f, 0.109871f,  0.106128f,  -0.010505f,
  -0.117233f, -0.223194f, 0.344105f,  -0.308754f, 0.386020f,  -0.305270f,
  -0.538281f, -0.270720f, -0.101688f, 0.207580f,  0.237153f,  -0.055730f,
  0.842779f,  0.393543f,  0.007886f,  -0.318167f, 0.603768f,  0.388241f,
  0.421536f,  0.632080f,  0.423965f,  0.371472f,  0.456827f,  0.488134f,
  0.358997f,  0.032621f,  -0.017104f, 0.032198f,  0.113266f,  -0.312277f,
  0.178189f,  0.234180f,  0.134271f,  -0.414889f, 0.774141f,  -0.225043f,
  0.614052f,  -0.279921f, 1.329141f,  -0.140827f, 0.797267f,  -0.171361f,
  0.066205f,  0.339976f,  0.015223f,  0.193725f,  -0.245067f, -0.035578f,
  -0.084043f, 0.086756f,  0.029478f,  -0.845370f, 0.388613f,  -1.215236f,
  0.304573f,  -0.439884f, -0.293969f, -0.107988f, -0.267837f, -0.695339f,
  -0.702099f, 0.359047f,  0.511730f,  1.429516f,  0.216959f,  -0.313828f,
  0.068062f,  -0.124917f, -0.648327f, -0.308411f, -0.378467f, -0.429288f,
  -0.032415f, -0.357005f, 0.170068f,  0.161167f,  -0.250280f, -0.320468f,
  -0.408987f, -0.201496f, -0.155996f, 0.021067f,  0.141083f,  -0.202733f,
  -0.130953f, -0.278148f, -0.042051f, 0.070576f,  0.009982f,  -0.044326f,
  -0.346851f, -0.255397f, -0.346456f, 0.281781f,  0.001618f,  0.120648f,
  0.297140f,  0.198343f,  0.186104f,  0.183548f,  -0.344482f, 0.182258f,
  0.291003f,  -0.330228f, -0.048174f, 0.133694f,  0.264582f,  0.229671f,
  -0.167251f, -0.316040f, 0.191829f,  0.153417f,  -0.345158f, -0.212790f,
  -0.878872f, -0.313099f, -0.028368f, 0.065869f,  -0.695388f, 1.102812f,
  -0.605539f, 0.400680f,  -0.350120f, -0.432965f, 0.034553f,  -0.693476f,
  -0.045708f, 0.492409f,  -0.043825f, -0.430522f, 0.071159f,  -0.317376f,
  -1.164842f, 0.112394f,  0.034137f,  -0.611882f, 0.251020f,  -0.245113f,
  0.286093f,  -0.187883f, 0.340263f,  -0.211592f, -0.065706f, -0.332148f,
  0.104026f,  -0.003206f, 0.036397f,  0.206499f,  0.161962f,  0.037663f,
  -0.313039f, -0.199837f, 0.117952f,  -0.182145f, -0.343724f, 0.017625f,
  0.033427f,  -0.288075f, -0.101873f, -0.083378f, 0.147870f,  0.049598f,
  -0.241824f, 0.070494f,  0.140942f,  -0.013795f, 0.020023f,  -0.192213f,
  -0.320505f, -0.193072f, 0.147260f,  0.311352f,  0.053486f,  0.183716f,
  0.142535f,  0.294333f,  -0.054853f, 0.293314f,  -0.025398f, 0.190815f,
  -0.137574f, -0.191864f, -0.190950f, -0.205988f, -0.199046f, -0.017582f,
  -0.149347f, 0.131040f,  0.006854f,  -0.350732f, 0.113301f,  -0.194371f,
  -0.296885f, -0.249199f, -0.193946f, 0.116150f,  -0.310411f, -0.325851f,
  -0.053275f, -0.063419f, 0.204170f,  -0.091940f, -0.146229f, 0.298173f,
  0.053349f,  -0.368540f, 0.235629f,  -0.317825f, -0.107304f, -0.114618f,
  0.058709f,  -0.272070f, 0.076224f,  0.110668f,  -0.193282f, -0.135440f,
  -0.267950f, -0.102285f, 0.102699f,  -0.159082f, 0.262721f,  -0.263227f,
  0.094509f,  -0.113405f, 0.069888f,  -0.169665f, 0.070800f,  0.035432f,
  0.054243f,  0.264229f,  0.117416f,  0.091568f,  -0.022069f, -0.069214f,
  0.124543f,  0.070413f,  -0.039343f, 0.082823f,  -0.838348f, 0.153727f,
  -0.000947f, 0.270348f,  -1.404952f, -0.159680f, -0.234320f, 0.061023f,
  0.271660f,  -0.541834f, 0.570828f,  -0.277254f,
};

static const float vp9_rect_part_nn_bias_16_layer0[32] = {
  0.045740f,  0.292685f,  -0.754007f, -0.150412f, -0.006171f, 0.005915f,
  0.000167f,  0.322797f,  -0.381793f, 0.349786f,  0.003878f,  -0.307203f,
  0.000000f,  0.029122f,  0.000000f,  0.625494f,  0.302105f,  -0.362807f,
  -0.034002f, -0.573278f, 0.240021f,  0.083965f,  0.000000f,  -0.018979f,
  -0.147739f, -0.036990f, 0.000000f,  0.000000f,  -0.026790f, -0.000036f,
  -0.073448f, 0.398328f,
};

static const float vp9_rect_part_nn_weights_16_layer1[32 * LABELS] = {
  0.095090f,  0.831754f,  0.484433f,  0.472945f,  0.086165f,  -0.442388f,
  0.176263f,  -0.760247f, 0.419932f,  -0.131377f, 0.075814f,  0.089844f,
  -0.294718f, 0.299808f,  -0.318435f, -0.623205f, -0.346703f, 0.494356f,
  0.949221f,  0.524653f,  0.044095f,  0.428540f,  0.402571f,  -0.216920f,
  0.423915f,  1.023334f,  -0.366449f, 0.395057f,  0.057576f,  0.094019f,
  0.247685f,  -0.007200f, -0.420023f, -0.728965f, -0.063040f, -0.071321f,
  0.209298f,  0.486625f,  -0.244375f, 0.263219f,  -0.250463f, -0.260301f,
  0.068579f,  0.177644f,  -0.155311f, -0.027606f, -0.101614f, 0.553046f,
  -0.462729f, -0.237568f, -0.589316f, 0.045182f,  0.551759f,  -0.196872f,
  0.183040f,  0.054341f,  0.252784f,  -0.536486f, -0.024425f, 0.154942f,
  -0.086636f, 0.360416f,  0.214773f,  -0.170876f, -0.363522f, -0.464099f,
  0.145494f,  -0.099329f, 0.343718f,  0.286427f,  0.085540f,  -0.105182f,
  0.155543f,  0.290939f,  -0.067069f, 0.228399f,  0.178247f,  0.113031f,
  -0.067336f, 0.441062f,  0.132364f,  -0.263403f, -0.263925f, -0.083613f,
  -0.268577f, -0.204442f, 0.052526f,  0.334787f,  -0.064285f, -0.197875f,
  0.296405f,  0.396440f,  0.033231f,  0.229087f,  0.118289f,  0.490894f,
  -0.527582f, -0.897206f, -0.325708f, -0.433018f, -0.053989f, 0.223814f,
  -0.352319f, 0.772440f,  -0.108648f, -0.082859f, -0.342718f, 0.033022f,
  -0.309199f, -0.560337f, 0.208476f,  0.520309f,  -0.241035f, -0.560391f,
  -1.268968f, -0.267567f, 0.129461f,  -0.385547f, 0.080142f,  0.065785f,
  -0.159324f, -0.580704f, -0.315150f, -0.224900f, -0.110807f, -0.230163f,
  0.307266f,  0.153446f,
};

static const float vp9_rect_part_nn_bias_16_layer1[LABELS] = {
  -0.455437f,
  0.255310f,
  0.452974f,
  -0.278733f,
};

static const NN_CONFIG vp9_rect_part_nnconfig_16 = {
  FEATURES,  // num_inputs
  LABELS,    // num_outputs
  1,         // num_hidden_layers
  {
      32,
  },  // num_hidden_nodes
  {
      vp9_rect_part_nn_weights_16_layer0,
      vp9_rect_part_nn_weights_16_layer1,
  },
  {
      vp9_rect_part_nn_bias_16_layer0,
      vp9_rect_part_nn_bias_16_layer1,
  },
};

static const float vp9_rect_part_nn_weights_32_layer0[FEATURES * 32] = {
  0.735110f,  -0.238477f, 0.101978f,  0.311671f,  -0.123833f, 1.596506f,
  -0.341982f, -0.480170f, -0.247587f, 0.613159f,  -0.279899f, -0.740856f,
  0.499051f,  0.039041f,  0.056763f,  0.258874f,  0.470812f,  -0.121635f,
  -0.318852f, -0.098677f, -0.214714f, -0.159974f, -0.305400f, -0.344477f,
  -0.260653f, -0.007737f, -0.053016f, -0.158079f, 0.151911f,  -0.057685f,
  -0.230948f, -0.165940f, -0.127591f, -0.192084f, 1.890390f,  -0.315123f,
  -0.714531f, -0.015355f, 0.186437f,  0.305504f,  0.035343f,  -0.556783f,
  0.239364f,  -0.297789f, 0.202735f,  -0.707576f, 0.710250f,  0.223346f,
  -0.291511f, 0.235778f,  0.455338f,  -0.059402f, 0.084530f,  -0.115117f,
  -0.103696f, -0.192821f, 0.114579f,  -0.223487f, 0.306864f,  0.021887f,
  -0.028040f, 0.087866f,  0.038870f,  -0.081742f, -0.056052f, -0.130837f,
  0.201058f,  0.293391f,  1.880344f,  0.339162f,  0.040928f,  -0.503942f,
  0.476333f,  0.259272f,  0.629416f,  0.869369f,  0.622841f,  1.012843f,
  0.715795f,  1.958844f,  -1.697462f, 0.071334f,  0.074189f,  0.014585f,
  -0.002536f, 0.021900f,  0.151883f,  0.169501f,  -0.333018f, -0.247512f,
  -0.418575f, -0.473960f, -0.004501f, -0.280939f, -0.162188f, -0.355632f,
  0.136654f,  -0.100967f, -0.350435f, -0.135386f, 0.037237f,  0.136982f,
  -0.084157f, -0.073248f, 0.021792f,  0.077429f,  -0.083042f, -3.169569f,
  0.016261f,  -3.351328f, 0.021120f,  -3.572247f, 0.023870f,  -4.312754f,
  0.040973f,  -0.038328f, -0.015052f, 0.017702f,  0.101427f,  0.115458f,
  -0.304792f, 0.021826f,  -0.157998f, 0.341022f,  -0.013465f, 0.105076f,
  -0.261465f, 0.318730f,  0.065701f,  0.314879f,  -0.064785f, 0.282824f,
  0.100542f,  0.057260f,  -0.003756f, -0.026214f, -0.264641f, 0.275545f,
  -0.049201f, -0.283015f, -0.057363f, 0.183570f,  0.243161f,  -0.255764f,
  0.099747f,  -0.156157f, -0.262494f, 0.231521f,  -0.262617f, -0.186096f,
  0.171720f,  0.018983f,  -0.145545f, 0.197662f,  -0.001502f, -0.267526f,
  0.001960f,  0.003260f,  0.045237f,  -0.377174f, -0.042499f, -0.015278f,
  -0.196779f, -0.262797f, -0.318427f, -0.126092f, -0.339723f, 0.205288f,
  -0.544284f, -0.507896f, -0.316622f, -0.090312f, -0.250917f, -0.337263f,
  -0.220199f, -0.296591f, -0.116816f, 0.052381f,  0.145681f,  0.016521f,
  -0.093549f, -0.097822f, 0.023140f,  -0.010346f, 0.036181f,  0.145826f,
  -0.139123f, -0.462638f, -0.007315f, 0.156533f,  -0.102787f, 0.143586f,
  -0.092094f, -0.144220f, -0.168994f, -0.045833f, 0.021628f,  -0.421794f,
  -0.055857f, 0.217931f,  -0.061937f, -0.028768f, -0.078250f, -0.426939f,
  -0.223118f, -0.230080f, -0.194988f, -0.197673f, -0.020918f, 0.139945f,
  0.186951f,  -0.071317f, -0.084007f, -0.138597f, 0.101950f,  0.093870f,
  0.153226f,  0.017799f,  -0.088539f, -0.037796f, 0.340412f,  0.183305f,
  0.391880f,  -1.127417f, 0.132762f,  -0.228565f, 0.399035f,  0.017483f,
  -0.041619f, 0.017849f,  0.092340f,  0.054204f,  0.681185f,  0.421034f,
  0.112520f,  -0.040618f, -0.040148f, -0.360647f, 0.053555f,  0.192854f,
  0.076968f,  -0.179224f, -0.081617f, -0.287661f, -0.191072f, -0.310227f,
  -0.332226f, -0.039786f, -0.247795f, -0.232201f, -0.333533f, -0.077995f,
  -0.471732f, 0.051829f,  0.090488f,  0.142465f,  -0.120490f, -0.286151f,
  -0.049117f, -0.251082f, 0.211884f,  -0.223366f, 0.063565f,  0.229938f,
  -0.059348f, -0.029573f, -0.064303f, -0.156148f, 0.086958f,  -0.297613f,
  -0.125107f, 0.062718f,  0.339137f,  -0.218896f, -0.057290f, -0.236670f,
  -0.143783f, -0.119429f, 0.242320f,  -0.323464f, -0.178377f, 0.238275f,
  -0.025042f, 0.074798f,  0.111329f,  -0.299773f, -0.151748f, -0.261607f,
  0.215626f,  0.202243f,  -0.121896f, -0.024283f, -0.293854f, -0.018232f,
  -0.012629f, -0.199297f, -0.060595f, 0.432339f,  -0.158735f, -0.028380f,
  0.326639f,  0.222546f,  -0.218135f, -0.495955f, -0.015055f, -0.104206f,
  -0.268823f, 0.116765f,  0.041769f,  -0.187095f, 0.225090f,  0.198195f,
  0.001502f,  -0.219212f, -0.244779f, -0.017690f, -0.033197f, -0.339813f,
  -0.325453f, 0.002499f,  -0.066113f, 0.043235f,  0.324275f,  -0.630642f,
  -1.440551f, 0.174527f,  0.124619f,  -1.187345f, 1.372693f,  -0.278393f,
  -0.058673f, -0.286338f, 1.708757f,  -0.325094f, -0.543172f, -0.229411f,
  0.169927f,  0.175064f,  0.198321f,  0.117351f,  0.220882f,  0.138078f,
  -0.158000f, -0.286708f, 0.096046f,  -0.321788f, 0.206949f,  -0.014473f,
  -0.321234f, 0.100033f,  -0.108266f, 0.166824f,  0.032904f,  -0.065760f,
  -0.303896f, 0.180342f,  -0.301145f, -0.352554f, 0.149089f,  0.013277f,
  0.256019f,  -0.109770f, 1.832588f,  -0.132568f, 1.527658f,  -0.164252f,
  -0.857880f, -0.242694f, -0.553797f, 0.334023f,  -0.332759f, -0.166203f,
  -0.223175f, 0.007953f,  -0.175865f, -0.134590f, -0.023858f, -0.011983f,
  0.054403f,  -0.147054f, -0.176901f, -0.166893f, -0.292662f, -0.010569f,
  -0.041744f, -0.060398f, -0.237584f, 0.154246f,  -0.083270f, -0.314016f,
  -0.374736f, 0.100063f,  0.048401f,  -0.061952f, -0.178816f, 0.157243f,
  0.221991f,  -0.065035f, 0.098517f,  -0.190704f, -0.210613f, -0.274884f,
  -0.341442f, -0.205281f, 0.073644f,  0.130667f,  0.149194f,  -0.018172f,
  1.796154f,  -1.017806f, -0.169655f, 0.104239f,  0.344313f,  0.643042f,
  0.730177f,  0.270776f,  0.581631f,  -1.090649f, 0.707472f,  1.411035f,
  0.268739f,  0.178860f,  -0.062251f, -0.118611f, -0.215759f, 0.023485f,
  -0.105320f, 0.036396f,  -0.059604f, 0.090024f,  0.095224f,  -0.053497f,
  -0.084040f, 0.055836f,  0.111678f,  0.014886f,  -0.178380f, 0.079662f,
  -0.123580f, 0.057379f,  -0.409844f, -0.305386f, -0.987808f, -0.291094f,
  0.063966f,  0.263709f,  -0.337221f, 0.720093f,  0.105030f,  0.848950f,
  0.071835f,  0.228972f,  0.057705f,  -2.154561f, -0.201303f, -0.058856f,
  -0.020081f, 0.029375f,  0.234837f,  -0.001063f, 0.042527f,  0.014567f,
  -0.299420f, -0.289117f, 0.275219f,  0.263596f,  -0.186026f, -0.111364f,
  -0.118393f, -0.318778f, 0.010710f,  -0.286836f, -0.070330f, -0.049497f,
  0.093162f,  -0.298085f, 0.204761f,  -0.206633f, -0.009057f, -0.235372f,
  0.185300f,  -0.271814f, 0.281732f,  0.268149f,  -0.018967f, 0.162748f,
  -0.086694f, -0.063839f, -0.097473f, -0.280120f, 0.324688f,  0.157911f,
  -0.064794f, -0.266017f, -0.305608f, -0.196854f, -0.185767f, 0.199455f,
  0.102264f,  0.070866f,  0.172045f,  0.266433f,  -0.176167f, 0.251657f,
  -0.239220f, 0.229667f,  0.156115f,  -0.221345f, 0.270720f,  0.109367f,
  0.230352f,  -0.384561f, -0.026329f, 0.005928f,  -0.087685f, -0.097995f,
  -0.153864f, 0.117211f,  -0.226492f, -0.379832f, -0.201714f, 0.049707f,
  -0.292120f, 0.114074f,  -0.085307f, -0.485356f, -0.347405f, 0.089361f,
  -0.419273f, -0.320764f, -0.107254f, -0.274615f, -0.292991f, 0.095602f,
  -0.078789f, 0.138927f,  0.270813f,  0.205814f,  0.065003f,  0.169171f,
  0.056142f,  -0.005792f, 0.059483f,  0.060149f,
};

static const float vp9_rect_part_nn_bias_32_layer0[32] = {
  -1.749808f, 0.000000f,  0.239736f,  -0.000424f, 0.431792f,  -0.150833f,
  2.866760f,  0.000000f,  0.000000f,  -0.281434f, 0.000000f,  -0.150086f,
  0.000000f,  -0.008346f, -0.204104f, -0.006581f, 0.000000f,  -0.197006f,
  0.000000f,  -0.735287f, -0.028345f, -1.180116f, -0.106524f, 0.000000f,
  0.075879f,  -0.150966f, -2.438914f, 0.000000f,  -0.011775f, -0.024204f,
  -0.138235f, -0.123763f,
};

static const float vp9_rect_part_nn_weights_32_layer1[32 * LABELS] = {
  0.622235f,  0.264894f,  -0.424216f, 0.103989f,  1.401192f,  -0.063838f,
  -5.216846f, 0.329234f,  -0.293113f, 0.457519f,  -0.271899f, 0.043771f,
  -0.203823f, 0.573535f,  -0.192703f, 0.054939f,  0.163019f,  0.124803f,
  0.160664f,  0.385406f,  -0.091403f, 0.320204f,  0.101181f,  -0.157792f,
  -0.095555f, -0.255011f, 1.326614f,  -0.138076f, -0.082434f, -0.342442f,
  0.184067f,  -0.076395f, 0.050263f,  0.251065f,  0.291743f,  0.197838f,
  -0.950922f, 0.280202f,  2.904905f,  -0.219434f, 0.284386f,  0.375005f,
  0.193817f,  -0.298663f, -0.255364f, -0.297545f, 0.030518f,  -0.023892f,
  -0.396120f, -0.253027f, 0.237235f,  -0.550249f, -0.076817f, -0.201374f,
  0.292708f,  0.341936f,  -0.532215f, 0.180634f,  -0.943291f, -0.217179f,
  0.251611f,  -0.306310f, 0.229054f,  -0.350337f, -0.192707f, 0.146781f,
  0.409007f,  0.279088f,  -0.307357f, 0.199059f,  2.780962f,  0.163723f,
  -0.226445f, 0.242830f,  0.220356f,  -0.057621f, 0.196677f,  -0.179975f,
  -0.314636f, 0.218271f,  -0.278653f, -0.226286f, 0.034275f,  -0.320149f,
  0.154779f,  0.074937f,  -0.015650f, -0.281735f, -0.495227f, -0.075036f,
  -0.871024f, -0.350643f, 0.343468f,  0.095665f,  0.447121f,  -0.059040f,
  0.244757f,  0.223122f,  0.272544f,  0.129678f,  -1.700183f, 0.254869f,
  2.528983f,  0.217362f,  0.327765f,  -0.129369f, -0.003560f, -0.532537f,
  0.080216f,  -0.739488f, -0.299813f, 0.185421f,  0.265994f,  0.152268f,
  -0.401829f, -0.901380f, 0.347747f,  -0.524845f, -0.201163f, 0.063585f,
  -0.517479f, -0.077816f, -0.735739f, -0.161411f, -0.113607f, -0.306188f,
  0.190817f,  -0.362567f,
};

static const float vp9_rect_part_nn_bias_32_layer1[LABELS] = {
  -0.833530f,
  0.860502f,
  0.708645f,
  -1.083700f,
};

static const NN_CONFIG vp9_rect_part_nnconfig_32 = {
  FEATURES,  // num_inputs
  LABELS,    // num_outputs
  1,         // num_hidden_layers
  {
      32,
  },  // num_hidden_nodes
  {
      vp9_rect_part_nn_weights_32_layer0,
      vp9_rect_part_nn_weights_32_layer1,
  },
  {
      vp9_rect_part_nn_bias_32_layer0,
      vp9_rect_part_nn_bias_32_layer1,
  },
};

static const float vp9_rect_part_nn_weights_64_layer0[FEATURES * 32] = {
  0.029424f,  -0.295893f, -0.313259f, -0.090484f, -0.104946f, 0.121361f,
  0.137971f,  -0.137984f, -0.328158f, -0.137280f, -0.276995f, -0.153118f,
  0.187893f,  0.105787f,  -0.236591f, -0.114325f, -0.000708f, 1.936191f,
  0.048491f,  -0.026048f, -0.206916f, 0.830237f,  -0.152354f, 0.074191f,
  -0.153813f, 0.148942f,  -0.103457f, 0.028252f,  1.758264f,  -2.123016f,
  0.120182f,  0.049954f,  0.110450f,  -0.199360f, 0.642198f,  0.040225f,
  -0.140886f, 0.091833f,  -0.122788f, 1.172115f,  -0.833333f, -0.505218f,
  0.736050f,  -0.109958f, -0.839030f, -0.399916f, 1.029718f,  0.408977f,
  -0.836882f, 0.389683f,  -1.134413f, -1.529672f, -0.146351f, 0.089298f,
  0.083772f,  -0.697869f, 1.683311f,  -0.882446f, 0.494428f,  -0.122128f,
  0.659819f,  -0.057178f, -0.915390f, -0.192412f, 0.046613f,  0.010697f,
  0.040782f,  0.110807f,  -0.225332f, -0.327730f, -0.114825f, 0.063511f,
  0.050503f,  0.023602f,  0.006524f,  -0.274547f, -0.607145f, -0.143812f,
  -0.327689f, -0.333072f, -0.017138f, -0.183992f, -0.200622f, -0.262463f,
  -0.132799f, -0.018155f, -0.534214f, -0.385994f, 0.116278f,  -0.752879f,
  -0.090734f, -0.249152f, 0.071716f,  0.029603f,  -0.382456f, -0.122894f,
  1.349552f,  -0.885192f, 0.257903f,  -0.265945f, -0.045579f, 0.112247f,
  -0.122810f, -0.258285f, -0.145427f, -0.127442f, 0.072778f,  0.072549f,
  0.182149f,  0.239403f,  0.167205f,  -0.291616f, -0.281237f, 0.335735f,
  0.208511f,  -0.239628f, -0.022236f, -0.177370f, 0.207808f,  0.023535f,
  0.137455f,  0.016406f,  -0.138685f, 0.188732f,  0.205513f,  0.209787f,
  0.060592f,  0.239954f,  -0.128341f, -0.291585f, 0.022141f,  -0.311201f,
  -0.010199f, -0.314224f, -0.351915f, -0.079775f, -0.260028f, -0.015953f,
  0.007404f,  0.051589f,  0.019771f,  -2.337926f, 0.024596f,  -2.512399f,
  -0.023138f, -2.421380f, 0.016515f,  -3.269775f, 0.026844f,  -0.053660f,
  -0.013213f, -0.029248f, 0.114357f,  0.259100f,  -0.141749f, -0.106802f,
  -0.117323f, -0.294698f, -0.316012f, -0.328013f, 0.016459f,  0.136175f,
  0.223327f,  0.322312f,  -0.297297f, 0.118286f,  -0.317197f, -0.116692f,
  0.262236f,  -0.032443f, -0.392128f, -0.199989f, -0.383621f, 0.008347f,
  -0.079302f, -0.005529f, 0.049261f,  0.145948f,  -0.263592f, -0.317109f,
  0.260015f,  -0.499341f, -0.171764f, -0.017815f, 0.149186f,  0.178294f,
  -0.492198f, 0.016956f,  0.008067f,  -0.057734f, -0.189979f, -0.131489f,
  -0.163303f, 0.121378f,  -0.172272f, 0.125891f,  0.120654f,  0.071314f,
  0.117423f,  -0.242167f, 0.047170f,  0.234302f,  -0.355370f, -0.336112f,
  -0.255471f, -0.267792f, -0.135367f, -0.284411f, 0.254592f,  0.098749f,
  0.224989f,  0.258450f,  -0.306878f, 0.153551f,  -0.175806f, -0.244459f,
  -0.274922f, 0.254346f,  0.110309f,  0.036054f,  0.095133f,  -0.589646f,
  0.080543f,  0.154155f,  0.133797f,  -0.401518f, 0.798127f,  0.066742f,
  1.449216f,  0.282498f,  1.210638f,  -0.280643f, 0.572386f,  -0.308133f,
  -0.053143f, 0.008437f,  0.269565f,  0.347616f,  0.087180f,  -0.771104f,
  0.200800f,  0.157578f,  0.474128f,  -0.971488f, 0.193451f,  0.340339f,
  -0.123425f, 0.560754f,  -0.139621f, -0.281721f, -0.100162f, 0.250926f,
  0.281100f,  0.197680f,  0.138629f,  1.045823f,  0.339047f,  0.036698f,
  -0.159210f, 0.727869f,  -1.371850f, 0.116241f,  -2.180194f, 0.214055f,
  -0.213691f, 0.447957f,  -1.129966f, 0.543598f,  0.147599f,  0.060034f,
  -0.049415f, -0.095858f, 0.290599f,  0.059512f,  0.198343f,  -0.211903f,
  0.158736f,  -0.090220f, -0.221992f, 0.198320f,  0.028632f,  -0.408238f,
  -0.368266f, -0.218740f, -0.379023f, -0.173573f, -0.035179f, 0.240176f,
  0.237714f,  -0.417132f, -0.184989f, 0.046818f,  -0.016965f, -0.524012f,
  -0.094848f, -0.225678f, 0.021766f,  -0.028366f, 0.072343f,  -0.039980f,
  0.023334f,  -0.392397f, 0.164450f,  -0.201650f, -0.519754f, -0.023352f,
  -4.559466f, -0.115996f, 0.135844f,  0.152599f,  -0.111570f, 1.870310f,
  0.003522f,  1.893098f,  -0.134055f, 1.850787f,  0.085160f,  -2.203354f,
  0.380799f,  -0.074047f, 0.023760f,  0.077310f,  0.273381f,  -1.163135f,
  -0.024976f, 0.093252f,  0.011445f,  -0.129009f, -2.200677f, -0.013703f,
  -1.964109f, -0.027246f, -2.135679f, 0.049465f,  -3.879032f, 0.195114f,
  -0.018085f, 0.016755f,  0.036330f,  0.169138f,  0.003548f,  -0.028565f,
  -0.178196f, -0.020577f, -0.104330f, -0.270961f, -0.282822f, -0.228735f,
  -0.292561f, 0.271648f,  0.129171f,  0.376168f,  -0.265005f, -0.093002f,
  -0.185514f, 0.025598f,  0.055265f,  -0.212784f, -0.249005f, 0.051507f,
  -0.267868f, 0.162227f,  -0.237365f, 0.267479f,  -0.051543f, -0.288800f,
  -0.246119f, 0.216296f,  0.226888f,  -0.123005f, 0.068040f,  -0.096630f,
  -0.100500f, 0.161640f,  -0.349187f, -0.061229f, 0.042915f,  0.024949f,
  -0.083086f, -0.407249f, -0.428306f, -0.381137f, -0.508822f, 0.354796f,
  -0.612346f, -0.230076f, -0.734103f, -0.550571f, -0.318788f, -0.300091f,
  -0.336045f, -0.494406f, -0.206900f, 0.079942f,  0.149065f,  -0.533360f,
  0.940431f,  -0.078860f, 1.418633f,  -0.117527f, 1.349170f,  0.242658f,
  0.559328f,  0.258770f,  -0.014508f, -0.204775f, -0.292631f, 0.498345f,
  -0.274918f, 0.051670f,  0.157748f,  -0.179721f, -0.183330f, -0.393550f,
  -0.208848f, 0.060742f,  -0.159654f, 0.047757f,  -0.400256f, -0.084606f,
  -0.080619f, -0.359664f, -0.078305f, -0.455653f, 0.227624f,  -0.385606f,
  -0.060326f, -0.209831f, -0.077008f, 0.148862f,  0.209908f,  0.047655f,
  -0.342292f, -0.088375f, -0.115465f, 0.082700f,  0.036465f,  -0.001792f,
  -0.285730f, 0.114632f,  0.239254f,  -0.348543f, 0.044916f,  -0.299003f,
  -0.244756f, -0.180802f, 0.314253f,  -0.127788f, -0.221512f, 0.034787f,
  -0.208388f, 0.349156f,  0.265975f,  -0.068335f, 0.261372f,  0.146705f,
  -0.098729f, 0.293699f,  -0.111342f, 0.207402f,  -0.038772f, 0.124135f,
  -0.237450f, -0.191511f, -0.052240f, -0.237151f, 0.005013f,  0.139441f,
  -0.153634f, -0.021596f, -0.036220f, -0.077873f, -0.085995f, -0.254555f,
  -0.204382f, -0.082362f, 0.941796f,  0.253800f,  -0.957468f, 0.095795f,
  0.122046f,  -0.310364f, 0.087301f,  0.012704f,  0.193265f,  -0.058303f,
  0.250452f,  0.835269f,  0.507383f,  0.109957f,  -0.145028f, -0.114419f,
  -0.225618f, 0.132387f,  -0.063335f, -0.325776f, -0.346173f, -0.006653f,
  -0.133534f, -0.085549f, -0.050177f, 0.173103f,  0.025421f,  0.105512f,
  0.258036f,  0.153116f,  0.290202f,  -0.333699f, -0.072405f, -0.124069f,
  -0.241933f, -0.313318f, 0.013623f,  -0.237440f, -0.232228f, -0.170850f,
  -0.039212f, 0.162468f,  -0.330162f, -0.218462f, -0.287064f, -0.181673f,
  -0.161059f, 0.024664f,  -0.108642f, -0.231707f, 0.217994f,  -1.128878f,
  0.093010f,  0.101513f,  0.055895f,  -0.354538f, 0.844174f,  0.254335f,
  1.920298f,  -0.230777f, 0.798144f,  0.206425f,  0.580655f,  -0.177645f,
  -0.412061f, 0.112629f,  -0.476438f, 0.209436f,
};

static const float vp9_rect_part_nn_bias_64_layer0[32] = {
  0.000000f,  0.345406f,  -0.499542f, -1.718246f, -0.147443f, -0.408843f,
  -0.008997f, -0.107946f, 2.117510f,  0.000000f,  -0.141830f, -0.049079f,
  0.000000f,  -1.331136f, -1.417843f, -0.485054f, -0.100856f, -0.230750f,
  -2.574372f, 2.310627f,  -0.030363f, 0.000000f,  -0.310119f, -1.314316f,
  -0.108766f, -0.107918f, 0.000000f,  0.000000f,  0.093643f,  0.000000f,
  0.000000f,  -0.902343f,
};

static const float vp9_rect_part_nn_weights_64_layer1[32 * LABELS] = {
  0.404567f,  1.168492f,  0.051714f,  0.827941f,  0.135334f,  0.456922f,
  -0.370524f, 0.062865f,  -3.076300f, -0.290613f, 0.280029f,  -0.101778f,
  0.250216f,  0.347721f,  0.466400f,  0.030845f,  0.114570f,  0.089456f,
  1.519938f,  -3.493788f, 0.264212f,  -0.109125f, 0.306644f,  0.368206f,
  -0.052168f, -0.229630f, -0.339932f, -0.080472f, 0.319845f,  0.143818f,
  -0.172595f, 0.372777f,  -0.082072f, -0.505781f, -0.288321f, -0.473028f,
  -0.027567f, -0.034329f, -0.291965f, -0.063262f, 1.721741f,  0.118914f,
  0.183681f,  0.041611f,  0.266371f,  0.005896f,  -0.484705f, 0.665535f,
  -0.240945f, -0.017963f, -1.409440f, 2.031976f,  0.240327f,  -0.116604f,
  0.273245f,  -0.170570f, -0.085491f, -0.340315f, -0.209651f, -0.217460f,
  -0.249373f, 0.009193f,  0.009467f,  -0.272909f, 0.308472f,  -0.551173f,
  0.168374f,  -0.583229f, 0.140082f,  -0.585715f, -0.010929f, 0.159779f,
  1.438104f,  0.293111f,  -0.053339f, -0.101828f, -0.280573f, -0.211265f,
  -0.323605f, -0.540908f, 0.101366f,  -0.005288f, -1.517046f, 2.078767f,
  0.215597f,  0.144012f,  0.315888f,  -0.251324f, 0.150482f,  -0.137871f,
  0.235116f,  -0.194202f, -0.153475f, -0.312384f, -0.375510f, 0.336488f,
  -0.379837f, -1.004979f, -0.312587f, -0.406174f, 0.154290f,  -0.539766f,
  -0.230074f, 0.303564f,  0.719439f,  -0.235108f, -0.204978f, 0.399229f,
  0.290222f,  -0.278713f, -0.667069f, -0.420550f, 0.164893f,  -0.459689f,
  -1.035368f, 0.818909f,  0.275137f,  -0.291006f, -0.061505f, 0.052737f,
  -0.084871f, -0.348335f, 0.312544f,  0.120753f,  -0.707222f, -0.010050f,
  -0.137148f, -0.351765f,
};

static const float vp9_rect_part_nn_bias_64_layer1[LABELS] = {
  -0.926768f,
  0.765832f,
  0.663683f,
  -0.621865f,
};

static const NN_CONFIG vp9_rect_part_nnconfig_64 = {
  FEATURES,  // num_inputs
  LABELS,    // num_outputs
  1,         // num_hidden_layers
  {
      32,
  },  // num_hidden_nodes
  {
      vp9_rect_part_nn_weights_64_layer0,
      vp9_rect_part_nn_weights_64_layer1,
  },
  {
      vp9_rect_part_nn_bias_64_layer0,
      vp9_rect_part_nn_bias_64_layer1,
  },
};
#undef FEATURES
#undef LABELS

#define FEATURES 7
// Partition pruning model(neural nets).
static const float vp9_partition_nn_weights_64x64_layer0[FEATURES * 8] = {
  -3.571348f, 0.014835f,  -3.255393f, -0.098090f, -0.013120f, 0.000221f,
  0.056273f,  0.190179f,  -0.268130f, -1.828242f, -0.010655f, 0.937244f,
  -0.435120f, 0.512125f,  1.610679f,  0.190816f,  -0.799075f, -0.377348f,
  -0.144232f, 0.614383f,  -0.980388f, 1.754150f,  -0.185603f, -0.061854f,
  -0.807172f, 1.240177f,  1.419531f,  -0.438544f, -5.980774f, 0.139045f,
  -0.032359f, -0.068887f, -1.237918f, 0.115706f,  0.003164f,  2.924212f,
  1.246838f,  -0.035833f, 0.810011f,  -0.805894f, 0.010966f,  0.076463f,
  -4.226380f, -2.437764f, -0.010619f, -0.020935f, -0.451494f, 0.300079f,
  -0.168961f, -3.326450f, -2.731094f, 0.002518f,  0.018840f,  -1.656815f,
  0.068039f,  0.010586f,
};

static const float vp9_partition_nn_bias_64x64_layer0[8] = {
  -3.469882f, 0.683989f, 0.194010f,  0.313782f,
  -3.153335f, 2.245849f, -1.946190f, -3.740020f,
};

static const float vp9_partition_nn_weights_64x64_layer1[8] = {
  -8.058566f, 0.108306f, -0.280620f, -0.818823f,
  -6.445117f, 0.865364f, -1.127127f, -8.808660f,
};

static const float vp9_partition_nn_bias_64x64_layer1[1] = {
  6.46909416f,
};

static const NN_CONFIG vp9_partition_nnconfig_64x64 = {
  FEATURES,  // num_inputs
  1,         // num_outputs
  1,         // num_hidden_layers
  {
      8,
  },  // num_hidden_nodes
  {
      vp9_partition_nn_weights_64x64_layer0,
      vp9_partition_nn_weights_64x64_layer1,
  },
  {
      vp9_partition_nn_bias_64x64_layer0,
      vp9_partition_nn_bias_64x64_layer1,
  },
};

static const float vp9_partition_nn_weights_32x32_layer0[FEATURES * 8] = {
  -0.295437f, -4.002648f, -0.205399f, -0.060919f, 0.708037f,  0.027221f,
  -0.039137f, -0.907724f, -3.151662f, 0.007106f,  0.018726f,  -0.534928f,
  0.022744f,  0.000159f,  -1.717189f, -3.229031f, -0.027311f, 0.269863f,
  -0.400747f, -0.394366f, -0.108878f, 0.603027f,  0.455369f,  -0.197170f,
  1.241746f,  -1.347820f, -0.575636f, -0.462879f, -2.296426f, 0.196696f,
  -0.138347f, -0.030754f, -0.200774f, 0.453795f,  0.055625f,  -3.163116f,
  -0.091003f, -0.027028f, -0.042984f, -0.605185f, 0.143240f,  -0.036439f,
  -0.801228f, 0.313409f,  -0.159942f, 0.031267f,  0.886454f,  -1.531644f,
  -0.089655f, 0.037683f,  -0.163441f, -0.130454f, -0.058344f, 0.060011f,
  0.275387f,  1.552226f,
};

static const float vp9_partition_nn_bias_32x32_layer0[8] = {
  -0.838372f, -2.609089f, -0.055763f, 1.329485f,
  -1.297638f, -2.636622f, -0.826909f, 1.012644f,
};

static const float vp9_partition_nn_weights_32x32_layer1[8] = {
  -1.792632f, -7.322353f, -0.683386f, 0.676564f,
  -1.488118f, -7.527719f, 1.240163f,  0.614309f,
};

static const float vp9_partition_nn_bias_32x32_layer1[1] = {
  4.97422546f,
};

static const NN_CONFIG vp9_partition_nnconfig_32x32 = {
  FEATURES,  // num_inputs
  1,         // num_outputs
  1,         // num_hidden_layers
  {
      8,
  },  // num_hidden_nodes
  {
      vp9_partition_nn_weights_32x32_layer0,
      vp9_partition_nn_weights_32x32_layer1,
  },
  {
      vp9_partition_nn_bias_32x32_layer0,
      vp9_partition_nn_bias_32x32_layer1,
  },
};

static const float vp9_partition_nn_weights_16x16_layer0[FEATURES * 8] = {
  -1.717673f, -4.718130f, -0.125725f, -0.183427f, -0.511764f, 0.035328f,
  0.130891f,  -3.096753f, 0.174968f,  -0.188769f, -0.640796f, 1.305661f,
  1.700638f,  -0.073806f, -4.006781f, -1.630999f, -0.064863f, -0.086410f,
  -0.148617f, 0.172733f,  -0.018619f, 2.152595f,  0.778405f,  -0.156455f,
  0.612995f,  -0.467878f, 0.152022f,  -0.236183f, 0.339635f,  -0.087119f,
  -3.196610f, -1.080401f, -0.637704f, -0.059974f, 1.706298f,  -0.793705f,
  -6.399260f, 0.010624f,  -0.064199f, -0.650621f, 0.338087f,  -0.001531f,
  1.023655f,  -3.700272f, -0.055281f, -0.386884f, 0.375504f,  -0.898678f,
  0.281156f,  -0.314611f, 0.863354f,  -0.040582f, -0.145019f, 0.029329f,
  -2.197880f, -0.108733f,
};

static const float vp9_partition_nn_bias_16x16_layer0[8] = {
  0.411516f,  -2.143737f, -3.693192f, 2.123142f,
  -1.356910f, -3.561016f, -0.765045f, -2.417082f,
};

static const float vp9_partition_nn_weights_16x16_layer1[8] = {
  -0.619755f, -2.202391f, -4.337171f, 0.611319f,
  0.377677f,  -4.998723f, -1.052235f, 1.949922f,
};

static const float vp9_partition_nn_bias_16x16_layer1[1] = {
  3.20981717f,
};

static const NN_CONFIG vp9_partition_nnconfig_16x16 = {
  FEATURES,  // num_inputs
  1,         // num_outputs
  1,         // num_hidden_layers
  {
      8,
  },  // num_hidden_nodes
  {
      vp9_partition_nn_weights_16x16_layer0,
      vp9_partition_nn_weights_16x16_layer1,
  },
  {
      vp9_partition_nn_bias_16x16_layer0,
      vp9_partition_nn_bias_16x16_layer1,
  },
};
#undef FEATURES

// Partition pruning model(linear).
static const float vp9_partition_feature_mean[24] = {
  303501.697372f, 3042630.372158f, 24.694696f, 1.392182f,
  689.413511f,    162.027012f,     1.478213f,  0.0,
  135382.260230f, 912738.513263f,  28.845217f, 1.515230f,
  544.158492f,    131.807995f,     1.436863f,  0.0f,
  43682.377587f,  208131.711766f,  28.084737f, 1.356677f,
  138.254122f,    119.522553f,     1.252322f,  0.0f,
};

static const float vp9_partition_feature_std[24] = {
  673689.212982f, 5996652.516628f, 0.024449f, 1.989792f,
  985.880847f,    0.014638f,       2.001898f, 0.0f,
  208798.775332f, 1812548.443284f, 0.018693f, 1.838009f,
  396.986910f,    0.015657f,       1.332541f, 0.0f,
  55888.847031f,  448587.962714f,  0.017900f, 1.904776f,
  98.652832f,     0.016598f,       1.320992f, 0.0f,
};

// Error tolerance: 0.01%-0.0.05%-0.1%
static const float vp9_partition_linear_weights[24] = {
  0.111736f, 0.289977f, 0.042219f, 0.204765f, 0.120410f, -0.143863f,
  0.282376f, 0.847811f, 0.637161f, 0.131570f, 0.018636f, 0.202134f,
  0.112797f, 0.028162f, 0.182450f, 1.124367f, 0.386133f, 0.083700f,
  0.050028f, 0.150873f, 0.061119f, 0.109318f, 0.127255f, 0.625211f,
};

#ifdef __cplusplus
}  // extern "C"
#endif

#endif  // VP9_ENCODER_VP9_PARTITION_MODELS_H_
