// Copyright (c) 2011 The Chromium OS Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <cmath>
#include <string>

#include <gtest/gtest.h>

#include "gestures/include/split_correcting_filter_interpreter.h"
#include "gestures/include/unittest_util.h"

using std::string;

namespace gestures {

class SplitCorrectingFilterInterpreterTest : public ::testing::Test {};

class SplitCorrectingFilterInterpreterTestInterpreter :
      public Interpreter {
 public:
  SplitCorrectingFilterInterpreterTestInterpreter()
      : Interpreter(NULL, NULL, false),
        expect_finger_ids_(true),
        iteration_(0),
        expect_warp_on_one_finger_only_(false) {}

  virtual void SyncInterpret(HardwareState* hwstate, stime_t* timeout) {
    if (expect_finger_ids_) {
      EXPECT_EQ(hwstate->finger_cnt, hwstate->touch_cnt);
      EXPECT_EQ(hwstate->finger_cnt, expected_ids_.size());
    }
    for (size_t i = 0; i < hwstate->finger_cnt; i++) {
      bool found = SetContainsValue(expected_ids_,
                                    hwstate->fingers[i].tracking_id);
      if (expect_finger_ids_)
        EXPECT_TRUE(found) << iteration_ << ","
                           << hwstate->fingers[i].tracking_id;
      if (expect_warp_on_one_finger_only_) {
        EXPECT_EQ(hwstate->finger_cnt == 1,
                  (hwstate->fingers[i].flags &
                   (GESTURES_FINGER_WARP_X | GESTURES_FINGER_WARP_Y)) ==
                  (GESTURES_FINGER_WARP_X | GESTURES_FINGER_WARP_Y));
      }
    }
    iteration_++;
  }

  virtual void HandleTimer(stime_t now, stime_t* timeout) {}

  set<short, kMaxFingers> expected_ids_;
  bool expect_finger_ids_;
  size_t iteration_;
  bool expect_warp_on_one_finger_only_;
};

struct InputEventWithExpectations {
  FingerState fs[kMaxFingers];
  short out_ids[kMaxFingers];
};

namespace {

void DoTest(InputEventWithExpectations* events, size_t events_len, bool t5r2) {
  SplitCorrectingFilterInterpreterTestInterpreter* base_interpreter
      = new SplitCorrectingFilterInterpreterTestInterpreter;
  SplitCorrectingFilterInterpreter interpreter(NULL, base_interpreter, NULL);
  interpreter.Enable();

  HardwareProperties hwprops = {
    0, 0, 100, 100,  // left, top, right, bottom
    1,  // res_x
    1,  // res_y
    133,  // screen_x_dpi
    133,  // screen_y_dpi
    -1,  // orientation minimum
    2,   // orientation maximum
    5,  // max finger cnt
    static_cast<unsigned short>(t5r2 ? 2 : 5),  // max touch cnt
    t5r2 ? 1u : 0u,  // supports_t5r2
    0,   // support_semi_mt
    1,  // is_button_pad
    0  // has_wheel
  };
  TestInterpreterWrapper wrapper(&interpreter, &hwprops);

  for (size_t i = 0; i < events_len; i++) {
    InputEventWithExpectations* event = &events[i];
    // Get finger count
    unsigned short finger_cnt = 0;
    for (size_t fidx = 0;
         fidx < arraysize(event->fs) && event->fs[fidx].tracking_id >= 0;
         fidx++)
      finger_cnt += 1;
    // Set up hardware state
    HardwareState hs = {
        0.0, 0, finger_cnt, finger_cnt, event->fs, 0, 0, 0, 0
    };
    // Set up expectations
    base_interpreter->expected_ids_.clear();
    for (size_t outidx = 0;
         outidx < arraysize(event->out_ids) && event->out_ids[outidx] >= 0;
         outidx++)
      base_interpreter->expected_ids_.insert(event->out_ids[outidx]);
    stime_t timestamp = -1.0;
    wrapper.SyncInterpret(&hs, &timestamp);
  }
}

}  // namespace {}

// Test that a full set of FingerStates doesn't cause a hang
TEST(SplitCorrectingFilterInterpreterTest, FullFingerSetTest) {
  InputEventWithExpectations events[2];

  for (size_t i = 0; i < kMaxFingers; i++) {
    short id = i + 100;
    events[0].fs[i].tracking_id = id;
    events[0].out_ids[i] = id;
  }

  // The test requires a second event to validate the finger states
  events[1] = events[0];

  DoTest(events, arraysize(events), false);
}

struct InCoords {
  short id_;
  float x_, y_, pressure_;
};

struct FalseMergeInputs {
  stime_t timestamp;
  InCoords in[2];
};

// Tests that when two fingers are scrolling and it looks like a merge
// at the beginning, we break out of merging.
TEST(SplitCorrectingFilterInterpreterTest, FalseMergeTest) {
  FalseMergeInputs inputs[] = {
    { 1324.3365, { { 66, 58.66, 16.60,  5.65 }, { -1,  0.00,  0.00,  0.00 } } },
    { 1324.3474, { { 66, 61.41, 17.00, 36.70 }, { 67, 53.33, 17.10, 17.30 } } },
    { 1324.3587, { { 66, 60.83, 17.30, 61.93 }, { 67, 53.00, 17.30, 50.28 } } },
    { 1324.3699, { { 66, 60.66, 17.40, 65.81 }, { 67, 53.08, 17.50, 58.04 } } },
    { 1324.3811, { { 66, 60.66, 17.40, 67.75 }, { 67, 53.00, 17.60, 58.04 } } },
    { 1324.3923, { { 66, 60.58, 17.60, 67.75 }, { 67, 53.00, 17.70, 61.93 } } },
    { 1324.4036, { { 66, 60.50, 18.00, 71.63 }, { 67, 53.00, 18.20, 65.81 } } },
    { 1324.4152, { { 66, 60.58, 18.30, 73.57 }, { 67, 53.00, 18.50, 65.81 } } },
    { 1324.4265, { { 66, 60.58, 19.00, 71.63 }, { 67, 52.91, 19.20, 67.75 } } },
    { 1324.4377, { { 66, 60.58, 19.50, 71.63 }, { 67, 53.00, 19.70, 65.81 } } },
    { 1324.4487, { { 66, 60.66, 20.10, 69.69 }, { 67, 52.91, 20.40, 63.87 } } },
    { 1324.4596, { { 66, 60.58, 20.90, 69.69 }, { 67, 53.00, 21.10, 65.81 } } },
    { 1324.4705, { { 66, 60.58, 21.50, 71.63 }, { 67, 53.00, 21.80, 65.81 } } },
    { 1324.4815, { { 66, 60.50, 22.40, 73.57 }, { 67, 53.08, 22.60, 67.75 } } },
    { 1324.4926, { { 66, 60.75, 23.10, 77.45 }, { 67, 53.00, 23.30, 69.69 } } },
    { 1324.5039, { { 66, 60.75, 24.00, 77.45 }, { 67, 53.00, 24.10, 69.69 } } },
    { 1324.5152, { { 66, 60.66, 24.50, 77.45 }, { 67, 53.00, 24.70, 71.63 } } },
  };

  SplitCorrectingFilterInterpreterTestInterpreter* base_interpreter
      = new SplitCorrectingFilterInterpreterTestInterpreter;
  SplitCorrectingFilterInterpreter interpreter(NULL, base_interpreter, NULL);
  interpreter.Enable();

  HardwareProperties hwprops = {
    0, 0, 100, 100,  // left, top, right, bottom
    1,  // res_x
    1,  // res_y
    133,  // screen_x_dpi
    133,  // screen_y_dpi
    -1,  // orientation minimum
    2,   // orientation maximum
    5,  // max finger cnt
    5,  // max touch cnt
    0,  // supports_t5r2
    0,   // support_semi_mt
    1,  // is_button_pad
    0  // has_wheel
  };
  TestInterpreterWrapper wrapper(&interpreter, &hwprops);

  for (size_t i = 0; i < arraysize(inputs); i++) {
    const FalseMergeInputs& input = inputs[i];
    // Get finger count
    unsigned short finger_cnt = 0;
    for (size_t fidx = 0;
         fidx < arraysize(input.in) && input.in[fidx].id_ >= 0;
         fidx++)
      finger_cnt += 1;
    // Set up hardware state
    FingerState fs[finger_cnt];
    for (size_t fidx = 0; fidx < finger_cnt; fidx++) {
      memset(&fs[fidx], 0, sizeof(fs[fidx]));
      fs[fidx].position_x  = input.in[fidx].x_;
      fs[fidx].position_y  = input.in[fidx].y_;
      fs[fidx].pressure    = input.in[fidx].pressure_;
      fs[fidx].tracking_id = input.in[fidx].id_;
    }
    HardwareState hs = {
        input.timestamp, 0, finger_cnt, finger_cnt, &fs[0], 0, 0, 0, 0
    };
    // Set up expectations
    base_interpreter->expect_finger_ids_ = false;
    if (i == 1) {
      // if the second iteration, expect merging to occur
      base_interpreter->expect_finger_ids_ = true;
      base_interpreter->expected_ids_.clear();
      base_interpreter->expected_ids_.insert(inputs[0].in[0].id_);
    } else if (i == (arraysize(inputs) - 1)) {
      // if the final iteration, expect no merging
      base_interpreter->expect_finger_ids_ = true;
      base_interpreter->expected_ids_.clear();
      base_interpreter->expected_ids_.insert(inputs[i].in[0].id_);
      base_interpreter->expected_ids_.insert(inputs[i].in[1].id_);
    }
    // if the last iteration
    stime_t timestamp = -1.0;

    wrapper.SyncInterpret(&hs, &timestamp);
    base_interpreter->expect_warp_on_one_finger_only_ = true;
  }
}

namespace {
struct DistFromPointToLineRec {
  // line is (x0, y0) to (x1, y1). Point is (px, py). Expected dist is dist.
  float x0, y0, x1, y1, px, py, dist;
};

bool WeakFloatEq(float f_a, float f_b) {
  return fabsf(f_a - f_b) < 0.001;
}
}  // namespace

// This test sets up some example inputs to send through the
// DistFromPointToLine function. It also transforms these inputs in a variety
// of ways to further stress this function.
TEST(SplitCorrectingFilterInterpreterTest, DistFromPointToLineTest) {
  // Imagine a 3-4-5 triangle. theta is the angle between 4 and 5.
  const float theta = acosf(4.0 / 5.0);

  DistFromPointToLineRec tests[] = {
    { 0.0, 0.0, 1.0, 0.0, 1.0, 1.0, 1.0 },
    { 1.0, 1.0, 2.0, 1.0, 2.0, 2.0, 1.0 },
    { 1.0, 1.0, 2.0, 0.0, 2.0, 1.0, sqrtf(0.5) },
    { 1.0, 1.0, 2.0, 1.0, 4.0, 2.0, 1.0 },
    { 0.0, 0.0, static_cast<float>(4.0 * cosf(theta)),
      static_cast<float>(4.0 * sinf(theta)), 5.0, 0.0, 3.0 },
  };

  // For each of the input tests, these are a variety of transforms we apply.
  // There is a for loop for each type of transform. Because they are nested,
  // this means we test every combination.
  for (size_t i = 0; i < arraysize(tests); i++) {
    for (int h_shift = 1; h_shift <= 4; h_shift++) {
      float horiz_shift = static_cast<float>(h_shift) / 2.0;
      for (int v_shift = 1; v_shift <= 4; v_shift++) {
        float vert_shift = static_cast<float>(v_shift) / 2.0;
        for (size_t h_invert = 0; h_invert < 2; h_invert++) {
          float horiz_invert = h_invert ? 1.0 : -1.0;
          for (size_t v_invert = 0; v_invert < 2; v_invert++) {
            float vert_invert = v_invert ? 1.0 : -1.0;
            for (size_t swap_inputs = 0; swap_inputs < 2; swap_inputs++) {
              DistFromPointToLineRec rec = {
                (tests[i].x0 + horiz_shift) * horiz_invert,
                (tests[i].y0 + vert_shift) * vert_invert,
                (tests[i].x1 + horiz_shift) * horiz_invert,
                (tests[i].y1 + vert_shift) * vert_invert,
                (tests[i].px + horiz_shift) * horiz_invert,
                (tests[i].py + vert_shift) * vert_invert,
                tests[i].dist,
              };
              if (swap_inputs) {
                std::swap(rec.x0, rec.x1);
                std::swap(rec.y0, rec.y1);
              }
              float actual = sqrtf(
                  SplitCorrectingFilterInterpreter::DistSqFromPointToLine(
                      rec.x0, rec.y0,
                      rec.x1, rec.y1,
                      rec.px, rec.py));
              EXPECT_TRUE(WeakFloatEq(
                  rec.dist,
                  actual)) << "(" << rec.x0 << ", "
                           << rec.y0 << ") ("
                           << rec.x1 << ", "
                           << rec.y1 << ") and ("
                           << rec.px << ", "
                           << rec.py << "): got "
                           << actual << ", expected "
                           << rec.dist;
            }
          }
        }
      }
    }
  }
}

struct LumpyThumbSplitTestInputs {
  stime_t now;
  unsigned short buttons_down;
  float x0, y0, pressure0;
  short id0;
  float x1, y1, pressure1;
  short id1;
};

// Test that a thumbsplit on lumpy doesn't cause two fingers to appear
TEST(SplitCorrectingFilterInterpreterTest, LumpyThumbSplitTest) {
  SplitCorrectingFilterInterpreterTestInterpreter* base_interpreter
      = new SplitCorrectingFilterInterpreterTestInterpreter;
  SplitCorrectingFilterInterpreter interpreter(NULL, base_interpreter, NULL);
  interpreter.Enable();

  base_interpreter->expected_ids_.insert(2);
  base_interpreter->expect_finger_ids_ = true;

  HardwareProperties hwprops = {
    0.0,  // left edge
    0.0,  // top edge
    106.666672,  // right edge
    68.0,  // bottom edge
    1.0,  // x pixels/TP width
    1.0,  // y pixels/TP height
    25.4,  // x screen DPI
    25.4,  // y screen DPI
    -1,  // orientation minimum
    2,   // orientation maximum
    15,  // max fingers
    5,  // max touch
    0,  // t5r2
    0,  // semi-mt
    1,  // is button pad,
    0  // has_wheel
  };
  TestInterpreterWrapper wrapper(&interpreter, &hwprops);

  LumpyThumbSplitTestInputs inputs[] = {
    { 8.5812, 0, 52.66, 66.09,  79.39, 2,  0.00,  0.00,  0.00, 0 },
    { 8.5904, 0, 52.66, 66.09,  81.33, 2,  0.00,  0.00,  0.00, 0 },
    { 8.6087, 0, 52.66, 66.00,  81.33, 2,  0.00,  0.00,  0.00, 0 },
    { 8.6179, 0, 52.66, 65.80,  83.27, 2,  0.00,  0.00,  0.00, 0 },
    { 8.6365, 0, 52.58, 65.80,  85.21, 2,  0.00,  0.00,  0.00, 0 },
    { 8.6457, 0, 52.41, 65.59,  89.09, 2,  0.00,  0.00,  0.00, 0 },
    { 8.6550, 0, 52.25, 65.59,  91.03, 2,  0.00,  0.00,  0.00, 0 },
    { 8.6642, 0, 52.00, 65.59,  92.97, 2,  0.00,  0.00,  0.00, 0 },
    { 8.6735, 0, 51.91, 65.59,  96.85, 2,  0.00,  0.00,  0.00, 0 },
    { 8.6828, 0, 51.58, 65.40, 102.67, 2,  0.00,  0.00,  0.00, 0 },
    { 8.6921, 0, 51.33, 65.30, 104.61, 2,  0.00,  0.00,  0.00, 0 },
    { 8.7015, 0, 51.25, 65.30, 106.55, 2,  0.00,  0.00,  0.00, 0 },
    { 8.7110, 0, 51.00, 65.20, 110.43, 2,  0.00,  0.00,  0.00, 0 },
    { 8.7205, 0, 50.83, 65.20, 116.26, 2,  0.00,  0.00,  0.00, 0 },
    { 8.7298, 0, 50.41, 65.20, 118.20, 2,  0.00,  0.00,  0.00, 0 },
    { 8.7393, 0, 50.08, 65.00, 118.20, 2,  0.00,  0.00,  0.00, 0 },
    { 8.7487, 0, 49.66, 65.00, 118.20, 2,  0.00,  0.00,  0.00, 0 },
    { 8.7582, 0, 49.25, 64.90, 118.20, 2,  0.00,  0.00,  0.00, 0 },
    { 8.7678, 0, 48.83, 64.70, 118.20, 2,  0.00,  0.00,  0.00, 0 },
    { 8.7773, 0, 48.58, 64.50, 118.20, 2,  0.00,  0.00,  0.00, 0 },
    { 8.7869, 1, 48.33, 64.50, 118.20, 2,  0.00,  0.00,  0.00, 0 },
    { 8.8088, 1, 49.83, 64.59, 118.20, 2, 39.91, 60.60, 98.79, 4 },
    { 8.8213, 1, 45.91, 61.90, 191.93, 2,  0.00,  0.00,  0.00, 0 },
  };
  for (size_t i = 0; i < arraysize(inputs); i++) {
    const LumpyThumbSplitTestInputs& input = inputs[i];
    FingerState fs[2] = {
      { 0, 0, 0, 0, input.pressure0, 0, input.x0, input.y0, input.id0, 0 },
      { 0, 0, 0, 0, input.pressure1, 0, input.x1, input.y1, input.id1, 0 },
    };
    unsigned short finger_cnt = input.id1 ? 2 : 1;
    HardwareState hs = {
      input.now, input.buttons_down, finger_cnt, finger_cnt, fs, 0, 0, 0, 0
    };
    stime_t timeout = -1;
    wrapper.SyncInterpret(&hs, &timeout);
  }
  EXPECT_EQ(arraysize(inputs), base_interpreter->iteration_);
}

// Tests that close fingers that are scrolling don't get incorrectly merged.
TEST(SplitCorrectingFilterInterpreterTest, TwoFingerScrollTest) {
  InputEventWithExpectations events[] = {
    {{{ 0, 0, 0, 0, 7.600345, 0, 56.250000, 61.400002, 3134, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, -1 }},
    {{{ 0, 0, 0, 0, 17.302191, 0, 56.583336, 61.200001, 3134, 0 },
      { 0, 0, 0, 0, 5.659976, 0, 42.000000, 64.099998, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 23.123301, 0, 56.416668, 61.200001, 3134, 0 },
      { 0, 0, 0, 0, 15.361822, 0, 42.000000, 64.000000, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 28.944407, 0, 56.333336, 61.100002, 3134, 0 },
      { 0, 0, 0, 0, 27.004038, 0, 41.916668, 63.600002, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 30.884779, 0, 56.416668, 60.700001, 3134, 0 },
      { 0, 0, 0, 0, 34.765518, 0, 41.916668, 62.700001, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 34.765518, 0, 56.500000, 59.799999, 3134, 0 },
      { 0, 0, 0, 0, 38.646255, 0, 42.000000, 61.600002, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 32.825150, 0, 56.750000, 58.799999, 3134, 0 },
      { 0, 0, 0, 0, 40.586628, 0, 42.166668, 60, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 28.944407, 0, 56.916668, 57.400002, 3134, 0 },
      { 0, 0, 0, 0, 40.586628, 0, 42.500000, 57.700001, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 34.765518, 0, 57.333336, 55.200001, 3134, 0 },
      { 0, 0, 0, 0, 36.705887, 0, 43.083336, 54.299999, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 32.825150, 0, 57.916668, 52.100002, 3134, 0 },
      { 0, 0, 0, 0, 34.765518, 0, 43.666668, 49.900002, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 30.884779, 0, 58.666668, 47.900002, 3134, 0 },
      { 0, 0, 0, 0, 34.765518, 0, 44.250000, 44.500000, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 23.123301, 0, 59.666668, 42.799999, 3134, 0 },
      { 0, 0, 0, 0, 44.467365, 0, 45.583336, 38.400002, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 28.944407, 0, 61.500000, 34.299999, 3134, 0 },
      { 0, 0, 0, 0, 27.004038, 0, 46.833336, 30.900000, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 28.944407, 0, 63.166668, 28.000000, 3134, 0 },
      { 0, 0, 0, 0, 32.825150, 0, 48.666668, 23.700001, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 28.944407, 0, 65.666672, 20.400000, 3134, 0 },
      { 0, 0, 0, 0, 34.765518, 0, 50.083336, 18.400000, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 21.182930, 0, 67.500000, 15.200000, 3134, 0 },
      { 0, 0, 0, 0, 17.302191, 0, 52.083336, 12.900001, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3134, 3135, -1 }},
    {{{ 0, 0, 0, 0, 1.779237, 0, 55.500000, 4.000000, 3135, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3135, -1 }}
  };

  DoTest(events, arraysize(events), false);
}

// Tests that a thumb-edge gets correctly merged to a single contact
TEST(SplitCorrectingFilterInterpreterTest, ThumbDragTest) {
  InputEventWithExpectations events[] = {
    {{{ 0, 0, 0, 0, 9.540714, 0, 96.416672, 20.300001, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 40.586628, 0, 96.250000, 20.600000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 65.811432, 0, 96.083336, 20.800001, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 81.334381, 0, 96.000000, 21.000000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 91.036232, 0, 95.916672, 21.100000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 98.797707, 0, 95.916672, 21.100000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 104.618820, 0, 95.916672, 21.100000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 79.394012, 0, 95.916672, 21.700001, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 83.274757, 0, 95.916672, 22.100000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 85.215126, 0, 95.916672, 22.300001, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 89.095863, 0, 95.833336, 22.500000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 92.976601, 0, 95.666672, 22.700001, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 94.916969, 0, 95.583336, 22.900000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 98.797707, 0, 95.416672, 23.000000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 100.738075, 0, 95.166672, 23.100000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 106.559189, 0, 94.916672, 23.200001, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 110.439926, 0, 94.666672, 23.400000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 112.380295, 0, 94.333336, 23.600000, 3087, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 114.320663, 0, 93.750000, 26.600000, 3087, 0 },
      { 0, 0, 0, 0, 102.678452, 0, 92.833336, 17.100000, 3088, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 116.261032, 0, 93.583336, 26.600000, 3087, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 92.666672, 17.200001, 3088, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 96.857338, 0, 92.333336, 20.700001, 3088, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 96.857338, 0, 92.083336, 20.700001, 3088, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 98.797707, 0, 91.750000, 20.700001, 3088, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 91.416672, 21.500000, 3088, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 90.916672, 22.000000, 3088, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 116.261032, 0, 90.416672, 22.800001, 3088, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 89.916672, 23.400000, 3088, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 178.352859, 0, 89.416672, 23.700001, 3088, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 178.352859, 0, 88.916672, 23.900000, 3088, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 87.583336, 27.200001, 3088, 0 },
      { 0, 0, 0, 0, 91.036232, 0, 86.166672, 18.000000, 3089, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 87.333336, 27.300001, 3088, 0 },
      { 0, 0, 0, 0, 87.155495, 0, 85.916672, 18.100000, 3089, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 85.215126, 0, 85.583336, 21.600000, 3089, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 85.416672, 22.700001, 3089, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 85.083336, 23.400000, 3089, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 84.583336, 23.900000, 3089, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 84.000000, 24.700001, 3089, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 180.293228, 0, 83.500000, 25.200001, 3089, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 182.233597, 0, 82.916672, 25.500000, 3089, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 180.293228, 0, 82.333336, 25.800001, 3089, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 106.559189, 0, 80.916672, 32.200001, 3089, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.916672, 22.500000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 110.439926, 0, 80.583336, 32.299999, 3089, 0 },
      { 0, 0, 0, 0, 116.261032, 0, 78.500000, 22.700001, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 114.320663, 0, 77.833336, 26.100000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 77.583336, 26.800001, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 114.320663, 0, 77.250000, 27.900000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 76.750000, 28.700001, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 186.114334, 0, 76.250000, 29.100000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 75.000000, 32.900002, 3090, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 72.666672, 23.900000, 3091, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 74.666672, 33.000000, 3090, 0 },
      { 0, 0, 0, 0, 96.857338, 0, 72.250000, 24.100000, 3091, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 72.333336, 29.900000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 71.916672, 29.900000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 71.416672, 30.100000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 70.833336, 30.600000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 184.173965, 0, 70.333336, 30.900000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 188.054703, 0, 69.750000, 31.100000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 188.054703, 0, 69.166672, 31.300001, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 65.583336, 28.500000, 3090, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 68.416672, 37.700001, 3092, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 65.166672, 28.600000, 3090, 0 },
      { 0, 0, 0, 0, 108.499557, 0, 68.083336, 37.799999, 3092, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 64.666672, 31.900000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 64.416672, 32.400002, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 63.916668, 32.900002, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 63.500000, 33.799999, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 62.833336, 34.400002, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 61.833336, 38.600002, 3090, 0 },
      { 0, 0, 0, 0, 110.439926, 0, 58.666668, 30, 3093, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 61.416668, 38.700001, 3090, 0 },
      { 0, 0, 0, 0, 106.559189, 0, 58.250000, 30.200001, 3093, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 58.333336, 35.799999, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 57.916668, 35.900002, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 57.333336, 36.200001, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 188.054703, 0, 56.750000, 36.500000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 191.935440, 0, 56.083336, 36.700001, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 193.875809, 0, 55.416668, 37.000000, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 51.000000, 35.000000, 3090, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 54.750000, 43.600002, 3094, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 51.166668, 38.200001, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 50.916668, 38.700001, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 50.416668, 39.200001, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 49.750000, 39.600002, 3090, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 112.380295, 0, 45.500000, 36.400002, 3090, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 49.500000, 44.500000, 3095, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 110.439926, 0, 45.166668, 36.600002, 3090, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 49.166668, 44.600002, 3095, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 46.083336, 42.200001, 3095, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 45.666668, 42.200001, 3095, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 45.083336, 42.299999, 3095, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 44.416668, 42.500000, 3095, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 43.750000, 42.900002, 3095, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 188.054703, 0, 43.166668, 43.200001, 3095, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 184.173965, 0, 42.500000, 43.500000, 3095, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 176.412491, 0, 41.833336, 43.900002, 3095, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 108.499557, 0, 41.666668, 49.900002, 3095, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 37.250000, 42.000000, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 112.380295, 0, 41.333336, 50, 3095, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 36.916668, 42.200001, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 37.416668, 46.400002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 37.083336, 46.600002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 36.583336, 46.900002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 36.000000, 47.299999, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 176.412491, 0, 35.416668, 47.500000, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 186.114334, 0, 34.916668, 47.799999, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 186.114334, 0, 34.333336, 48.100002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 184.173965, 0, 33.666668, 48.600002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 184.173965, 0, 33.000000, 49.100002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 186.114334, 0, 32.333336, 49.600002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 184.173965, 0, 31.666668, 50.200001, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 184.173965, 0, 31.083334, 50.799999, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 182.233597, 0, 30.500000, 51.299999, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 184.173965, 0, 29.916668, 51.799999, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 186.114334, 0, 29.333334, 52.299999, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 178.352859, 0, 28.750000, 52.600002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 178.352859, 0, 28.166668, 52.900002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 178.352859, 0, 27.666668, 53.200001, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 27.166668, 53.700001, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.666668, 54.200001, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.250000, 54.700001, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.916668, 55.000000, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.666668, 55.200001, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.416668, 55.400002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.250000, 55.500000, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.166668, 55.600002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.083334, 55.600002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.083334, 55.700001, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.166668, 55.700001, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 102.678452, 0, 25.166668, 55.799999, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 63.871063, 0, 25.250000, 56.000000, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
    {{{ 0, 0, 0, 0, 27.004038, 0, 25.333334, 56.100002, 3096, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3087, -1 }},
  };

  DoTest(events, arraysize(events), false);
}

// Two thumb-edges as the same time
TEST(SplitCorrectingFilterInterpreterTest, DualThumbsTest) {
  InputEventWithExpectations events[] = {
    {{{ 0, 0, 0, 0, 36.705887, 0, 22.916668, 7.300000, 3111, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, -1 }},
    {{{ 0, 0, 0, 0, 52.228844, 0, 22.916668, 7.200000, 3111, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, -1 }},
    {{{ 0, 0, 0, 0, 65.811432, 0, 22.916668, 7.100000, 3111, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, -1 }},
    {{{ 0, 0, 0, 0, 73.572906, 0, 22.916668, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 5.659976, 0, 76.333336, 3.100000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 77.453644, 0, 22.916668, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 40.586628, 0, 76.416672, 2.000000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 81.334381, 0, 22.916668, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 63.871063, 0, 76.500000, 1.400000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 83.274757, 0, 22.916668, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 79.394012, 0, 76.583336, 1.100000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 85.215126, 0, 22.916668, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 91.036232, 0, 76.666672, 0.900000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 87.155495, 0, 22.916668, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 94.916969, 0, 76.666672, 0.800000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 89.095863, 0, 22.916668, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 76.666672, 0.800000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 89.095863, 0, 22.916668, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 102.678452, 0, 76.666672, 0.800000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 23.666668, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 76.666672, 0.900000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.166668, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 110.439926, 0, 76.666672, 1.100000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.500000, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 131.783997, 0, 77.000000, 2.000000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.666668, 6.800000, 3111, 0 },
      { 0, 0, 0, 0, 133.724365, 0, 77.333336, 3.000000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.833334, 6.900000, 3111, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 77.666672, 3.900000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.250000, 7.000000, 3111, 0 },
      { 0, 0, 0, 0, 102.678452, 0, 81.416672, 6.900000, 3112, 0 },
      { 0, 0, 0, 0, 77.453644, 0, 73.500000, 3.200000, 3113, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.250000, 7.200000, 3111, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 81.416672, 7.100000, 3112, 0 },
      { 0, 0, 0, 0, 19.242561, 0, 72.833336, 3.200000, 3113, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.500000, 7.600000, 3111, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 78.500000, 6.900000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.500000, 7.800000, 3111, 0 },
      { 0, 0, 0, 0, 108.499557, 0, 78.500000, 7.100000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.500000, 8.100000, 3111, 0 },
      { 0, 0, 0, 0, 110.439926, 0, 78.500000, 7.400000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.583334, 8.400001, 3111, 0 },
      { 0, 0, 0, 0, 145.366577, 0, 78.333336, 7.700000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.666668, 8.900001, 3111, 0 },
      { 0, 0, 0, 0, 157.008789, 0, 78.250000, 8.100000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.833334, 9.500000, 3111, 0 },
      { 0, 0, 0, 0, 157.008789, 0, 78.250000, 8.600000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.000000, 10.100000, 3111, 0 },
      { 0, 0, 0, 0, 157.008789, 0, 78.250000, 9.100000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 174.472122, 0, 26.083334, 10.600000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.166672, 9.900001, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.166668, 11.400001, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.166672, 10.700000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.416668, 13.100000, 3111, 0 },
      { 0, 0, 0, 0, 116.261032, 0, 81.750000, 13.200000, 3112, 0 },
      { 0, 0, 0, 0, 91.036232, 0, 73.166672, 11.500000, 3114, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.416668, 13.400001, 3111, 0 },
      { 0, 0, 0, 0, 116.261032, 0, 81.750000, 13.500000, 3112, 0 },
      { 0, 0, 0, 0, 91.036232, 0, 73.083336, 11.800000, 3114, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.416668, 14.200000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.833336, 13.900001, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.416668, 14.500000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.833336, 14.200000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.416668, 14.900001, 3111, 0 },
      { 0, 0, 0, 0, 172.531754, 0, 78.750000, 14.500000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.416668, 15.400001, 3111, 0 },
      { 0, 0, 0, 0, 170.591385, 0, 78.666672, 14.800000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.666668, 15.900001, 3111, 0 },
      { 0, 0, 0, 0, 166.710632, 0, 78.666672, 15.200000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.750000, 16.400000, 3111, 0 },
      { 0, 0, 0, 0, 166.710632, 0, 78.583336, 15.600000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.833334, 16.900000, 3111, 0 },
      { 0, 0, 0, 0, 168.651001, 0, 78.583336, 16.100000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.916668, 17.400000, 3111, 0 },
      { 0, 0, 0, 0, 172.531754, 0, 78.583336, 16.600000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.916668, 18.700001, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.916672, 19.200001, 3112, 0 },
      { 0, 0, 0, 0, 96.857338, 0, 73.000000, 17.200001, 3115, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.916668, 18.900000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.916672, 19.400000, 3112, 0 },
      { 0, 0, 0, 0, 96.857338, 0, 72.916672, 17.500000, 3115, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.916668, 19.300001, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.916672, 19.700001, 3112, 0 },
      { 0, 0, 0, 0, 98.797707, 0, 72.833336, 17.800001, 3115, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.916668, 20.500000, 3111, 0 },
      { 0, 0, 0, 0, 164.770264, 0, 79.833336, 20.100000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.916668, 20.700001, 3111, 0 },
      { 0, 0, 0, 0, 164.770264, 0, 79.416672, 20.100000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.916668, 21.000000, 3111, 0 },
      { 0, 0, 0, 0, 162.829895, 0, 78.916672, 20.200001, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.916668, 21.400000, 3111, 0 },
      { 0, 0, 0, 0, 170.591385, 0, 78.583336, 20.500000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.916668, 21.900000, 3111, 0 },
      { 0, 0, 0, 0, 172.531754, 0, 78.416672, 20.900000, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 27.000000, 22.500000, 3111, 0 },
      { 0, 0, 0, 0, 174.472122, 0, 78.250000, 21.300001, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 27.083334, 23.000000, 3111, 0 },
      { 0, 0, 0, 0, 176.412491, 0, 78.166672, 21.800001, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 27.166668, 23.500000, 3111, 0 },
      { 0, 0, 0, 0, 176.412491, 0, 78.166672, 22.300001, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 27.083334, 24.000000, 3111, 0 },
      { 0, 0, 0, 0, 176.412491, 0, 78.166672, 22.700001, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 27.000000, 24.500000, 3111, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 77.500000, 23.200001, 3112, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 27.000000, 25.700001, 3111, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 72.750000, 23.700001, 3112, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.833336, 25.400000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.916668, 25.900000, 3111, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 72.750000, 23.900000, 3112, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.833336, 25.600000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.833334, 26.200001, 3111, 0 },
      { 0, 0, 0, 0, 102.678452, 0, 72.666672, 24.100000, 3112, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.833336, 25.900000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.583334, 27.300001, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 79.000000, 26.200001, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.500000, 27.500000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.916672, 26.400000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.500000, 27.800001, 3111, 0 },
      { 0, 0, 0, 0, 168.651001, 0, 78.583336, 26.500000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.500000, 28.200001, 3111, 0 },
      { 0, 0, 0, 0, 168.651001, 0, 78.250000, 26.600000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.583334, 28.700001, 3111, 0 },
      { 0, 0, 0, 0, 176.412491, 0, 78.083336, 26.800001, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.583334, 29.100000, 3111, 0 },
      { 0, 0, 0, 0, 176.412491, 0, 78.083336, 27.100000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.583334, 29.500000, 3111, 0 },
      { 0, 0, 0, 0, 176.412491, 0, 78.083336, 27.400000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.583334, 29.900000, 3111, 0 },
      { 0, 0, 0, 0, 176.412491, 0, 78.083336, 27.800001, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.500000, 30.400000, 3111, 0 },
      { 0, 0, 0, 0, 176.412491, 0, 78.083336, 28.200001, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.416668, 30.800001, 3111, 0 },
      { 0, 0, 0, 0, 178.352859, 0, 78.083336, 28.500000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.250000, 31.300001, 3111, 0 },
      { 0, 0, 0, 0, 174.472122, 0, 78.083336, 28.900000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 26.166668, 31.800001, 3111, 0 },
      { 0, 0, 0, 0, 178.352859, 0, 78.083336, 29.300001, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.916668, 33.100002, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.833336, 31.600000, 3116, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 72.750000, 29.800001, 3117, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.833334, 33.299999, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.833336, 31.800001, 3116, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 72.666672, 30, 3117, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.750000, 34.100002, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.833336, 32.000000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.666668, 34.299999, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.833336, 32.200001, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.750000, 34.500000, 3111, 0 },
      { 0, 0, 0, 0, 170.591385, 0, 78.583336, 32.299999, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.750000, 34.799999, 3111, 0 },
      { 0, 0, 0, 0, 172.531754, 0, 78.250000, 32.500000, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.750000, 35.200001, 3111, 0 },
      { 0, 0, 0, 0, 178.352859, 0, 78.083336, 32.799999, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.750000, 35.600002, 3111, 0 },
      { 0, 0, 0, 0, 178.352859, 0, 78.000000, 33.200001, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 182.233597, 0, 25.750000, 35.900002, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.000000, 33.799999, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 182.233597, 0, 25.750000, 36.200001, 3111, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 77.416672, 34.299999, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 98.797707, 0, 24.750000, 36.799999, 3111, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 77.000000, 34.799999, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 98.797707, 0, 24.083334, 37.299999, 3111, 0 },
      { 0, 0, 0, 0, 102.678452, 0, 76.750000, 35.299999, 3116, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.333334, 38.299999, 3111, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 72.666672, 36.100002, 3116, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.833336, 37.700001, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.333334, 38.500000, 3111, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 72.583336, 36.299999, 3116, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.833336, 37.900002, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.166668, 39.200001, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.833336, 38.100002, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.083334, 39.400002, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.833336, 38.299999, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.000000, 39.700001, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.833336, 38.600002, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.000000, 40, 3111, 0 },
      { 0, 0, 0, 0, 170.591385, 0, 78.583336, 38.799999, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.000000, 40.400002, 3111, 0 },
      { 0, 0, 0, 0, 178.352859, 0, 78.416672, 39.100002, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.000000, 40.799999, 3111, 0 },
      { 0, 0, 0, 0, 178.352859, 0, 78.250000, 39.500000, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.166668, 41.200001, 3111, 0 },
      { 0, 0, 0, 0, 180.293228, 0, 78.166672, 39.900002, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.250000, 41.600002, 3111, 0 },
      { 0, 0, 0, 0, 178.352859, 0, 78.166672, 40.400002, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.416668, 41.900002, 3111, 0 },
      { 0, 0, 0, 0, 180.293228, 0, 78.166672, 40.799999, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 186.114334, 0, 25.416668, 42.200001, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.083336, 41.400002, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 186.114334, 0, 25.333334, 42.500000, 3111, 0 },
      { 0, 0, 0, 0, 102.678452, 0, 77.416672, 41.700001, 3118, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 102.678452, 0, 22.916668, 43.799999, 3111, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 72.666672, 42.299999, 3118, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.916672, 43.799999, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 104.618820, 0, 22.833334, 44.000000, 3111, 0 },
      { 0, 0, 0, 0, 102.678452, 0, 72.666672, 42.500000, 3118, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.916672, 44.000000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 104.618820, 0, 22.750000, 44.299999, 3111, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 72.666672, 42.700001, 3118, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 81.916672, 44.200001, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.583334, 45.100002, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 79.000000, 44.500000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.583334, 45.400002, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 79.000000, 44.700001, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.833334, 45.700001, 3111, 0 },
      { 0, 0, 0, 0, 184.173965, 0, 78.833336, 44.900002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.000000, 46.100002, 3111, 0 },
      { 0, 0, 0, 0, 170.591385, 0, 78.583336, 45.000000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.083334, 46.500000, 3111, 0 },
      { 0, 0, 0, 0, 166.710632, 0, 78.333336, 45.200001, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.166668, 47.000000, 3111, 0 },
      { 0, 0, 0, 0, 182.233597, 0, 78.166672, 45.500000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.250000, 47.400002, 3111, 0 },
      { 0, 0, 0, 0, 184.173965, 0, 78.166672, 45.799999, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.250000, 47.700001, 3111, 0 },
      { 0, 0, 0, 0, 184.173965, 0, 78.166672, 46.200001, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.250000, 48.100002, 3111, 0 },
      { 0, 0, 0, 0, 180.293228, 0, 78.166672, 46.600002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 186.114334, 0, 25.166668, 48.400002, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.166672, 47.000000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 186.114334, 0, 25.083334, 48.700001, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.166672, 47.400002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 102.678452, 0, 24.166668, 49.200001, 3111, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 77.500000, 47.700001, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 102.678452, 0, 23.583334, 49.700001, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 77.666672, 48.100002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 104.618820, 0, 22.666668, 50.600002, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 82.000000, 49.900002, 3119, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 72.666672, 48.400002, 3120, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 23.333334, 50.799999, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 82.000000, 50.100002, 3119, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 72.666672, 48.600002, 3120, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.500000, 51.200001, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 79.166672, 50.200001, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.416668, 51.400002, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 79.166672, 50.299999, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.416668, 51.600002, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 79.166672, 50.500000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.416668, 51.799999, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 79.166672, 50.700001, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.333334, 52.100002, 3111, 0 },
      { 0, 0, 0, 0, 184.173965, 0, 79.000000, 50.900002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.250000, 52.400002, 3111, 0 },
      { 0, 0, 0, 0, 184.173965, 0, 78.833336, 51.100002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.250000, 52.700001, 3111, 0 },
      { 0, 0, 0, 0, 164.770264, 0, 78.583336, 51.299999, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.250000, 53.100002, 3111, 0 },
      { 0, 0, 0, 0, 164.770264, 0, 78.333336, 51.500000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.583334, 53.400002, 3111, 0 },
      { 0, 0, 0, 0, 178.352859, 0, 78.250000, 51.900002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.833334, 53.700001, 3111, 0 },
      { 0, 0, 0, 0, 178.352859, 0, 78.250000, 52.299999, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.000000, 54.000000, 3111, 0 },
      { 0, 0, 0, 0, 178.352859, 0, 78.250000, 52.799999, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 25.083334, 54.400002, 3111, 0 },
      { 0, 0, 0, 0, 180.293228, 0, 78.250000, 53.299999, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 186.114334, 0, 25.000000, 54.700001, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.166672, 53.900002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 178.352859, 0, 24.916668, 55.000000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.416672, 54.500000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 102.678452, 0, 24.000000, 55.500000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.583336, 55.100002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 102.678452, 0, 23.416668, 56.000000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.750000, 55.799999, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 104.618820, 0, 23.083334, 56.500000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.833336, 56.400002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 102.678452, 0, 22.833334, 57.000000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.916672, 57.100002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 102.678452, 0, 22.666668, 57.500000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 79.000000, 57.900002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 23.166668, 58.000000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.833336, 58.799999, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 23.500000, 58.500000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.666672, 59.600002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 23.750000, 59.000000, 3111, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 78.500000, 60.400002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 23.833334, 59.400002, 3111, 0 },
      { 0, 0, 0, 0, 168.651001, 0, 78.333336, 60.900002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.250000, 59.700001, 3111, 0 },
      { 0, 0, 0, 0, 162.829895, 0, 78.250000, 61.299999, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.583334, 60.100002, 3111, 0 },
      { 0, 0, 0, 0, 157.008789, 0, 78.166672, 61.700001, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.750000, 60.600002, 3111, 0 },
      { 0, 0, 0, 0, 151.187683, 0, 78.083336, 62.000000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.916668, 61.100002, 3111, 0 },
      { 0, 0, 0, 0, 141.485840, 0, 78.000000, 62.299999, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 172.531754, 0, 25.000000, 61.400002, 3111, 0 },
      { 0, 0, 0, 0, 91.036232, 0, 78.666672, 64.000000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.666668, 62.799999, 3111, 0 },
      { 0, 0, 0, 0, 106.559189, 0, 78.250000, 65.200005, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.500000, 63.799999, 3111, 0 },
      { 0, 0, 0, 0, 104.618820, 0, 78.000000, 66.000000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.333334, 64.500000, 3111, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 77.833336, 66.500000, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.250000, 65.000000, 3111, 0 },
      { 0, 0, 0, 0, 94.916969, 0, 77.666672, 66.900002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 24.166668, 65.500000, 3111, 0 },
      { 0, 0, 0, 0, 69.692169, 0, 77.666672, 67.200005, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 114.320663, 0, 24.083334, 65.800003, 3111, 0 },
      { 0, 0, 0, 0, 38.646255, 0, 77.750000, 67.400002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 94.916969, 0, 24.000000, 65.800003, 3111, 0 },
      { 0, 0, 0, 0, 15.361822, 0, 77.833336, 67.400002, 3119, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, 3112, -1 }},
    {{{ 0, 0, 0, 0, 58.049950, 0, 23.916668, 65.599998, 3111, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, -1 }},
    {{{ 0, 0, 0, 0, 30.884779, 0, 23.833334, 65.500000, 3111, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3111, -1 }},
  };

  DoTest(events, arraysize(events), false);
}

// Thumb edge dragging, and mid-way through a normal finger is added
TEST(SplitCorrectingFilterInterpreterTest, ThumbEdgePlusFingerTest) {
  InputEventWithExpectations events[] = {
    {{{ 0, 0, 0, 0, 11.481083, 0, 102.500000, 10.500000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 42.526997, 0, 103.083336, 10.800000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 59.990318, 0, 104.333336, 11.700000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 75.513275, 0, 105.000000, 12.500000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 89.095863, 0, 105.000000, 13.200000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 91.036232, 0, 103.416672, 13.000000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 94.916969, 0, 102.166672, 13.000000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 94.916969, 0, 101.000000, 13.100000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 94.916969, 0, 100, 13.300000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 96.857338, 0, 99.083336, 13.500000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 98.333336, 14.300000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 97.583336, 14.900001, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 102.678452, 0, 96.750000, 15.800000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 108.499557, 0, 95.166672, 20.200001, 3351, 0 },
      { 0, 0, 0, 0, 87.155495, 0, 92.250000, 12.200000, 3352, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 92.583336, 17.600000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 92.000000, 17.800001, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 91.166672, 18.300001, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 160.889526, 0, 90.416672, 18.700001, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 110.439926, 0, 85.750000, 16.500000, 3351, 0 },
      { 0, 0, 0, 0, 91.036232, 0, 89.250000, 25.500000, 3353, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 108.499557, 0, 85.250000, 16.700001, 3351, 0 },
      { 0, 0, 0, 0, 98.797707, 0, 88.750000, 25.700001, 3353, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 104.618820, 0, 84.666672, 20.300001, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 84.416672, 21.100000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 83.916672, 21.700001, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 83.333336, 22.600000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 82.416672, 26.700001, 3351, 0 },
      { 0, 0, 0, 0, 91.036232, 0, 78.583336, 18.400000, 3354, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 80.083336, 25.100000, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 77.583336, 22.200001, 3351, 0 },
      { 0, 0, 0, 0, 87.155495, 0, 81.500000, 31.200001, 3355, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 77.666672, 25.300001, 3351, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 77.083336, 26.600000, 3351, 0 },
      { 0, 0, 0, 0, 9.540714, 0, 36.500000, 19.300001, 3356, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 76.750000, 26.800001, 3351, 0 },
      { 0, 0, 0, 0, 21.182930, 0, 36.333336, 19.300001, 3356, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 76.166672, 27.100000, 3351, 0 },
      { 0, 0, 0, 0, 32.825150, 0, 36.083336, 19.300001, 3356, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 110.439926, 0, 76.250000, 32.299999, 3351, 0 },
      { 0, 0, 0, 0, 38.646255, 0, 35.583336, 19.400000, 3356, 0 },
      { 0, 0, 0, 0, 102.678452, 0, 71.916672, 23.900000, 3357, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 112.380295, 0, 75.833336, 32.500000, 3351, 0 },
      { 0, 0, 0, 0, 44.467365, 0, 35.416668, 19.500000, 3356, 0 },
      { 0, 0, 0, 0, 98.797707, 0, 71.500000, 24.100000, 3357, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 72.166672, 29.800001, 3351, 0 },
      { 0, 0, 0, 0, 48.348103, 0, 35.000000, 19.700001, 3356, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 71.750000, 30.100000, 3351, 0 },
      { 0, 0, 0, 0, 52.228844, 0, 34.750000, 19.800001, 3356, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 71.083336, 30.700001, 3351, 0 },
      { 0, 0, 0, 0, 54.169212, 0, 34.500000, 20, 3356, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 118.201401, 0, 70.166672, 31.300001, 3351, 0 },
      { 0, 0, 0, 0, 56.109581, 0, 34.250000, 20.200001, 3356, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 112.380295, 0, 69.083336, 31.700001, 3351, 0 },
      { 0, 0, 0, 0, 58.049950, 0, 33.916668, 20.500000, 3356, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 108.499557, 0, 64.333336, 29.900000, 3351, 0 },
      { 0, 0, 0, 0, 58.049950, 0, 32.666668, 22.000000, 3356, 0 },
      { 0, 0, 0, 0, 106.559189, 0, 69.250000, 38.000000, 3358, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 58.049950, 0, 32.250000, 22.400000, 3356, 0 },
      { 0, 0, 0, 0, 108.499557, 0, 66.166672, 35.299999, 3358, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 58.049950, 0, 32.000000, 22.600000, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 65.750000, 35.299999, 3358, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 59.990318, 0, 31.750000, 22.700001, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 65.166672, 35.600002, 3358, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 59.990318, 0, 31.500000, 22.900000, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 64.500000, 36.100002, 3358, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 61.930687, 0, 31.250000, 23.200001, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 63.750000, 36.700001, 3358, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 59.990318, 0, 31.000000, 23.400000, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 63.000000, 37.400002, 3358, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 59.990318, 0, 30.750000, 23.700001, 3356, 0 },
      { 0, 0, 0, 0, 112.380295, 0, 62.083336, 37.900002, 3358, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 59.990318, 0, 30, 24.200001, 3356, 0 },
      { 0, 0, 0, 0, 114.320663, 0, 57.500000, 36.299999, 3358, 0 },
      { 0, 0, 0, 0, 106.559189, 0, 63.000000, 43.799999, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 61.930687, 0, 29.583334, 24.800001, 3356, 0 },
      { 0, 0, 0, 0, 112.380295, 0, 59.750000, 41.000000, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 61.930687, 0, 29.416668, 24.900000, 3356, 0 },
      { 0, 0, 0, 0, 114.320663, 0, 59.500000, 41.200001, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 59.990318, 0, 29.250000, 25.200001, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 59.000000, 41.500000, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 58.049950, 0, 29.166668, 25.700001, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 58.416668, 41.799999, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 56.109581, 0, 29.250000, 26.500000, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 57.750000, 42.299999, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 56.109581, 0, 28.666668, 26.800001, 3356, 0 },
      { 0, 0, 0, 0, 112.380295, 0, 55.833336, 41.900002, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 56.109581, 0, 28.250000, 27.300001, 3356, 0 },
      { 0, 0, 0, 0, 114.320663, 0, 54.416668, 41.900002, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 58.049950, 0, 27.916668, 27.800001, 3356, 0 },
      { 0, 0, 0, 0, 116.261032, 0, 53.250000, 42.100002, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 58.049950, 0, 29.166668, 29.300001, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 53.500000, 43.500000, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 58.049950, 0, 29.916668, 30.400000, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 53.583336, 44.600002, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 58.049950, 0, 30.416668, 31.200001, 3356, 0 },
      { 0, 0, 0, 0, 116.261032, 0, 53.500000, 45.500000, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 58.049950, 0, 29.833334, 31.400000, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 53.250000, 46.299999, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 56.109581, 0, 29.000000, 31.500000, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 52.916668, 47.100002, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 52.228844, 0, 28.166668, 31.600000, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 52.500000, 48.000000, 3359, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 48.348103, 0, 25.750000, 32.700001, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 54.333336, 52.100002, 3359, 0 },
      { 0, 0, 0, 0, 106.559189, 0, 46.083336, 47.400002, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 50.288475, 0, 25.416668, 33.600002, 3356, 0 },
      { 0, 0, 0, 0, 108.499557, 0, 45.666668, 48.100002, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 50.288475, 0, 26.166668, 34.799999, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 47.166668, 49.400002, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 50.288475, 0, 26.583334, 35.700001, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 48.000000, 50.400002, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 54.169212, 0, 26.833334, 36.299999, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 48.333336, 51.200001, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 54.169212, 0, 26.833334, 36.600002, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 48.416668, 52.000000, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 54.169212, 0, 26.000000, 36.200001, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 47.000000, 52.299999, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 54.169212, 0, 25.500000, 36.100002, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 46.000000, 52.799999, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 54.169212, 0, 25.250000, 36.400002, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 46.333336, 54.200001, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 54.169212, 0, 24.916668, 36.600002, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 46.500000, 55.400002, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 52.228844, 0, 25.916668, 38.700001, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 46.500000, 56.400002, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 52.228844, 0, 26.333334, 40, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 46.333336, 57.200001, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 52.228844, 0, 25.083334, 39.299999, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 45.416668, 57.600002, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 52.228844, 0, 24.083334, 39.200001, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 44.833336, 58.100002, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 50.288475, 0, 23.250000, 39.400002, 3356, 0 },
      { 0, 0, 0, 0, 118.201401, 0, 44.416668, 58.799999, 3360, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 50.288475, 0, 21.583334, 40.100002, 3356, 0 },
      { 0, 0, 0, 0, 106.559189, 0, 38.333336, 60.200001, 3360, 0 },
      { 0, 0, 0, 0, 94.916969, 0, 48.583336, 64.200005, 3361, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 48.348103, 0, 21.750000, 41.700001, 3356, 0 },
      { 0, 0, 0, 0, 83.274757, 0, 45.000000, 65.300003, 3361, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 44.467365, 0, 21.416668, 41.799999, 3356, 0 },
      { 0, 0, 0, 0, 110.439926, 0, 43.833336, 65.599998, 3361, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 42.526997, 0, 20.916668, 42.000000, 3356, 0 },
      { 0, 0, 0, 0, 100.738075, 0, 41.916668, 65.900002, 3361, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 42.526997, 0, 20.500000, 42.200001, 3356, 0 },
      { 0, 0, 0, 0, 98.797707, 0, 40.666668, 66.300003, 3361, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 40.586628, 0, 20.166668, 42.400002, 3356, 0 },
      { 0, 0, 0, 0, 116.261032, 0, 39.916668, 66.099998, 3361, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 27.004038, 0, 19.916668, 42.600002, 3356, 0 },
      { 0, 0, 0, 0, 106.559189, 0, 39.416668, 65.800003, 3361, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 13.421453, 0, 19.750000, 42.700001, 3356, 0 },
      { 0, 0, 0, 0, 79.394012, 0, 39.083336, 65.500000, 3361, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, 3356, -1 }},
    {{{ 0, 0, 0, 0, 32.825150, 0, 37.833336, 67.599998, 3361, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }},
    {{{ 0, 0, 0, 0, 11.481083, 0, 37.833336, 67.700005, 3361, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 },
      { 0, 0, 0, 0, 0, 0, 0, 0, -1, 0 }},
     { 3351, -1 }}
  };

  DoTest(events, arraysize(events), false);
}

}  // namespace gestures
