// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "content/browser/media/url_provision_fetcher.h"

#include "content/public/browser/provision_fetcher_factory.h"
#include "media/base/bind_to_current_loop.h"
#include "net/base/load_flags.h"
#include "net/traffic_annotation/network_traffic_annotation.h"
#include "services/network/public/cpp/resource_request.h"
#include "services/network/public/cpp/shared_url_loader_factory.h"
#include "services/network/public/cpp/simple_url_loader.h"

namespace content {

// Implementation of URLProvisionFetcher.

URLProvisionFetcher::URLProvisionFetcher(
    scoped_refptr<network::SharedURLLoaderFactory> url_loader_factory)
    : url_loader_factory_(std::move(url_loader_factory)) {
  DCHECK(url_loader_factory_);
}

URLProvisionFetcher::~URLProvisionFetcher() {}

void URLProvisionFetcher::Retrieve(
    const std::string& default_url,
    const std::string& request_data,
    const media::ProvisionFetcher::ResponseCB& response_cb) {
  response_cb_ = response_cb;

  const std::string request_string =
      default_url + "&signedRequest=" + request_data;
  DVLOG(1) << __func__ << ": request:" << request_string;

  DCHECK(!simple_url_loader_);
  net::NetworkTrafficAnnotationTag traffic_annotation =
      net::DefineNetworkTrafficAnnotation("url_prevision_fetcher", R"(
        semantics {
          sender: "Content Decryption Module"
          description:
            "For a Content Decryption Module (CDM) to obtain origin-specific "
            "identifiers from an individualization or provisioning server. See "
            "https://w3c.github.io/encrypted-media/#direct-individualization."
          trigger:
            "During protected content playback, if the CDM hasn’t been "
            "provisioned yet, it may trigger a provision request which will be "
            "sent to a provisioning server."
          data:
            "Opaque provision request generated by the CDM. It may contain "
            "distinctive identifiers (see "
            "https://w3c.github.io/encrypted-media/#distinctive-identifier) "
            "and/or distinctive permanent identifiers (see "
            "https://w3c.github.io/encrypted-media/#distinctive-permanent-"
            "identifier), which must be encrypted. It does NOT contain origin "
            "information, even in encrypted form."
          destination: OTHER
        }
        policy {
          cookies_allowed: NO
          setting:
            "On Android, users can disable this feature by disabling Protected "
            "Media Identifier permissions."
          policy_exception_justification: "Not implemented."
        })");
  auto resource_request = std::make_unique<network::ResourceRequest>();
  resource_request->url = GURL(request_string);
  resource_request->load_flags =
      net::LOAD_DO_NOT_SAVE_COOKIES | net::LOAD_DO_NOT_SEND_COOKIES;
  resource_request->method = "POST";
  resource_request->headers.SetHeader("User-Agent", "Widevine CDM v1.0");
  simple_url_loader_ = network::SimpleURLLoader::Create(
      std::move(resource_request), traffic_annotation);
  simple_url_loader_->AttachStringForUpload("", "application/json");
  simple_url_loader_->DownloadToStringOfUnboundedSizeUntilCrashAndDie(
      url_loader_factory_.get(),
      base::BindOnce(&URLProvisionFetcher::OnSimpleLoaderComplete,
                     base::Unretained(this)));
}

void URLProvisionFetcher::OnSimpleLoaderComplete(
    std::unique_ptr<std::string> response_body) {
  bool success = false;
  std::string response;
  if (response_body) {
    success = true;
    response = std::move(*response_body);
  } else {
    int response_code = -1;
    if (simple_url_loader_->ResponseInfo() &&
        simple_url_loader_->ResponseInfo()->headers) {
      response_code =
          simple_url_loader_->ResponseInfo()->headers->response_code();
    }
    DVLOG(1) << "CDM provision: server returned error code " << response_code;
  }

  simple_url_loader_.reset();
  response_cb_.Run(success, response);
}

// Implementation of content public method CreateProvisionFetcher().

std::unique_ptr<media::ProvisionFetcher> CreateProvisionFetcher(
    scoped_refptr<network::SharedURLLoaderFactory> url_loader_factory) {
  DCHECK(url_loader_factory);
  return std::make_unique<URLProvisionFetcher>(std::move(url_loader_factory));
}

}  // namespace content
