// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_WEB_APPLICATIONS_EXTENSIONS_PENDING_BOOKMARK_APP_MANAGER_H_
#define CHROME_BROWSER_WEB_APPLICATIONS_EXTENSIONS_PENDING_BOOKMARK_APP_MANAGER_H_

#include <deque>
#include <memory>
#include <string>
#include <vector>

#include "base/callback.h"
#include "base/macros.h"
#include "base/memory/weak_ptr.h"
#include "base/optional.h"
#include "chrome/browser/web_applications/components/install_options.h"
#include "chrome/browser/web_applications/components/pending_app_manager.h"
#include "chrome/browser/web_applications/components/web_app_url_loader.h"
#include "chrome/browser/web_applications/extensions/bookmark_app_installation_task.h"
#include "chrome/browser/web_applications/extensions/bookmark_app_uninstaller.h"
#include "chrome/browser/web_applications/extensions/web_app_extension_ids_map.h"

class GURL;
class Profile;

namespace content {
class WebContents;
}  // namespace content

namespace web_app {
class AppRegistrar;
class InstallFinalizer;
class WebAppUiDelegate;
}  // namespace web_app

namespace extensions {

// Implementation of web_app::PendingAppManager that manages the set of
// Bookmark Apps which are being installed, uninstalled, and updated.
//
// WebAppProvider creates an instance of this class and manages its
// lifetime. This class should only be used from the UI thread.
class PendingBookmarkAppManager final : public web_app::PendingAppManager {
 public:
  using WebContentsFactory =
      base::RepeatingCallback<std::unique_ptr<content::WebContents>(Profile*)>;
  using TaskFactory =
      base::RepeatingCallback<std::unique_ptr<BookmarkAppInstallationTask>(
          Profile*,
          web_app::InstallFinalizer*,
          web_app::InstallOptions)>;

  explicit PendingBookmarkAppManager(
      Profile* profile,
      web_app::AppRegistrar* registrar,
      web_app::InstallFinalizer* install_finalizer);
  ~PendingBookmarkAppManager() override;

  // web_app::PendingAppManager
  void Install(web_app::InstallOptions install_options,
               OnceInstallCallback callback) override;
  void InstallApps(std::vector<web_app::InstallOptions> install_options_list,
                   const RepeatingInstallCallback& callback) override;
  void UninstallApps(std::vector<GURL> uninstall_urls,
                     const UninstallCallback& callback) override;
  std::vector<GURL> GetInstalledAppUrls(
      web_app::InstallSource install_source) const override;
  base::Optional<web_app::AppId> LookupAppId(const GURL& url) const override;
  bool HasAppIdWithInstallSource(
      const web_app::AppId& app_id,
      web_app::InstallSource install_source) const override;

  void SetTaskFactoryForTesting(TaskFactory task_factory);
  void SetUninstallerForTesting(
      std::unique_ptr<BookmarkAppUninstaller> uninstaller);
  void SetUrlLoaderForTesting(
      std::unique_ptr<web_app::WebAppUrlLoader> url_loader);

 private:
  struct TaskAndCallback;

  web_app::WebAppUiDelegate& GetUiDelegate();

  void MaybeStartNextInstallation();

  bool UninstallPlaceholderIfNecessary(
      const web_app::InstallOptions install_options);

  void StartInstallationTask(std::unique_ptr<TaskAndCallback> task);

  void CreateWebContentsIfNecessary();

  void OnUrlLoaded(web_app::WebAppUrlLoader::Result result);

  void OnInstalled(BookmarkAppInstallationTask::Result result);

  void CurrentInstallationFinished(const base::Optional<std::string>& app_id);

  Profile* profile_;
  web_app::AppRegistrar* registrar_;
  web_app::InstallFinalizer* install_finalizer_;
  std::unique_ptr<BookmarkAppUninstaller> uninstaller_;
  web_app::ExtensionIdsMap extension_ids_map_;

  // unique_ptr so that it can be replaced in tests.
  std::unique_ptr<web_app::WebAppUrlLoader> url_loader_;

  TaskFactory task_factory_;

  std::unique_ptr<content::WebContents> web_contents_;

  std::unique_ptr<TaskAndCallback> current_task_and_callback_;

  std::deque<std::unique_ptr<TaskAndCallback>> pending_tasks_and_callbacks_;

  base::WeakPtrFactory<PendingBookmarkAppManager> weak_ptr_factory_{this};

  DISALLOW_COPY_AND_ASSIGN(PendingBookmarkAppManager);
};

}  // namespace extensions

#endif  // CHROME_BROWSER_WEB_APPLICATIONS_EXTENSIONS_PENDING_BOOKMARK_APP_MANAGER_H_
