// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/omnibox/browser/omnibox_pedal_implementations.h"

#include "base/strings/utf_string_conversions.h"
#include "base/test/scoped_task_environment.h"
#include "base/time/time.h"
#include "components/omnibox/browser/mock_autocomplete_provider_client.h"
#include "components/omnibox/browser/omnibox_pedal_provider.h"
#include "components/omnibox/browser/test_omnibox_client.h"
#include "components/omnibox/browser/test_omnibox_edit_controller.h"
#include "testing/gtest/include/gtest/gtest.h"

class OmniboxPedalImplementationsTest : public testing::Test {
 protected:
  OmniboxPedalImplementationsTest()
      : omnibox_client_(new TestOmniboxClient),
        omnibox_edit_controller_(new TestOmniboxEditController) {}

  base::test::ScopedTaskEnvironment task_environment_;
  std::unique_ptr<TestOmniboxClient> omnibox_client_;
  std::unique_ptr<TestOmniboxEditController> omnibox_edit_controller_;
};

TEST_F(OmniboxPedalImplementationsTest, ClientReadiesPedalUpdateChrome) {
  MockAutocompleteProviderClient client;
  const OmniboxPedalUpdateChrome pedal;
  EXPECT_EQ(false, pedal.IsReadyToTrigger(client));
  client.set_browser_update_available(true);
  EXPECT_EQ(true, pedal.IsReadyToTrigger(client));
}

TEST_F(OmniboxPedalImplementationsTest, ProviderFiltersPedalUpdateChrome) {
  MockAutocompleteProviderClient client;
  OmniboxPedalProvider provider(client);
  const base::string16 trigger = base::ASCIIToUTF16("update chrome");
  const OmniboxPedal* pedal = provider.FindPedalMatch(trigger);
  EXPECT_EQ(pedal, nullptr) << "Pedal not filtered by condition.";
  client.set_browser_update_available(true);
  pedal = provider.FindPedalMatch(trigger);
  EXPECT_NE(pedal, nullptr) << "Pedal not discovered though condition is met.";
  EXPECT_TRUE(pedal->IsTriggerMatch(provider.Tokenize(trigger)));
}

TEST_F(OmniboxPedalImplementationsTest, PedalClearBrowsingDataExecutes) {
  MockAutocompleteProviderClient client;
  base::TimeTicks match_selection_timestamp;
  OmniboxPedal::ExecutionContext context(
      *omnibox_client_, *omnibox_edit_controller_, match_selection_timestamp);
  const OmniboxPedalClearBrowsingData pedal;
  pedal.Execute(context);
  const GURL& url = omnibox_edit_controller_->destination_url();
  EXPECT_EQ(url, GURL("chrome://settings/clearBrowserData"));
}

// Exhaustive test of unordered synonym groups for concept matches; this is
// useful in detecting possible clashes between cross-group synonym strings.
// By ensuring that each set matches for exactly one Pedal, we can also
// prevent clashes between concepts for different Pedals.
TEST_F(OmniboxPedalImplementationsTest,
       UnorderedSynonymExpressionsAreConceptMatches) {
  const std::vector<std::vector<const char*>> literal_concept_expressions = {
      // Note: The lists below are auto-generated from raw synonym group data.

      // Clear Browsing Data
      {
          "browser cache clear",
          "browser cache delete",
          "browser cache erase",
          "browser cache remove",
          "browser cache wipe",
          "browser clear cache",
          "browser clear data",
          "browser clear history",
          "browser data clear",
          "browser data delete",
          "browser data erase",
          "browser data remove",
          "browser data wipe",
          "browser delete cache",
          "browser delete data",
          "browser delete history",
          "browser erase cache",
          "browser erase data",
          "browser erase history",
          "browser history clear",
          "browser history delete",
          "browser history erase",
          "browser history remove",
          "browser history wipe",
          "browser remove cache",
          "browser remove data",
          "browser remove history",
          "browser wipe cache",
          "browser wipe data",
          "browser wipe history",
          "cache browser clear",
          "cache browser delete",
          "cache browser erase",
          "cache browser remove",
          "cache browser wipe",
          "cache chrome clear",
          "cache chrome delete",
          "cache chrome erase",
          "cache chrome remove",
          "cache chrome wipe",
          "cache clear",
          "cache clear browser",
          "cache clear chrome",
          "cache clear google chrome",
          "cache delete",
          "cache delete browser",
          "cache delete chrome",
          "cache delete google chrome",
          "cache erase",
          "cache erase browser",
          "cache erase chrome",
          "cache erase google chrome",
          "cache google chrome clear",
          "cache google chrome delete",
          "cache google chrome erase",
          "cache google chrome remove",
          "cache google chrome wipe",
          "cache remove",
          "cache remove browser",
          "cache remove chrome",
          "cache remove google chrome",
          "cache wipe",
          "cache wipe browser",
          "cache wipe chrome",
          "cache wipe google chrome",
          "chrome cache clear",
          "chrome cache delete",
          "chrome cache erase",
          "chrome cache remove",
          "chrome cache wipe",
          "chrome clear cache",
          "chrome clear data",
          "chrome clear history",
          "chrome data clear",
          "chrome data delete",
          "chrome data erase",
          "chrome data remove",
          "chrome data wipe",
          "chrome delete cache",
          "chrome delete data",
          "chrome delete history",
          "chrome erase cache",
          "chrome erase data",
          "chrome erase history",
          "chrome history clear",
          "chrome history delete",
          "chrome history erase",
          "chrome history remove",
          "chrome history wipe",
          "chrome remove cache",
          "chrome remove data",
          "chrome remove history",
          "chrome wipe cache",
          "chrome wipe data",
          "chrome wipe history",
          "clear browser cache",
          "clear browser data",
          "clear browser history",
          "clear cache",
          "clear cache browser",
          "clear cache chrome",
          "clear cache google chrome",
          "clear chrome cache",
          "clear chrome data",
          "clear chrome history",
          "clear data",
          "clear data browser",
          "clear data chrome",
          "clear data google chrome",
          "clear google chrome cache",
          "clear google chrome data",
          "clear google chrome history",
          "clear history",
          "clear history browser",
          "clear history chrome",
          "clear history google chrome",
          "data browser clear",
          "data browser delete",
          "data browser erase",
          "data browser remove",
          "data browser wipe",
          "data chrome clear",
          "data chrome delete",
          "data chrome erase",
          "data chrome remove",
          "data chrome wipe",
          "data clear",
          "data clear browser",
          "data clear chrome",
          "data clear google chrome",
          "data delete",
          "data delete browser",
          "data delete chrome",
          "data delete google chrome",
          "data erase",
          "data erase browser",
          "data erase chrome",
          "data erase google chrome",
          "data google chrome clear",
          "data google chrome delete",
          "data google chrome erase",
          "data google chrome remove",
          "data google chrome wipe",
          "data remove",
          "data remove browser",
          "data remove chrome",
          "data remove google chrome",
          "data wipe",
          "data wipe browser",
          "data wipe chrome",
          "data wipe google chrome",
          "delete browser cache",
          "delete browser data",
          "delete browser history",
          "delete cache",
          "delete cache browser",
          "delete cache chrome",
          "delete cache google chrome",
          "delete chrome cache",
          "delete chrome data",
          "delete chrome history",
          "delete data",
          "delete data browser",
          "delete data chrome",
          "delete data google chrome",
          "delete google chrome cache",
          "delete google chrome data",
          "delete google chrome history",
          "delete history",
          "delete history browser",
          "delete history chrome",
          "delete history google chrome",
          "erase browser cache",
          "erase browser data",
          "erase browser history",
          "erase cache",
          "erase cache browser",
          "erase cache chrome",
          "erase cache google chrome",
          "erase chrome cache",
          "erase chrome data",
          "erase chrome history",
          "erase data",
          "erase data browser",
          "erase data chrome",
          "erase data google chrome",
          "erase google chrome cache",
          "erase google chrome data",
          "erase google chrome history",
          "erase history",
          "erase history browser",
          "erase history chrome",
          "erase history google chrome",
          "google chrome cache clear",
          "google chrome cache delete",
          "google chrome cache erase",
          "google chrome cache remove",
          "google chrome cache wipe",
          "google chrome clear cache",
          "google chrome clear data",
          "google chrome clear history",
          "google chrome data clear",
          "google chrome data delete",
          "google chrome data erase",
          "google chrome data remove",
          "google chrome data wipe",
          "google chrome delete cache",
          "google chrome delete data",
          "google chrome delete history",
          "google chrome erase cache",
          "google chrome erase data",
          "google chrome erase history",
          "google chrome history clear",
          "google chrome history delete",
          "google chrome history erase",
          "google chrome history remove",
          "google chrome history wipe",
          "google chrome remove cache",
          "google chrome remove data",
          "google chrome remove history",
          "google chrome wipe cache",
          "google chrome wipe data",
          "google chrome wipe history",
          "history browser clear",
          "history browser delete",
          "history browser erase",
          "history browser remove",
          "history browser wipe",
          "history chrome clear",
          "history chrome delete",
          "history chrome erase",
          "history chrome remove",
          "history chrome wipe",
          "history clear",
          "history clear browser",
          "history clear chrome",
          "history clear google chrome",
          "history delete",
          "history delete browser",
          "history delete chrome",
          "history delete google chrome",
          "history erase",
          "history erase browser",
          "history erase chrome",
          "history erase google chrome",
          "history google chrome clear",
          "history google chrome delete",
          "history google chrome erase",
          "history google chrome remove",
          "history google chrome wipe",
          "history remove",
          "history remove browser",
          "history remove chrome",
          "history remove google chrome",
          "history wipe",
          "history wipe browser",
          "history wipe chrome",
          "history wipe google chrome",
          "remove browser cache",
          "remove browser data",
          "remove browser history",
          "remove cache",
          "remove cache browser",
          "remove cache chrome",
          "remove cache google chrome",
          "remove chrome cache",
          "remove chrome data",
          "remove chrome history",
          "remove data",
          "remove data browser",
          "remove data chrome",
          "remove data google chrome",
          "remove google chrome cache",
          "remove google chrome data",
          "remove google chrome history",
          "remove history",
          "remove history browser",
          "remove history chrome",
          "remove history google chrome",
          "wipe browser cache",
          "wipe browser data",
          "wipe browser history",
          "wipe cache",
          "wipe cache browser",
          "wipe cache chrome",
          "wipe cache google chrome",
          "wipe chrome cache",
          "wipe chrome data",
          "wipe chrome history",
          "wipe data",
          "wipe data browser",
          "wipe data chrome",
          "wipe data google chrome",
          "wipe google chrome cache",
          "wipe google chrome data",
          "wipe google chrome history",
          "wipe history",
          "wipe history browser",
          "wipe history chrome",
          "wipe history google chrome",
      },

      // Change Search Engine
      {
          "browser change default search engine",
          "browser change search",
          "browser change search engine",
          "browser change standard search engine",
          "browser choose default search engine",
          "browser choose search",
          "browser choose search engine",
          "browser choose standard search engine",
          "browser default search engine change",
          "browser default search engine choose",
          "browser default search engine select",
          "browser default search engine switch",
          "browser search change",
          "browser search choose",
          "browser search engine change",
          "browser search engine choose",
          "browser search engine select",
          "browser search engine switch",
          "browser search select",
          "browser search switch",
          "browser select default search engine",
          "browser select search",
          "browser select search engine",
          "browser select standard search engine",
          "browser standard search engine change",
          "browser standard search engine choose",
          "browser standard search engine select",
          "browser standard search engine switch",
          "browser switch default search engine",
          "browser switch search",
          "browser switch search engine",
          "browser switch standard search engine",
          "change browser default search engine",
          "change browser search",
          "change browser search engine",
          "change browser standard search engine",
          "change chrome default search engine",
          "change chrome search",
          "change chrome search engine",
          "change chrome standard search engine",
          "change default search engine",
          "change default search engine browser",
          "change default search engine chrome",
          "change default search engine google chrome",
          "change google chrome default search engine",
          "change google chrome search",
          "change google chrome search engine",
          "change google chrome standard search engine",
          "change search",
          "change search browser",
          "change search chrome",
          "change search engine",
          "change search engine browser",
          "change search engine chrome",
          "change search engine google chrome",
          "change search google chrome",
          "change standard search engine",
          "change standard search engine browser",
          "change standard search engine chrome",
          "change standard search engine google chrome",
          "choose browser default search engine",
          "choose browser search",
          "choose browser search engine",
          "choose browser standard search engine",
          "choose chrome default search engine",
          "choose chrome search",
          "choose chrome search engine",
          "choose chrome standard search engine",
          "choose default search engine",
          "choose default search engine browser",
          "choose default search engine chrome",
          "choose default search engine google chrome",
          "choose google chrome default search engine",
          "choose google chrome search",
          "choose google chrome search engine",
          "choose google chrome standard search engine",
          "choose search",
          "choose search browser",
          "choose search chrome",
          "choose search engine",
          "choose search engine browser",
          "choose search engine chrome",
          "choose search engine google chrome",
          "choose search google chrome",
          "choose standard search engine",
          "choose standard search engine browser",
          "choose standard search engine chrome",
          "choose standard search engine google chrome",
          "chrome change default search engine",
          "chrome change search",
          "chrome change search engine",
          "chrome change standard search engine",
          "chrome choose default search engine",
          "chrome choose search",
          "chrome choose search engine",
          "chrome choose standard search engine",
          "chrome default search engine change",
          "chrome default search engine choose",
          "chrome default search engine select",
          "chrome default search engine switch",
          "chrome search change",
          "chrome search choose",
          "chrome search engine change",
          "chrome search engine choose",
          "chrome search engine select",
          "chrome search engine switch",
          "chrome search select",
          "chrome search switch",
          "chrome select default search engine",
          "chrome select search",
          "chrome select search engine",
          "chrome select standard search engine",
          "chrome standard search engine change",
          "chrome standard search engine choose",
          "chrome standard search engine select",
          "chrome standard search engine switch",
          "chrome switch default search engine",
          "chrome switch search",
          "chrome switch search engine",
          "chrome switch standard search engine",
          "default search engine browser change",
          "default search engine browser choose",
          "default search engine browser select",
          "default search engine browser switch",
          "default search engine change",
          "default search engine change browser",
          "default search engine change chrome",
          "default search engine change google chrome",
          "default search engine choose",
          "default search engine choose browser",
          "default search engine choose chrome",
          "default search engine choose google chrome",
          "default search engine chrome change",
          "default search engine chrome choose",
          "default search engine chrome select",
          "default search engine chrome switch",
          "default search engine google chrome change",
          "default search engine google chrome choose",
          "default search engine google chrome select",
          "default search engine google chrome switch",
          "default search engine select",
          "default search engine select browser",
          "default search engine select chrome",
          "default search engine select google chrome",
          "default search engine switch",
          "default search engine switch browser",
          "default search engine switch chrome",
          "default search engine switch google chrome",
          "google chrome change default search engine",
          "google chrome change search",
          "google chrome change search engine",
          "google chrome change standard search engine",
          "google chrome choose default search engine",
          "google chrome choose search",
          "google chrome choose search engine",
          "google chrome choose standard search engine",
          "google chrome default search engine change",
          "google chrome default search engine choose",
          "google chrome default search engine select",
          "google chrome default search engine switch",
          "google chrome search change",
          "google chrome search choose",
          "google chrome search engine change",
          "google chrome search engine choose",
          "google chrome search engine select",
          "google chrome search engine switch",
          "google chrome search select",
          "google chrome search switch",
          "google chrome select default search engine",
          "google chrome select search",
          "google chrome select search engine",
          "google chrome select standard search engine",
          "google chrome standard search engine change",
          "google chrome standard search engine choose",
          "google chrome standard search engine select",
          "google chrome standard search engine switch",
          "google chrome switch default search engine",
          "google chrome switch search",
          "google chrome switch search engine",
          "google chrome switch standard search engine",
          "search browser change",
          "search browser choose",
          "search browser select",
          "search browser switch",
          "search change",
          "search change browser",
          "search change chrome",
          "search change google chrome",
          "search choose",
          "search choose browser",
          "search choose chrome",
          "search choose google chrome",
          "search chrome change",
          "search chrome choose",
          "search chrome select",
          "search chrome switch",
          "search engine browser change",
          "search engine browser choose",
          "search engine browser select",
          "search engine browser switch",
          "search engine change",
          "search engine change browser",
          "search engine change chrome",
          "search engine change google chrome",
          "search engine choose",
          "search engine choose browser",
          "search engine choose chrome",
          "search engine choose google chrome",
          "search engine chrome change",
          "search engine chrome choose",
          "search engine chrome select",
          "search engine chrome switch",
          "search engine google chrome change",
          "search engine google chrome choose",
          "search engine google chrome select",
          "search engine google chrome switch",
          "search engine select",
          "search engine select browser",
          "search engine select chrome",
          "search engine select google chrome",
          "search engine switch",
          "search engine switch browser",
          "search engine switch chrome",
          "search engine switch google chrome",
          "search google chrome change",
          "search google chrome choose",
          "search google chrome select",
          "search google chrome switch",
          "search select",
          "search select browser",
          "search select chrome",
          "search select google chrome",
          "search switch",
          "search switch browser",
          "search switch chrome",
          "search switch google chrome",
          "select browser default search engine",
          "select browser search",
          "select browser search engine",
          "select browser standard search engine",
          "select chrome default search engine",
          "select chrome search",
          "select chrome search engine",
          "select chrome standard search engine",
          "select default search engine",
          "select default search engine browser",
          "select default search engine chrome",
          "select default search engine google chrome",
          "select google chrome default search engine",
          "select google chrome search",
          "select google chrome search engine",
          "select google chrome standard search engine",
          "select search",
          "select search browser",
          "select search chrome",
          "select search engine",
          "select search engine browser",
          "select search engine chrome",
          "select search engine google chrome",
          "select search google chrome",
          "select standard search engine",
          "select standard search engine browser",
          "select standard search engine chrome",
          "select standard search engine google chrome",
          "standard search engine browser change",
          "standard search engine browser choose",
          "standard search engine browser select",
          "standard search engine browser switch",
          "standard search engine change",
          "standard search engine change browser",
          "standard search engine change chrome",
          "standard search engine change google chrome",
          "standard search engine choose",
          "standard search engine choose browser",
          "standard search engine choose chrome",
          "standard search engine choose google chrome",
          "standard search engine chrome change",
          "standard search engine chrome choose",
          "standard search engine chrome select",
          "standard search engine chrome switch",
          "standard search engine google chrome change",
          "standard search engine google chrome choose",
          "standard search engine google chrome select",
          "standard search engine google chrome switch",
          "standard search engine select",
          "standard search engine select browser",
          "standard search engine select chrome",
          "standard search engine select google chrome",
          "standard search engine switch",
          "standard search engine switch browser",
          "standard search engine switch chrome",
          "standard search engine switch google chrome",
          "switch browser default search engine",
          "switch browser search",
          "switch browser search engine",
          "switch browser standard search engine",
          "switch chrome default search engine",
          "switch chrome search",
          "switch chrome search engine",
          "switch chrome standard search engine",
          "switch default search engine",
          "switch default search engine browser",
          "switch default search engine chrome",
          "switch default search engine google chrome",
          "switch google chrome default search engine",
          "switch google chrome search",
          "switch google chrome search engine",
          "switch google chrome standard search engine",
          "switch search",
          "switch search browser",
          "switch search chrome",
          "switch search engine",
          "switch search engine browser",
          "switch search engine chrome",
          "switch search engine google chrome",
          "switch search google chrome",
          "switch standard search engine",
          "switch standard search engine browser",
          "switch standard search engine chrome",
          "switch standard search engine google chrome",
      },

      // Manage Passwords
      {
          "browser change passwords",
          "browser manage passwords",
          "browser manager passwords",
          "browser passwords change",
          "browser passwords manage",
          "browser passwords manager",
          "browser passwords update",
          "browser update passwords",
          "change browser passwords",
          "change chrome passwords",
          "change google chrome passwords",
          "change passwords",
          "change passwords browser",
          "change passwords chrome",
          "change passwords google chrome",
          "chrome change passwords",
          "chrome manage passwords",
          "chrome manager passwords",
          "chrome passwords change",
          "chrome passwords manage",
          "chrome passwords manager",
          "chrome passwords update",
          "chrome update passwords",
          "google chrome change passwords",
          "google chrome manage passwords",
          "google chrome manager passwords",
          "google chrome passwords change",
          "google chrome passwords manage",
          "google chrome passwords manager",
          "google chrome passwords update",
          "google chrome update passwords",
          "manage browser passwords",
          "manage chrome passwords",
          "manage google chrome passwords",
          "manage passwords",
          "manage passwords browser",
          "manage passwords chrome",
          "manage passwords google chrome",
          "manager browser passwords",
          "manager chrome passwords",
          "manager google chrome passwords",
          "manager passwords",
          "manager passwords browser",
          "manager passwords chrome",
          "manager passwords google chrome",
          "passwords browser change",
          "passwords browser manage",
          "passwords browser manager",
          "passwords browser update",
          "passwords change",
          "passwords change browser",
          "passwords change chrome",
          "passwords change google chrome",
          "passwords chrome change",
          "passwords chrome manage",
          "passwords chrome manager",
          "passwords chrome update",
          "passwords google chrome change",
          "passwords google chrome manage",
          "passwords google chrome manager",
          "passwords google chrome update",
          "passwords manage",
          "passwords manage browser",
          "passwords manage chrome",
          "passwords manage google chrome",
          "passwords manager",
          "passwords manager browser",
          "passwords manager chrome",
          "passwords manager google chrome",
          "passwords update",
          "passwords update browser",
          "passwords update chrome",
          "passwords update google chrome",
          "update browser passwords",
          "update chrome passwords",
          "update google chrome passwords",
          "update passwords",
          "update passwords browser",
          "update passwords chrome",
          "update passwords google chrome",
      },

      // Change Home Page
      {
          "browser change home page",
          "browser change homepage",
          "browser choose home page",
          "browser choose homepage",
          "browser home page change",
          "browser home page choose",
          "browser home page set",
          "browser homepage change",
          "browser homepage choose",
          "browser homepage set",
          "browser set home page",
          "browser set homepage",
          "change browser home page",
          "change browser homepage",
          "change chrome home page",
          "change chrome homepage",
          "change google chrome home page",
          "change google chrome homepage",
          "change home page",
          "change home page browser",
          "change home page chrome",
          "change home page google chrome",
          "change homepage",
          "change homepage browser",
          "change homepage chrome",
          "change homepage google chrome",
          "choose browser home page",
          "choose browser homepage",
          "choose chrome home page",
          "choose chrome homepage",
          "choose google chrome home page",
          "choose google chrome homepage",
          "choose home page",
          "choose home page browser",
          "choose home page chrome",
          "choose home page google chrome",
          "choose homepage",
          "choose homepage browser",
          "choose homepage chrome",
          "choose homepage google chrome",
          "chrome change home page",
          "chrome change homepage",
          "chrome choose home page",
          "chrome choose homepage",
          "chrome home page change",
          "chrome home page choose",
          "chrome home page set",
          "chrome homepage change",
          "chrome homepage choose",
          "chrome homepage set",
          "chrome set home page",
          "chrome set homepage",
          "google chrome change home page",
          "google chrome change homepage",
          "google chrome choose home page",
          "google chrome choose homepage",
          "google chrome home page change",
          "google chrome home page choose",
          "google chrome home page set",
          "google chrome homepage change",
          "google chrome homepage choose",
          "google chrome homepage set",
          "google chrome set home page",
          "google chrome set homepage",
          "home page browser change",
          "home page browser choose",
          "home page browser set",
          "home page change",
          "home page change browser",
          "home page change chrome",
          "home page change google chrome",
          "home page choose",
          "home page choose browser",
          "home page choose chrome",
          "home page choose google chrome",
          "home page chrome change",
          "home page chrome choose",
          "home page chrome set",
          "home page google chrome change",
          "home page google chrome choose",
          "home page google chrome set",
          "home page set",
          "home page set browser",
          "home page set chrome",
          "home page set google chrome",
          "homepage browser change",
          "homepage browser choose",
          "homepage browser set",
          "homepage change",
          "homepage change browser",
          "homepage change chrome",
          "homepage change google chrome",
          "homepage choose",
          "homepage choose browser",
          "homepage choose chrome",
          "homepage choose google chrome",
          "homepage chrome change",
          "homepage chrome choose",
          "homepage chrome set",
          "homepage google chrome change",
          "homepage google chrome choose",
          "homepage google chrome set",
          "homepage set",
          "homepage set browser",
          "homepage set chrome",
          "homepage set google chrome",
          "set browser home page",
          "set browser homepage",
          "set chrome home page",
          "set chrome homepage",
          "set google chrome home page",
          "set google chrome homepage",
          "set home page",
          "set home page browser",
          "set home page chrome",
          "set home page google chrome",
          "set homepage",
          "set homepage browser",
          "set homepage chrome",
          "set homepage google chrome",
      },

      // Update Credit Card
      {
          "browser card info edit",
          "browser card info update",
          "browser cards edit",
          "browser cards update",
          "browser credit card edit",
          "browser credit card info edit",
          "browser credit card info update",
          "browser credit card update",
          "browser credit cards edit",
          "browser credit cards update",
          "browser edit card info",
          "browser edit cards",
          "browser edit credit card",
          "browser edit credit card info",
          "browser edit credit cards",
          "browser update card info",
          "browser update cards",
          "browser update credit card",
          "browser update credit card info",
          "browser update credit cards",
          "card info browser edit",
          "card info browser update",
          "card info chrome edit",
          "card info chrome update",
          "card info edit",
          "card info edit browser",
          "card info edit chrome",
          "card info edit google chrome",
          "card info google chrome edit",
          "card info google chrome update",
          "card info update",
          "card info update browser",
          "card info update chrome",
          "card info update google chrome",
          "cards browser edit",
          "cards browser update",
          "cards chrome edit",
          "cards chrome update",
          "cards edit",
          "cards edit browser",
          "cards edit chrome",
          "cards edit google chrome",
          "cards google chrome edit",
          "cards google chrome update",
          "cards update",
          "cards update browser",
          "cards update chrome",
          "cards update google chrome",
          "chrome card info edit",
          "chrome card info update",
          "chrome cards edit",
          "chrome cards update",
          "chrome credit card edit",
          "chrome credit card info edit",
          "chrome credit card info update",
          "chrome credit card update",
          "chrome credit cards edit",
          "chrome credit cards update",
          "chrome edit card info",
          "chrome edit cards",
          "chrome edit credit card",
          "chrome edit credit card info",
          "chrome edit credit cards",
          "chrome update card info",
          "chrome update cards",
          "chrome update credit card",
          "chrome update credit card info",
          "chrome update credit cards",
          "credit card browser edit",
          "credit card browser update",
          "credit card chrome edit",
          "credit card chrome update",
          "credit card edit",
          "credit card edit browser",
          "credit card edit chrome",
          "credit card edit google chrome",
          "credit card google chrome edit",
          "credit card google chrome update",
          "credit card info browser edit",
          "credit card info browser update",
          "credit card info chrome edit",
          "credit card info chrome update",
          "credit card info edit",
          "credit card info edit browser",
          "credit card info edit chrome",
          "credit card info edit google chrome",
          "credit card info google chrome edit",
          "credit card info google chrome update",
          "credit card info update",
          "credit card info update browser",
          "credit card info update chrome",
          "credit card info update google chrome",
          "credit card update",
          "credit card update browser",
          "credit card update chrome",
          "credit card update google chrome",
          "credit cards browser edit",
          "credit cards browser update",
          "credit cards chrome edit",
          "credit cards chrome update",
          "credit cards edit",
          "credit cards edit browser",
          "credit cards edit chrome",
          "credit cards edit google chrome",
          "credit cards google chrome edit",
          "credit cards google chrome update",
          "credit cards update",
          "credit cards update browser",
          "credit cards update chrome",
          "credit cards update google chrome",
          "edit browser card info",
          "edit browser cards",
          "edit browser credit card",
          "edit browser credit card info",
          "edit browser credit cards",
          "edit card info",
          "edit card info browser",
          "edit card info chrome",
          "edit card info google chrome",
          "edit cards",
          "edit cards browser",
          "edit cards chrome",
          "edit cards google chrome",
          "edit chrome card info",
          "edit chrome cards",
          "edit chrome credit card",
          "edit chrome credit card info",
          "edit chrome credit cards",
          "edit credit card",
          "edit credit card browser",
          "edit credit card chrome",
          "edit credit card google chrome",
          "edit credit card info",
          "edit credit card info browser",
          "edit credit card info chrome",
          "edit credit card info google chrome",
          "edit credit cards",
          "edit credit cards browser",
          "edit credit cards chrome",
          "edit credit cards google chrome",
          "edit google chrome card info",
          "edit google chrome cards",
          "edit google chrome credit card",
          "edit google chrome credit card info",
          "edit google chrome credit cards",
          "google chrome card info edit",
          "google chrome card info update",
          "google chrome cards edit",
          "google chrome cards update",
          "google chrome credit card edit",
          "google chrome credit card info edit",
          "google chrome credit card info update",
          "google chrome credit card update",
          "google chrome credit cards edit",
          "google chrome credit cards update",
          "google chrome edit card info",
          "google chrome edit cards",
          "google chrome edit credit card",
          "google chrome edit credit card info",
          "google chrome edit credit cards",
          "google chrome update card info",
          "google chrome update cards",
          "google chrome update credit card",
          "google chrome update credit card info",
          "google chrome update credit cards",
          "update browser card info",
          "update browser cards",
          "update browser credit card",
          "update browser credit card info",
          "update browser credit cards",
          "update card info",
          "update card info browser",
          "update card info chrome",
          "update card info google chrome",
          "update cards",
          "update cards browser",
          "update cards chrome",
          "update cards google chrome",
          "update chrome card info",
          "update chrome cards",
          "update chrome credit card",
          "update chrome credit card info",
          "update chrome credit cards",
          "update credit card",
          "update credit card browser",
          "update credit card chrome",
          "update credit card google chrome",
          "update credit card info",
          "update credit card info browser",
          "update credit card info chrome",
          "update credit card info google chrome",
          "update credit cards",
          "update credit cards browser",
          "update credit cards chrome",
          "update credit cards google chrome",
          "update google chrome card info",
          "update google chrome cards",
          "update google chrome credit card",
          "update google chrome credit card info",
          "update google chrome credit cards",
      },

      // Launch Incognito
      {
          "browser enter incognito",
          "browser enter incognito mode",
          "browser enter incognito tab",
          "browser enter incognito window",
          "browser enter private mode",
          "browser enter private tab",
          "browser enter private window",
          "browser incognito enter",
          "browser incognito launch",
          "browser incognito mode enter",
          "browser incognito mode launch",
          "browser incognito mode open",
          "browser incognito mode start",
          "browser incognito open",
          "browser incognito start",
          "browser incognito tab enter",
          "browser incognito tab launch",
          "browser incognito tab open",
          "browser incognito tab start",
          "browser incognito window enter",
          "browser incognito window launch",
          "browser incognito window open",
          "browser incognito window start",
          "browser launch incognito",
          "browser launch incognito mode",
          "browser launch incognito tab",
          "browser launch incognito window",
          "browser launch private mode",
          "browser launch private tab",
          "browser launch private window",
          "browser open incognito",
          "browser open incognito mode",
          "browser open incognito tab",
          "browser open incognito window",
          "browser open private mode",
          "browser open private tab",
          "browser open private window",
          "browser private mode enter",
          "browser private mode launch",
          "browser private mode open",
          "browser private mode start",
          "browser private tab enter",
          "browser private tab launch",
          "browser private tab open",
          "browser private tab start",
          "browser private window enter",
          "browser private window launch",
          "browser private window open",
          "browser private window start",
          "browser start incognito",
          "browser start incognito mode",
          "browser start incognito tab",
          "browser start incognito window",
          "browser start private mode",
          "browser start private tab",
          "browser start private window",
          "chrome enter incognito",
          "chrome enter incognito mode",
          "chrome enter incognito tab",
          "chrome enter incognito window",
          "chrome enter private mode",
          "chrome enter private tab",
          "chrome enter private window",
          "chrome incognito enter",
          "chrome incognito launch",
          "chrome incognito mode enter",
          "chrome incognito mode launch",
          "chrome incognito mode open",
          "chrome incognito mode start",
          "chrome incognito open",
          "chrome incognito start",
          "chrome incognito tab enter",
          "chrome incognito tab launch",
          "chrome incognito tab open",
          "chrome incognito tab start",
          "chrome incognito window enter",
          "chrome incognito window launch",
          "chrome incognito window open",
          "chrome incognito window start",
          "chrome launch incognito",
          "chrome launch incognito mode",
          "chrome launch incognito tab",
          "chrome launch incognito window",
          "chrome launch private mode",
          "chrome launch private tab",
          "chrome launch private window",
          "chrome open incognito",
          "chrome open incognito mode",
          "chrome open incognito tab",
          "chrome open incognito window",
          "chrome open private mode",
          "chrome open private tab",
          "chrome open private window",
          "chrome private mode enter",
          "chrome private mode launch",
          "chrome private mode open",
          "chrome private mode start",
          "chrome private tab enter",
          "chrome private tab launch",
          "chrome private tab open",
          "chrome private tab start",
          "chrome private window enter",
          "chrome private window launch",
          "chrome private window open",
          "chrome private window start",
          "chrome start incognito",
          "chrome start incognito mode",
          "chrome start incognito tab",
          "chrome start incognito window",
          "chrome start private mode",
          "chrome start private tab",
          "chrome start private window",
          "enter browser incognito",
          "enter browser incognito mode",
          "enter browser incognito tab",
          "enter browser incognito window",
          "enter browser private mode",
          "enter browser private tab",
          "enter browser private window",
          "enter chrome incognito",
          "enter chrome incognito mode",
          "enter chrome incognito tab",
          "enter chrome incognito window",
          "enter chrome private mode",
          "enter chrome private tab",
          "enter chrome private window",
          "enter google chrome incognito",
          "enter google chrome incognito mode",
          "enter google chrome incognito tab",
          "enter google chrome incognito window",
          "enter google chrome private mode",
          "enter google chrome private tab",
          "enter google chrome private window",
          "enter incognito",
          "enter incognito browser",
          "enter incognito chrome",
          "enter incognito google chrome",
          "enter incognito mode",
          "enter incognito mode browser",
          "enter incognito mode chrome",
          "enter incognito mode google chrome",
          "enter incognito tab",
          "enter incognito tab browser",
          "enter incognito tab chrome",
          "enter incognito tab google chrome",
          "enter incognito window",
          "enter incognito window browser",
          "enter incognito window chrome",
          "enter incognito window google chrome",
          "enter private mode",
          "enter private mode browser",
          "enter private mode chrome",
          "enter private mode google chrome",
          "enter private tab",
          "enter private tab browser",
          "enter private tab chrome",
          "enter private tab google chrome",
          "enter private window",
          "enter private window browser",
          "enter private window chrome",
          "enter private window google chrome",
          "google chrome enter incognito",
          "google chrome enter incognito mode",
          "google chrome enter incognito tab",
          "google chrome enter incognito window",
          "google chrome enter private mode",
          "google chrome enter private tab",
          "google chrome enter private window",
          "google chrome incognito enter",
          "google chrome incognito launch",
          "google chrome incognito mode enter",
          "google chrome incognito mode launch",
          "google chrome incognito mode open",
          "google chrome incognito mode start",
          "google chrome incognito open",
          "google chrome incognito start",
          "google chrome incognito tab enter",
          "google chrome incognito tab launch",
          "google chrome incognito tab open",
          "google chrome incognito tab start",
          "google chrome incognito window enter",
          "google chrome incognito window launch",
          "google chrome incognito window open",
          "google chrome incognito window start",
          "google chrome launch incognito",
          "google chrome launch incognito mode",
          "google chrome launch incognito tab",
          "google chrome launch incognito window",
          "google chrome launch private mode",
          "google chrome launch private tab",
          "google chrome launch private window",
          "google chrome open incognito",
          "google chrome open incognito mode",
          "google chrome open incognito tab",
          "google chrome open incognito window",
          "google chrome open private mode",
          "google chrome open private tab",
          "google chrome open private window",
          "google chrome private mode enter",
          "google chrome private mode launch",
          "google chrome private mode open",
          "google chrome private mode start",
          "google chrome private tab enter",
          "google chrome private tab launch",
          "google chrome private tab open",
          "google chrome private tab start",
          "google chrome private window enter",
          "google chrome private window launch",
          "google chrome private window open",
          "google chrome private window start",
          "google chrome start incognito",
          "google chrome start incognito mode",
          "google chrome start incognito tab",
          "google chrome start incognito window",
          "google chrome start private mode",
          "google chrome start private tab",
          "google chrome start private window",
          "incognito browser enter",
          "incognito browser launch",
          "incognito browser open",
          "incognito browser start",
          "incognito chrome enter",
          "incognito chrome launch",
          "incognito chrome open",
          "incognito chrome start",
          "incognito enter",
          "incognito enter browser",
          "incognito enter chrome",
          "incognito enter google chrome",
          "incognito google chrome enter",
          "incognito google chrome launch",
          "incognito google chrome open",
          "incognito google chrome start",
          "incognito launch",
          "incognito launch browser",
          "incognito launch chrome",
          "incognito launch google chrome",
          "incognito mode browser enter",
          "incognito mode browser launch",
          "incognito mode browser open",
          "incognito mode browser start",
          "incognito mode chrome enter",
          "incognito mode chrome launch",
          "incognito mode chrome open",
          "incognito mode chrome start",
          "incognito mode enter",
          "incognito mode enter browser",
          "incognito mode enter chrome",
          "incognito mode enter google chrome",
          "incognito mode google chrome enter",
          "incognito mode google chrome launch",
          "incognito mode google chrome open",
          "incognito mode google chrome start",
          "incognito mode launch",
          "incognito mode launch browser",
          "incognito mode launch chrome",
          "incognito mode launch google chrome",
          "incognito mode open",
          "incognito mode open browser",
          "incognito mode open chrome",
          "incognito mode open google chrome",
          "incognito mode start",
          "incognito mode start browser",
          "incognito mode start chrome",
          "incognito mode start google chrome",
          "incognito open",
          "incognito open browser",
          "incognito open chrome",
          "incognito open google chrome",
          "incognito start",
          "incognito start browser",
          "incognito start chrome",
          "incognito start google chrome",
          "incognito tab browser enter",
          "incognito tab browser launch",
          "incognito tab browser open",
          "incognito tab browser start",
          "incognito tab chrome enter",
          "incognito tab chrome launch",
          "incognito tab chrome open",
          "incognito tab chrome start",
          "incognito tab enter",
          "incognito tab enter browser",
          "incognito tab enter chrome",
          "incognito tab enter google chrome",
          "incognito tab google chrome enter",
          "incognito tab google chrome launch",
          "incognito tab google chrome open",
          "incognito tab google chrome start",
          "incognito tab launch",
          "incognito tab launch browser",
          "incognito tab launch chrome",
          "incognito tab launch google chrome",
          "incognito tab open",
          "incognito tab open browser",
          "incognito tab open chrome",
          "incognito tab open google chrome",
          "incognito tab start",
          "incognito tab start browser",
          "incognito tab start chrome",
          "incognito tab start google chrome",
          "incognito window browser enter",
          "incognito window browser launch",
          "incognito window browser open",
          "incognito window browser start",
          "incognito window chrome enter",
          "incognito window chrome launch",
          "incognito window chrome open",
          "incognito window chrome start",
          "incognito window enter",
          "incognito window enter browser",
          "incognito window enter chrome",
          "incognito window enter google chrome",
          "incognito window google chrome enter",
          "incognito window google chrome launch",
          "incognito window google chrome open",
          "incognito window google chrome start",
          "incognito window launch",
          "incognito window launch browser",
          "incognito window launch chrome",
          "incognito window launch google chrome",
          "incognito window open",
          "incognito window open browser",
          "incognito window open chrome",
          "incognito window open google chrome",
          "incognito window start",
          "incognito window start browser",
          "incognito window start chrome",
          "incognito window start google chrome",
          "launch browser incognito",
          "launch browser incognito mode",
          "launch browser incognito tab",
          "launch browser incognito window",
          "launch browser private mode",
          "launch browser private tab",
          "launch browser private window",
          "launch chrome incognito",
          "launch chrome incognito mode",
          "launch chrome incognito tab",
          "launch chrome incognito window",
          "launch chrome private mode",
          "launch chrome private tab",
          "launch chrome private window",
          "launch google chrome incognito",
          "launch google chrome incognito mode",
          "launch google chrome incognito tab",
          "launch google chrome incognito window",
          "launch google chrome private mode",
          "launch google chrome private tab",
          "launch google chrome private window",
          "launch incognito",
          "launch incognito browser",
          "launch incognito chrome",
          "launch incognito google chrome",
          "launch incognito mode",
          "launch incognito mode browser",
          "launch incognito mode chrome",
          "launch incognito mode google chrome",
          "launch incognito tab",
          "launch incognito tab browser",
          "launch incognito tab chrome",
          "launch incognito tab google chrome",
          "launch incognito window",
          "launch incognito window browser",
          "launch incognito window chrome",
          "launch incognito window google chrome",
          "launch private mode",
          "launch private mode browser",
          "launch private mode chrome",
          "launch private mode google chrome",
          "launch private tab",
          "launch private tab browser",
          "launch private tab chrome",
          "launch private tab google chrome",
          "launch private window",
          "launch private window browser",
          "launch private window chrome",
          "launch private window google chrome",
          "open browser incognito",
          "open browser incognito mode",
          "open browser incognito tab",
          "open browser incognito window",
          "open browser private mode",
          "open browser private tab",
          "open browser private window",
          "open chrome incognito",
          "open chrome incognito mode",
          "open chrome incognito tab",
          "open chrome incognito window",
          "open chrome private mode",
          "open chrome private tab",
          "open chrome private window",
          "open google chrome incognito",
          "open google chrome incognito mode",
          "open google chrome incognito tab",
          "open google chrome incognito window",
          "open google chrome private mode",
          "open google chrome private tab",
          "open google chrome private window",
          "open incognito",
          "open incognito browser",
          "open incognito chrome",
          "open incognito google chrome",
          "open incognito mode",
          "open incognito mode browser",
          "open incognito mode chrome",
          "open incognito mode google chrome",
          "open incognito tab",
          "open incognito tab browser",
          "open incognito tab chrome",
          "open incognito tab google chrome",
          "open incognito window",
          "open incognito window browser",
          "open incognito window chrome",
          "open incognito window google chrome",
          "open private mode",
          "open private mode browser",
          "open private mode chrome",
          "open private mode google chrome",
          "open private tab",
          "open private tab browser",
          "open private tab chrome",
          "open private tab google chrome",
          "open private window",
          "open private window browser",
          "open private window chrome",
          "open private window google chrome",
          "private mode browser enter",
          "private mode browser launch",
          "private mode browser open",
          "private mode browser start",
          "private mode chrome enter",
          "private mode chrome launch",
          "private mode chrome open",
          "private mode chrome start",
          "private mode enter",
          "private mode enter browser",
          "private mode enter chrome",
          "private mode enter google chrome",
          "private mode google chrome enter",
          "private mode google chrome launch",
          "private mode google chrome open",
          "private mode google chrome start",
          "private mode launch",
          "private mode launch browser",
          "private mode launch chrome",
          "private mode launch google chrome",
          "private mode open",
          "private mode open browser",
          "private mode open chrome",
          "private mode open google chrome",
          "private mode start",
          "private mode start browser",
          "private mode start chrome",
          "private mode start google chrome",
          "private tab browser enter",
          "private tab browser launch",
          "private tab browser open",
          "private tab browser start",
          "private tab chrome enter",
          "private tab chrome launch",
          "private tab chrome open",
          "private tab chrome start",
          "private tab enter",
          "private tab enter browser",
          "private tab enter chrome",
          "private tab enter google chrome",
          "private tab google chrome enter",
          "private tab google chrome launch",
          "private tab google chrome open",
          "private tab google chrome start",
          "private tab launch",
          "private tab launch browser",
          "private tab launch chrome",
          "private tab launch google chrome",
          "private tab open",
          "private tab open browser",
          "private tab open chrome",
          "private tab open google chrome",
          "private tab start",
          "private tab start browser",
          "private tab start chrome",
          "private tab start google chrome",
          "private window browser enter",
          "private window browser launch",
          "private window browser open",
          "private window browser start",
          "private window chrome enter",
          "private window chrome launch",
          "private window chrome open",
          "private window chrome start",
          "private window enter",
          "private window enter browser",
          "private window enter chrome",
          "private window enter google chrome",
          "private window google chrome enter",
          "private window google chrome launch",
          "private window google chrome open",
          "private window google chrome start",
          "private window launch",
          "private window launch browser",
          "private window launch chrome",
          "private window launch google chrome",
          "private window open",
          "private window open browser",
          "private window open chrome",
          "private window open google chrome",
          "private window start",
          "private window start browser",
          "private window start chrome",
          "private window start google chrome",
          "start browser incognito",
          "start browser incognito mode",
          "start browser incognito tab",
          "start browser incognito window",
          "start browser private mode",
          "start browser private tab",
          "start browser private window",
          "start chrome incognito",
          "start chrome incognito mode",
          "start chrome incognito tab",
          "start chrome incognito window",
          "start chrome private mode",
          "start chrome private tab",
          "start chrome private window",
          "start google chrome incognito",
          "start google chrome incognito mode",
          "start google chrome incognito tab",
          "start google chrome incognito window",
          "start google chrome private mode",
          "start google chrome private tab",
          "start google chrome private window",
          "start incognito",
          "start incognito browser",
          "start incognito chrome",
          "start incognito google chrome",
          "start incognito mode",
          "start incognito mode browser",
          "start incognito mode chrome",
          "start incognito mode google chrome",
          "start incognito tab",
          "start incognito tab browser",
          "start incognito tab chrome",
          "start incognito tab google chrome",
          "start incognito window",
          "start incognito window browser",
          "start incognito window chrome",
          "start incognito window google chrome",
          "start private mode",
          "start private mode browser",
          "start private mode chrome",
          "start private mode google chrome",
          "start private tab",
          "start private tab browser",
          "start private tab chrome",
          "start private tab google chrome",
          "start private window",
          "start private window browser",
          "start private window chrome",
          "start private window google chrome",
      },

      // Translate
      {
          "browser change language page",
          "browser change language this",
          "browser change language this page",
          "browser page change language",
          "browser page translate",
          "browser this change language",
          "browser this page change language",
          "browser this page translate",
          "browser this translate",
          "browser translate page",
          "browser translate this",
          "browser translate this page",
          "change language browser page",
          "change language browser this",
          "change language browser this page",
          "change language chrome page",
          "change language chrome this",
          "change language chrome this page",
          "change language google chrome page",
          "change language google chrome this",
          "change language google chrome this page",
          "change language page",
          "change language page browser",
          "change language page chrome",
          "change language page google chrome",
          "change language this",
          "change language this browser",
          "change language this chrome",
          "change language this google chrome",
          "change language this page",
          "change language this page browser",
          "change language this page chrome",
          "change language this page google chrome",
          "chrome change language page",
          "chrome change language this",
          "chrome change language this page",
          "chrome page change language",
          "chrome page translate",
          "chrome this change language",
          "chrome this page change language",
          "chrome this page translate",
          "chrome this translate",
          "chrome translate page",
          "chrome translate this",
          "chrome translate this page",
          "google chrome change language page",
          "google chrome change language this",
          "google chrome change language this page",
          "google chrome page change language",
          "google chrome page translate",
          "google chrome this change language",
          "google chrome this page change language",
          "google chrome this page translate",
          "google chrome this translate",
          "google chrome translate page",
          "google chrome translate this",
          "google chrome translate this page",
          "page browser change language",
          "page browser translate",
          "page change language",
          "page change language browser",
          "page change language chrome",
          "page change language google chrome",
          "page chrome change language",
          "page chrome translate",
          "page google chrome change language",
          "page google chrome translate",
          "page translate",
          "page translate browser",
          "page translate chrome",
          "page translate google chrome",
          "this browser change language",
          "this browser translate",
          "this change language",
          "this change language browser",
          "this change language chrome",
          "this change language google chrome",
          "this chrome change language",
          "this chrome translate",
          "this google chrome change language",
          "this google chrome translate",
          "this page browser change language",
          "this page browser translate",
          "this page change language",
          "this page change language browser",
          "this page change language chrome",
          "this page change language google chrome",
          "this page chrome change language",
          "this page chrome translate",
          "this page google chrome change language",
          "this page google chrome translate",
          "this page translate",
          "this page translate browser",
          "this page translate chrome",
          "this page translate google chrome",
          "this translate",
          "this translate browser",
          "this translate chrome",
          "this translate google chrome",
          "translate browser page",
          "translate browser this",
          "translate browser this page",
          "translate chrome page",
          "translate chrome this",
          "translate chrome this page",
          "translate google chrome page",
          "translate google chrome this",
          "translate google chrome this page",
          "translate page",
          "translate page browser",
          "translate page chrome",
          "translate page google chrome",
          "translate this",
          "translate this browser",
          "translate this chrome",
          "translate this google chrome",
          "translate this page",
          "translate this page browser",
          "translate this page chrome",
          "translate this page google chrome",
      },

      // Update Chrome
      {
          "browser install",
          "browser update",
          "browser upgrade",
          "chrome install",
          "chrome update",
          "chrome upgrade",
          "google chrome install",
          "google chrome update",
          "google chrome upgrade",
          "install browser",
          "install chrome",
          "install google chrome",
          "update browser",
          "update chrome",
          "update google chrome",
          "upgrade browser",
          "upgrade chrome",
          "upgrade google chrome",
      },

      // End of generated test code
  };

  // The test code below ensures that each element of the outer vector above,
  // |literal_concept_expressions|, fully corresponds to exactly one Pedal
  // implementation.  For each one, the full list of literal expressions are
  // confirmed as concept matches for the Pedal.  Finally, we verify that every
  // implemented Pedal has been tested using set logic.
  MockAutocompleteProviderClient client;
  OmniboxPedalProvider provider(client);
  const auto& pedals = provider.pedals_;
  std::unordered_set<const OmniboxPedal*> found_pedals(pedals.size());
  for (const auto& pedal_concept : literal_concept_expressions) {
    const base::string16 first_trigger = base::ASCIIToUTF16(pedal_concept[0]);
    auto iter =
        std::find_if(pedals.begin(), pedals.end(), [&](const auto& pedal) {
          const auto sequence = provider.Tokenize(first_trigger);
          return pedal.second->IsConceptMatch(sequence);
        });
    EXPECT_NE(iter, pedals.end())
        << "Canonical pedal not found for: " << first_trigger;
    const OmniboxPedal* canonical_pedal = iter->second.get();
    const bool is_newly_found = found_pedals.insert(canonical_pedal).second;
    EXPECT_TRUE(is_newly_found)
        << "Found the same Pedal more than once with: " << first_trigger;
    for (const char* literal : pedal_concept) {
      const base::string16 expression = base::ASCIIToUTF16(literal);
      const auto is_match = [&](const auto& pedal) {
        const auto sequence = provider.Tokenize(expression);
        return pedal.second->IsConceptMatch(sequence);
      };
      iter = std::find_if(pedals.begin(), pedals.end(), is_match);
      EXPECT_NE(iter, pedals.end()) << "Pedal not found for: " << expression;
      EXPECT_EQ(iter->second.get(), canonical_pedal)
          << "Found wrong Pedal for: " << expression;
      std::advance(iter, 1);
      iter = std::find_if(iter, pedals.end(), is_match);
      EXPECT_EQ(iter, pedals.end())
          << "Found more than one Pedal match for: " << expression;
    }
  }
  EXPECT_EQ(pedals.size(), found_pedals.size())
      << "Not all implemented Pedals were represented in test cases.";
}
